// ------------------------------------------------------------------------
// interface.cpp: Main user interface widget for qtecasound
// Copyright (C) 1999-2001 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <fstream>
#include <vector>
#include <unistd.h>
#include <pthread.h>

#include <qapplication.h>
#include <qpushbutton.h>
#include <qgroupbox.h>
#include <qfont.h>
#include <qaccel.h>
#include <qlayout.h>
#include <qlineedit.h>
#include <qlabel.h>
#include <qmessagebox.h>
#include <qstatusbar.h>
#include <qtabwidget.h>
#include <qtimer.h>
#include <qaccel.h>

#include <kvutils/kvutils.h>
#include "qebuttonrow.h"

#include <ecasound/eca-version.h>
#include <ecasound/eca-control.h>

#include "chainsetup.h"
#include "qtecasound.h"
#include "rtposition.h"
#include "debug.h"
#include "debugtab.h"
#include "interface.h"

QEInterface::QEInterface(ECA_CONTROL* control, QWidget *parent, const char *name)
  : QWidget(parent, name), ctrl_repp(control)
{
  setCaption("qtecasound - control panel");
  qApp->setMainWidget(this);
  init_layout();
}

QEInterface::~QEInterface(void) {
  if (ctrl_repp != 0) {
    if (ctrl_repp->is_running()) ctrl_repp->stop();
    ctrl_repp->quit();
  }
}

void QEInterface::get_focus(void) {
  raise();
  setActiveWindow();
  setFocus();
}

void QEInterface::init_layout(void) {
  QBoxLayout* topLayout = new QVBoxLayout( this );
  QBoxLayout* qtbuttons = new QHBoxLayout();
  QBoxLayout* textinput = new QHBoxLayout();
  QBoxLayout* tabwidget = new QHBoxLayout();
  QBoxLayout* bottomrow = new QHBoxLayout();

  init_buttons();
  topLayout->addWidget(buttonrow_repp);
  topLayout->addLayout(qtbuttons,0);
  topLayout->addLayout(textinput,0);
  topLayout->addLayout(tabwidget,1);
  topLayout->addLayout(bottomrow,0);

  init_tabwidget(tabwidget);
  init_runtimebar(qtbuttons);
  init_textinput(textinput);
  init_bottomrow(bottomrow);
}

void QEInterface::init_bottomrow(QBoxLayout* bottomrow) {
  init_statusbar();
  bottomrow->addWidget(statusbar_repp, 2, 0);
}

void QEInterface::init_statusbar(void) {
  statusbar_repp = new QStatusBar(this, "qsbar");
  statusbar_repp->message(QString("qtecasound v") + QString(ecasound_library_version) + QString(" ready.")); 

  QTimer *timer = new QTimer( this );
  connect( timer, SIGNAL(timeout()), this, SLOT(update_statusbar()));
  timer->start( 500, false);
}

void QEInterface::update_statusbar(void) {
  static string last_epstatus;
  string new_epstatus = ctrl_repp->engine_status();

  if (new_epstatus == last_epstatus) return;
  else last_epstatus = new_epstatus;

  statusbar_repp->message(QString("qtecasound v") + ecasound_library_version
			    + " . (C) 1997-2001 Kai Vehmanen . engine status: [" +
			    last_epstatus.c_str() + "]"); 
}

void QEInterface::init_tabwidget(QBoxLayout* layout) {
  qtab_repp = new QTabWidget(this, "qtab");

  debugtab_repp = new QEDebugTab(this, qtab_repp, "debugtab");
  qtab_repp->addTab(debugtab_repp, "(E)ngine output");
  QAccel* accel = new QAccel(this);
  accel->connectItem(accel->insertItem(CTRL+Key_E), this, SLOT(popup_debugtab()));

  session_repp = new QEChainsetup(ctrl_repp, qtab_repp, "qechainsetup");
  qtab_repp->addTab(session_repp, "Session setu(p)");
  accel = new QAccel(this);
  accel->connectItem(accel->insertItem(CTRL+Key_P), this, SLOT(popup_chainsetup()));

  layout->addWidget(qtab_repp, 1, 0);
}

void QEInterface::popup_debugtab(void) { qtab_repp->showPage(debugtab_repp); }
void QEInterface::popup_chainsetup(void) { qtab_repp->showPage(session_repp); }

void QEInterface::init_runtimebar(QBoxLayout* buttons) {
  rpos_repp = new QERuntimePosition(ctrl_repp->length_in_seconds_exact(),
			       this, "rpos");
  buttons->addWidget( rpos_repp, 10, 0 );
  connect( rpos_repp, SIGNAL(position_changed_from_widget(double)), this, SLOT(emsg_setpos(double)));

  QTimer *timer = new QTimer( this );
  connect( timer, SIGNAL(timeout()), this, SLOT(update_runtimebar()));
  timer->start( 1000, false);
}

void QEInterface::update_runtimebar(void) {
  if (rpos_repp->does_widget_have_control() == false) {
    rpos_repp->position_in_seconds(ctrl_repp->position_in_seconds_exact());
    rpos_repp->length_in_seconds(ctrl_repp->length_in_seconds_exact());
  }
}

void QEInterface::init_buttons(void) {
  buttonrow_repp = new QEButtonRow(this, "buttonrow");
  buttonrow_repp->add_button(new QPushButton("(B)egin",buttonrow_repp), 
			CTRL+Key_B,
			this, SLOT(emsg_rw_begin()));

  buttonrow_repp->add_button(new QPushButton("Rew (<)",buttonrow_repp), 
			Key_Less,
			this, SLOT(emsg_rewind()));

  buttonrow_repp->add_button(new QPushButton("S(t)art",buttonrow_repp), 
			CTRL+Key_T,
			this, SLOT(emsg_start()));

  buttonrow_repp->add_button(new QPushButton("(S)top",buttonrow_repp), 
			CTRL+Key_S,
			this, SLOT(emsg_stop()));

  buttonrow_repp->add_button(new QPushButton("Fw (>)",buttonrow_repp), 
			SHIFT+Key_Greater,
			this, SLOT(emsg_forward()));

  buttonrow_repp->add_button(new QPushButton("(Q)uit",buttonrow_repp), 
			CTRL+Key_Q,
			qApp, SLOT(closeAllWindows()));


}

void QEInterface::init_textinput(QBoxLayout* textinput) {
  QLabel* tekstiinfo = new QLabel("qtecasound CL(I) ('h' for help): ", this, "tekstiinfo");
  textinput->addWidget( tekstiinfo, 5);
  
  tekstirivi_repp = new QLineEdit(this, "tekstirivi");
  textinput->addWidget( tekstirivi_repp, 10);

  QAccel* accel = new QAccel(this);
  accel->connectItem(accel->insertItem(CTRL+Key_I), tekstirivi_repp, SLOT(setFocus()));
  
  connect(tekstirivi_repp, SIGNAL(returnPressed ()), this, SLOT(emsg_general()) );
  connect(this, SIGNAL(clear_textinput()), tekstirivi_repp, SLOT(clear()) );
}

void QEInterface::emsg_general(void) {
  string s (tekstirivi_repp->text());
  if (s == "q" || s == "quit") {
    qApp->closeAllWindows();
  }
  else {
    try {
      eiam_command_wrapper(s);
    }
    catch(ECA_ERROR& e) {
      if (e.error_action() != ECA_ERROR::stop) {
	QMessageBox* mbox = new QMessageBox(this, "mbox");
	QString errormsg = "";
	errormsg += "Ecasound engine error: \"";
	errormsg += e.error_section().c_str();
	errormsg += "\"; ";
	errormsg += e.error_message().c_str();
	mbox->information(this, "qtecasound", errormsg,0);
      }
      else throw;
    }
  }
  
  emit clear_textinput();
}

void QEInterface::eiam_command_wrapper(const string& cmd) {  
  ctrl_repp->command(cmd);
  ecadebug->msg(ctrl_repp->last_error());
  ecadebug->msg(ctrl_repp->last_string());
}

void QEInterface::emsg_quit(void) {  }
void QEInterface::emsg_stop(void) { eiam_command_wrapper("s"); }
void QEInterface::emsg_start(void) { eiam_command_wrapper("t"); }
void QEInterface::emsg_rw_begin(void) { eiam_command_wrapper("setpos 0"); }
void QEInterface::emsg_forward(void) { eiam_command_wrapper("fw 5"); }
void QEInterface::emsg_rewind(void) { eiam_command_wrapper("rw 5"); }
void QEInterface::emsg_status(void) { eiam_command_wrapper("st"); }
void QEInterface::emsg_csstatus(void) { eiam_command_wrapper("cs-status"); }
void QEInterface::emsg_ctrlstatus(void) { eiam_command_wrapper("ctrl-status"); }
void QEInterface::emsg_estatus(void) { eiam_command_wrapper("es"); }
void QEInterface::emsg_fstatus(void) { eiam_command_wrapper("fs"); }
void QEInterface::emsg_cstatus(void) { eiam_command_wrapper("cs"); }
void QEInterface::emsg_exec(void) { ctrl_repp->start(); }
void QEInterface::emsg_setpos(double pos_seconds) {
  ctrl_repp->command("setpos " + kvu_numtostr(pos_seconds));
  rpos_repp->control_back_to_parent();
}

void QEInterface::not_implemented(void) {
  QMessageBox* mbox = new QMessageBox(this, "mbox");
  mbox->information(this, "qtecasound", "This feature is not implemented...",0);
}
