#ifndef __qm_stringptr_h__
#define __qm_stringptr_h__

/* This trickery is designed to solve a basic problem with Csound's
   otherwise very efficient code model. The values in the arguments
   to an opcode are all declared pointers-to-Number, and in the vast
   majority of cases, thats just what they are supposed to be.

   But sometimes, they need to actually be pointers-to-string. Csound
   worked around this by using a special value for the pointer (SSTRCOD),
   and typically had a single "strarg" holder in the opcode text.

   This limits opcodes to a single string argument, which is a silly
   limitation.

   This new trick relies on the format of IEEE 32 bit floats. All
   strings in Quasimodo that can be passed to an opcode are first
   assigned an index in a global set of strings. Then we take this
   index, and munge it a little so that it looks like an IEEE NaN
   representation. This then means that we can use the isnan()
   macro from math.h to decide if a pointer-to-Number is really
   a pointer-to-string. If it looks like NaN, then we grab the
   index to the string from the fractional part of the IEEE representation.

   There are some limitations, of course.

   1) This can only represent up to 2^24 different strings. I believe
      this to be adequate :)

   2) If a numeric value really has come out to be NaN, then clearly
      this approach could cause a problem. The assumption here is
      the trick is only used in cases where its known that the opcode
      expects *either* a Number or a string, and if its a Number, it
      will definitely *not* be NaN.

   3) zero is not a valid index into the string array, because it
      makes NaN look like Infinity.

   But other than this, I hope you'll agree that this is a pretty
   neat trick.
*/

#include <math.h>

extern const unsigned int nan_string_mask;
extern const char **qm_strings;

extern const Number *string_ptr (const char *);

inline bool
is_string_ptr (const Number * const Np) 
{
	return isnan (*Np);
}

inline const char *
get_string (const Number * const nptr) {
	/* Returns the string referenced by the NaN float
	   pointed to by nptr.
	*/

	return qm_strings[*((unsigned int *) nptr) & nan_string_mask];
}

inline Number
string_ptr_val (const char *str) 

{
	/* returns the NaN float that contains
	   an index into `Strings' where the
	   string can be found. Use for
	   setting data locations with a 
	   value that actually references
	   a string.
	*/

	const Number *nptr = string_ptr (str);
	return *nptr;
}

#endif // __qm_stringptr_h__
