/*
    Copyright (C) 1998-99 Paul Barton-Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: function_tables.h,v 1.4 1999/12/15 19:14:23 pbd Exp $
*/

#ifndef __function_tables_h__
#define __function_tables_h__

#include <pbd/rcpointer.h>
#include <map>

class DspRequest;
class DSP;
class Generator;
struct FTGEN;

class FunctionTableRequest {
 public:
	size_t argc;
	const char *filename;
	Number *args;
	bool rescale;
	int32 nelements;

	FunctionTableRequest () { memset (this, 0, sizeof (*this)); }
	FunctionTableRequest (DspRequest *, bool force_reload = false);
	~FunctionTableRequest ();

	Generator *generator () { return gen; }

	bool ok() { return _ok; }

  private:
	bool _ok;
	const char *function_name;
	int32 start_time;
	size_t element_size;
	Generator *gen;
	bool defer_alloc;
	bool ignore_duplicates;
	
	friend class FunctionTable;
	friend void ftgen (FTGEN *);
};

class FunctionTable;
typedef std::map<const char *, RCPointer<FunctionTable> > WaveTables;

class FunctionTable {
  public:
	enum GuardType { 
		GuardWrap,
		GuardExtend
	};

	static const size_t MaximumTableSize;

	const char *name;           /* ... of this table */
	Number *ftable;             /* the table itself */
	uint32	flen;               /* length of table */
	uint32	lenmask;            /* &-ed with an index to provide wrap */
	uint32	lobits;             /* flen * (2^lobits) >= MAXLEN */
	uint32	lomask;             /* what ? */
	Number	lodiv;              /* what ? */
	Dumber  tpdlen;             /* (2 * PI) / flen */
	Number	cvtbas, cpscvt;     /* what ? */

	/* derived from soundFile, where applicable */

	int16	loopmode1;
	int16	loopmode2;
	int32	begin1, end1;	    /* frames */
	int32	begin2, end2;       /* frames */
	int32	soundend, flenfrms; /* frames */
	int32   nchnls;
	uint32  frame_rate;          

	void clear ();
	void rescale ();
	int undefer ();
	int store (const char *path);
	void set_mapped (bool yn) { mapped = yn; }
	void set_guard_type (GuardType);
	bool deferred () { return deferred_request != 0; }
	void request_length (size_t len, size_t nchannels);
	void set_table (Number *, size_t len, size_t nchannels);

	void remove_from_tables () { 
		in_tables = false;
	}

	static const char *name_as_string (Number *nameptr);

  protected:
	~FunctionTable ();
	int32 count; 
	friend class RCPointer<FunctionTable>;
	friend RCPointer<FunctionTable> 
		ftfind (const char *name, 
			bool deferred_legal, 
			bool undefer);

  protected:
	/* This and the ftfind() functions are the only ways
	   to get your hands on a FunctionTable (pointer).
	*/

	static RCPointer<FunctionTable> generate (FunctionTableRequest *);
	friend class DSP;
	friend void ftgen(FTGEN *);

  private:
	FunctionTable (FunctionTableRequest *);
	

	bool       in_tables;       /* inserted into global wavetables */
	bool       mapped;          /* data is mapped if true */	
	bool       should_defer;    /* lazy evaluation */
	GuardType  guard_type;      
	FunctionTableRequest *deferred_request;
	WaveTables::iterator wavetables_position;

	void initialize (const char *, size_t, size_t);
	int  ask_about_duplicates (FunctionTableRequest *);
	int  instantiate (FunctionTableRequest *);
	void set_guard_point ();
	void set_length ();
};

typedef FunctionTable FUNC; /* for old Csound opcodes */


RCPointer<FunctionTable> ftfind (const char *name, 
				 bool deferred_legal = false, 
				 bool undefer = false);

RCPointer<FunctionTable> ftfind (Number *argp, 
				 bool deferred_legal = false, 
				 bool undefer = false);

#define  PFRAC(x)   ((Number)((x) & ftp->lomask) * ftp->lodiv)


#endif // __function_tables_h__
