/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: vdelay.cc,v 1.1 1999/11/01 04:29:07 pbd Exp $
*/


/*      vdelay, multitap, reverb2 coded by Paris Smaragdis 1994 */
/*      Berklee College of Music Csound development team        */
/*      Copyright (c) May 1994.  All rights reserved            */

#include <stdlib.h>
#include <stdio.h>
#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dsp.h>

#include "vdelay.h"

#define ESR     (esr/1000.0f)

void
vdelset(VDEL * p)
{				/*  vdelay set-up   */
	uint32 n = (long) (*p->imaxd * ESR);
	Number *buf;

	if (n == 0)
		n = 1;		/* fix due to Troxler */

	if (!*p->istod) {

	        /* allocate space for delay buffer */

		if (p->aux.auxp == NULL ||
		    (n * sizeof (Number)) > p->aux.size)
			 p->aux.alloc (n * sizeof (Number));
		else {
			buf = (Number *) p->aux.auxp;	
			do {
				*buf++ = 0.0f;
			} while (--n);
		}
		p->left = 0;
	}
}

void
vdelay(VDEL * p)
{				/*      vdelay  routine */
	uint32 nn, maxd, indx;
	Number *out = p->sr;	/* assign object data to local variables  
				 */
	Number *in = p->ain;
	Number *del = p->adel;
	Number *buf = (Number *) p->aux.auxp;

	maxd = (unsigned long) (*p->imaxd * ESR);
	if (maxd == 0)
		maxd = 1;	/* Degenerate case */
	nn = ksmps;
	indx = p->left;

	if (p->nth_arg_is_audio (1)) {
		do {
			Number fv1, fv2;
			int32 v1, v2;

			buf[indx] = *in++;
			fv1 = indx - (*del++) * ESR;	/* Make sure Inside 
							   the buffer      */
			while (fv1 > maxd)
				fv1 -= maxd;
			while (fv1 < 0)
				fv1 += maxd;

			if (fv1 < maxd - 1)	/* Find next sample for
						   interpolation      */
				fv2 = fv1 + 1;
			else
				fv2 = 0.0f;

			v1 = (long) fv1;
			v2 = (long) fv2;
			*out++ = buf[v1] + (fv1 - (long) fv1) * (buf[v2] -
								 buf[v1]);

			if (++indx == maxd)
				indx = 0;	/* Advance current pointer */

		} while (--nn);
	} else {		/* and, if delay is k-rate */
		do {
			Number fv1, fv2;
			int32 v1, v2;

			buf[indx] = *in++;
			fv1 = indx - *del * ESR;	/* Make sure inside the 
							   buffer      */
			while (fv1 > maxd)
				fv1 -= maxd;
			while (fv1 < 0)
				fv1 += maxd;

			if (fv1 < maxd - 1)	/* Find next sample for
						   interpolation      */
				fv2 = fv1 + 1;
			else
				fv2 = 0.0f;

			v1 = (long) fv1;
			v2 = (long) fv2;
			*out++ = buf[v1] + (fv1 - (long) fv1) * (buf[v2] -
								 buf[v1]);

			if (++indx == maxd)
				indx = 0;	/*      Advance current
						   pointer */

		} while (--nn);
	}
	p->left = indx;		/*      and keep track of where you are
				 */
}

void
multitap_set(MDEL * p)
{
	size_t n;
	Number *buf, max = 0.0f;

	if (p->INOCOUNT / 2 == (Number) p->INOCOUNT / 2.)
		fatal << "Wrong input count in multitap\n" << endmsg;


	for (n = 0; n < p->INOCOUNT - 1; n += 2) {
		if (max < *p->ndel[n])
			max = *p->ndel[n];
	}

	n = (long) (esr * max * sizeof (Number));

	if (p->aux.auxp == NULL ||	/* allocate space for delay buffer */
	    n > p->aux.size) {
		p->aux.alloc (n);
	} else {
		buf = (Number *) p->aux.auxp;	/* make sure buffer is
						   empty       */
		n = p->max;
		do {
			*buf++ = 0.0f;
		} while (--n);
	}

	p->left = 0;
	p->max = (long) (esr * max);
}

void
multitap_play(MDEL * p)
{				/* assign object data to local variables   */
	size_t n, nn = ksmps, indx = p->left, delay;
	Number *out = p->sr, *in = p->ain;
	Number *buf = (Number *) p->aux.auxp;
	Number max = (Number) p->max;

	do {
		buf[indx] = *in++;	/*      Write input     */
		*out = 0.0f;	/*      Clear output buffer     */

		if (++indx == max)
			indx = 0;	/*      Advance input pointer   */
		for (n = 0; n < p->INOCOUNT - 1; n += 2) {
			delay = indx - (long) (esr * *p->ndel[n]);
			if (delay < 0)
				delay += (long) max;
			*out += buf[delay] * *p->ndel[n + 1];	/*     
								   Write output    */
		}
		out++;		/*      Advance output pointer  */
	} while (--nn);
	p->left = indx;
}

#ifdef OLD_CODE
#define LOG001  (-6.9078)	/* log(.001) */

void
reverb2_set(STVB * p)
{				/* 6-comb/lowpass, 5-allpass reverberator */
	int32 i, n;
	Number *temp;

	if (*p->hdif > 1.0 || *p->hdif < 0)
		fatal << "High frequency diffusion not in (0, 1)\n" << endmsg;


	if (*p->istor == 0.0 || p->temp.auxp == NULL) {
		p->temp.alloc (ksmps * sizeof (Number));
		temp = (Number *) p->temp.auxp;
		for (n = 0; n < ksmps; n++)
			*temp++ = 0.0;
		for (i = 0; i < Combs; i++) {
			p->c_time[i] = gc_time[i];
			p->c_gain[i] = exp((double) (LOG001 * (p->c_time[i] /
							       esr) /
					       (gc_gain[i] * *p->time)));
			p->g[i] = *p->hdif;
			p->c_gain[i] = p->c_gain[i] * (1 - p->g[i]);
			p->z[i] = 0.0;

			p->caux[i].alloc ((size_t) 
					  (p->c_time[i] * sizeof (Number)));
			p->cbuf_cur[i] = (Number *) p->caux[i].auxp;
			for (n = 0; n < p->c_time[i]; n++)
				*(p->cbuf_cur[i] + n) = 0.0;
		}

		for (i = 0; i < Alpas; i++) {
			p->a_time[i] = ga_time[i];
			p->a_gain[i] = exp((double) (LOG001 * (p->c_time[i] /
							       esr) /
					       (ga_gain[i] * *p->time)));
			p->aaux[i].alloc ((size_t)
					  p->a_time[i] * sizeof (Number));
			p->abuf_cur[i] = (Number *) p->aaux[i].auxp;
		}
	}
	p->prev_time = *p->time;
	p->prev_hdif = *p->hdif;
}

void
reverb2_play(STVB * p)
{
	int32 i, n = ksmps;
	Number *in, *out = p->out, *buf, *end;
	Number gain, z;

	do
		*out++ = 0.0;
	while (--n);
	if (*p->time != p->prev_time || *p->hdif != p->prev_hdif) {
		if (*p->hdif > 1.) {
			info << "Warning: High frequency diffusion>1\n" << endmsg;

			*p->hdif = 1.;
		}
		if (*p->hdif < 0) {
			info << "Warning: High frequency diffusion<0\n" << endmsg;

			*p->hdif = 0.;
		}
		if (*p->time < 0) {
			info << "Negative time?\n" << endmsg;

			*p->time = 0.0;
		}
		for (i = 0; i < Combs; i++) {
			p->c_gain[i] = exp((double) (LOG001 * (p->c_time[i] /
							       esr) /
					       (gc_gain[i] * *p->time)));
			p->g[i] = *p->hdif;
			p->c_gain[i] = p->c_gain[i] * (1 - p->g[i]);
			p->z[i] = 0.0;
		}

		for (i = 0; i < Alpas; i++)
			p->a_gain[i] = exp((double) (LOG001 * (p->a_time[i] /
							       esr) /
					       (ga_gain[i] * *p->time)));

		p->prev_time = *p->time;
		p->prev_hdif = *p->hdif;
	}
	for (i = 0; i < Combs; i++) {
		buf = p->cbuf_cur[i];
		end = (Number *) p->caux[i].endp;
		gain = p->c_gain[i];
		in = p->in;
		out = p->out;
		n = ksmps;
		do {
			*out++ += *buf;
			*buf += p->z[i] * p->g[i];
			p->z[i] = *buf;
			*buf *= gain;
			*buf += *in++;
			if (++buf >= end)
				buf = (Number *) p->caux[i].auxp;
		} while (--n);
		p->cbuf_cur[i] = buf;
	}

	for (i = 0; i < Alpas; i++) {
		in = (Number *) p->temp.auxp;
		out = p->out;
		n = ksmps;
		do
			*in++ = *out++;
		while (--n);
		buf = p->abuf_cur[i];
		end = (Number *) p->aaux[i].endp;
		gain = p->a_gain[i];
		in = (Number *) p->temp.auxp;
		out = p->out;
		n = ksmps;
		do {
			z = *buf;
			*buf = gain * z + *in++;
			*out++ = z - gain * *buf;
			if (++buf >= end)
				buf = (Number *) p->aaux[i].auxp;
		} while (--n);
		p->abuf_cur[i] = buf;
	}
}
#endif

/*      nreverb coded by Paris Smaragdis 1994 and Richard Karpen 1998 */

#define LOG001  (-6.9078)	/* log(.001) */

Number ngc_time[Combs] =
{1433.0f, 1601.0f, 1867.0f, 2053.0f, 2251.0f, 2399.0f};
Number ngc_gain[Combs] =
{.822f, .802f, .773f, .753f, .753f, .753f};
Number nga_time[Alpas] =
{347.0f, 113.0f, 37.0f, 59.0f, 43.0f};
Number nga_gain = 0.7f;

bool
prime(int16 val)
{
	int16 i, last;

	last = (int16) sqrt((double) val);
	for (i = 3; i <= last; i += 2) {
		if ((val % i) == 0)
			return false;
	}
	return true;
}

void
nreverb_set(NREV * p)
{				/* 6-comb/lowpass, 5-allpass reverberator */
	size_t i, n;
	Number *temp;
	Number srscale = esr / 25641.0f;		/* denominator is
						   probably CCRMA "samson box" sampling-rate! */
	int16 c_time, a_time;

	if (*p->hdif > 1.0f || *p->hdif < 0.0f)
		fatal << "High frequency diffusion not in (0, 1)\n" << endmsg;


	if (*p->istor == 0.0f || p->temp.auxp == NULL) {
		p->temp.alloc (ksmps * sizeof (Number));
		temp = (Number *) p->temp.auxp;
		for (n = 0; n < ksmps; n++)
			*temp++ = 0.0f;

		for (i = 0; i < Combs; i++) {
			/* derive new primes to make delay times work with
			   orch. sampling-rate */
			c_time = (int16) (ngc_time[i] * srscale);
			if (c_time % 2 == 0)
				c_time += 1;
			while (!prime(c_time))
				c_time += 2;
			p->c_time[i] = (Number) c_time;

			p->c_gain[i] = (Number) exp((double) (LOG001 *
						   (p->c_time[i] / esr) /
					      (ngc_gain[i] * *p->time)));
			p->g[i] = *p->hdif;
			p->c_gain[i] = p->c_gain[i] * (1 - p->g[i]);
			p->z[i] = 0.0f;

			p->caux[i].alloc ((size_t) 
					  (p->c_time[i] * sizeof (Number)));
			p->cbuf_cur[i] = (Number *) p->caux[i].auxp;
			for (n = 0; n < p->c_time[i]; n++)
				*(p->cbuf_cur[i] + n) = 0.0f;
		}
		for (i = 0; i < 5; i++) {
			a_time = (int16) (nga_time[i] * srscale);
			if (a_time % 2 == 0)
				a_time += 1;
			while (!prime(a_time))
				a_time += 2;
			p->a_time[i] = (Number) a_time;
			p->a_gain[i] = (Number) exp((double) (LOG001 *
						   (p->a_time[i] / esr) /
						 (nga_gain * *p->time)));
			p->aaux[i].alloc ((size_t) 
					  p->a_time[i] * sizeof (Number));
			p->abuf_cur[i] = (Number *) p->aaux[i].auxp;
		}

	}
	p->prev_time = *p->time;
	p->prev_hdif = *p->hdif;
}

void
nreverb(NREV * p)
{
	int32 i, n = ksmps;
	Number *in, *out = p->out, *buf, *end;
	Number gain, z;

	do
		*out++ = 0.0f;
	while (--n);
	if (*p->time != p->prev_time || *p->hdif != p->prev_hdif) {
		if (*p->hdif > 1.0f) {
			info << "Warning: High frequency diffusion>1\n" << endmsg;

			*p->hdif = 1.0f;
		}
		if (*p->hdif < 0.0f) {
			info << "Warning: High frequency diffusion<0\n" << endmsg;

			*p->hdif = 0.0f;
		}
		if (*p->time < 0.0f) {
			info << "Negative time?\n" << endmsg;

			*p->time = 0.0f;
		}
		for (i = 0; i < Combs; i++) {
			p->c_gain[i] = (Number) exp((double) (LOG001 *
						   (p->c_time[i] / esr) /
					      (ngc_gain[i] * *p->time)));
			p->g[i] = *p->hdif;
			p->c_gain[i] = p->c_gain[i] * (1 - p->g[i]);
			p->z[i] = 0.0f;
		}

		for (i = 0; i < Alpas; i++)
			p->a_gain[i] = (Number) exp((double) (LOG001 *
						   (p->a_time[i] / esr) /
						 (nga_gain * *p->time)));

		p->prev_time = *p->time;
		p->prev_hdif = *p->hdif;
	}
	for (i = 0; i < Combs; i++) {
		buf = p->cbuf_cur[i];
		end = (Number *) p->caux[i].endp;
		gain = p->c_gain[i];
		in = p->in;
		out = p->out;
		n = ksmps;
		do {
			*out++ += *buf;
			*buf += p->z[i] * p->g[i];
			p->z[i] = *buf;
			*buf *= gain;
			*buf += *in++;
			if (++buf >= end)
				buf = (Number *) p->caux[i].auxp;
		} while (--n);
		p->cbuf_cur[i] = buf;
	}

	for (i = 0; i < 5; i++) {
		in = (Number *) p->temp.auxp;
		out = p->out;
		n = ksmps;
		do
			*in++ = *out++;
		while (--n);
		buf = p->abuf_cur[i];
		end = (Number *) p->aaux[i].endp;
		gain = p->a_gain[i];
		in = (Number *) p->temp.auxp;
		out = p->out;
		n = ksmps;
		do {
			z = *buf;
			*buf = gain * z + *in++;
			*out++ = z - gain * *buf;
			if (++buf >= end)
				buf = (Number *) p->aaux[i].auxp;
		} while (--n);
		p->abuf_cur[i] = buf;
	}
}

Opcode opcodes[] =
{
	VDELAY_OPCODE_LIST,
	{NULL}
};
