#ifndef __wavegde_h__
#define __wavegde_h__



/* waveguide.h -- primitive data types and declarations for waveguides */

/*
 * Code conversion from C++ to C (October 1994)
 * Author: Michael A. Casey MIT Media Labs
 * Language: C
 * Copyright (c) 1994 MIT Media Lab, All Rights Reserved
 */

#ifndef _waveguide_h
#define _waveguide_h

#include <math.h>
#ifndef sinf
#define sinf(a) (float)sin((double)(a))
#define cosf(a) (float)cos((double)(a))
#define sqrtf(a) (float)sqrt((double)(a))
#define atan2f(a,b) (float)atan2((double)(a),(double)(b))
#define powf(a,b) (float)pow((double)(a),(double)(b))
#endif
#ifndef PI
#ifdef M_PI
#define PI M_PI
#else
#define PI	3.14159265358979323846
#endif
#endif

/* TYPEDEFS */
typedef float   sampT;    /* sample type */
typedef float*  sampPtr;
typedef float   paramT;   /* paramater type */
typedef float*  paramPtr;
typedef long    len_t;    /* length type */



/* CLASS DEFINITIONS */

/* circularBuffer -- circular buffer class */
/* serves as base class for waveguides and lattice filters */
typedef struct {
  int _inited;              /* Data initialization flag */
  len_t size;               /* Size of the digital filter lattice */
  sampPtr insertionPoint;   /* Position in queue to place new data */
  sampPtr extractionPoint;  /* Position to read data from */
  sampPtr data;             /* The lattice data */
  sampPtr endPoint;         /* The end of the data */
  sampPtr pointer;          /* pointer to current position in data */
}circularBuffer;


/* circular buffer member functions */
void circularBufferCircularBuffer(circularBuffer*,len_t);/* constructor */  
void circularBufferWrite(circularBuffer*, sampT);  /* write a sample */
sampT circularBufferRead(circularBuffer*);         /* read next sample */
void circularBufferResize(circularBuffer*,len_t);  /* resize the buffer */


/* class filter -- recursive filter implementation class */
typedef struct{
  circularBuffer buffer; /* The filter's delay line */
  paramPtr coeffs;        /* The filter's coefficients */
} filter;

/* filter member functions */
void filterFilter(filter*,len_t); /* constructor */
void filterSet(filter*,paramPtr); /* set the coefficients */
void filterReset(filter*);        /* clears the delay lines */
paramT filterGet(filter*,len_t);  /* check index range, return coefficient */
sampT filterIIR(filter*,sampT);   /* recursive filter routine */
sampT filterFIR(filter*,sampT);   /* convolution filter routine */


/* waveguide rail implementation class */
typedef circularBuffer guideRail; /* It's just a circular buffer really */

/* guideRail member functions */
void guideRailGuideRail(guideRail*,len_t);/* constructor */
sampT guideRailAccess(guideRail*,len_t);  /* delay line access routine */
void guideRailUpdate(guideRail*,sampT);   /* delay line update routine */


/* waveguide -- abstract base class definition for waveguide classes */
typedef struct{
  int excited;         /* excitation flag */
  guideRail upperRail; /* the right-going wave */
  guideRail lowerRail; /* the left-going wave */
  paramT c;            /* The tuning filter coefficient */
  paramT p;            /* The tuning fitler state */
  paramT w0;           /* The fundamental frequency (PI normalized) */
  paramT f0;           /* The fundamental frequency (Hertz) */
}waveguide;

sampT filterAllpass(waveguide*,sampT);/* 1st-order allpass filtering*/

/* waveguide member functions */
void waveguideWaveguide(waveguide*,
			paramT,
			sampT*,
			sampT*);
paramT waveguideGetFreq(waveguide*);   /* return f0 frequency */
void waveguideSetTuning(waveguide*, paramT); /* Set tuning filters */
#endif


extern void circularBufferCircularBuffer (circularBuffer *cb, len_t N);
extern void circularBufferWrite (circularBuffer *cb, sampT val);
extern sampT circularBufferRead (circularBuffer *cb);
extern void guideRailGuideRail (guideRail *gr, len_t d);
extern void guideRailUpdate (guideRail *gr, sampT samp);
extern sampT guideRailAccess (guideRail *gr, len_t pos);
extern void filterFilter (filter *filt, len_t n);
extern void filterSet (filter *filt, paramPtr c);
extern void filterReset (filter *filt);
extern paramT filterGet (filter *filt, len_t i);
extern sampT filterIIR (filter *filt, sampT s);
extern sampT filterFIR (filter *filt, sampT s);
extern sampT filterAllpass (waveguide *wg, sampT s);
extern void waveguideWaveguide (waveguide *wg, paramT freq, sampT *upperData, sampT *lowerData);
extern void waveguideSetTuning (waveguide *wg, paramT df);

#endif __wavegde_h__

