/***************************************************************\
*   repluck.c                                                  *
*   Various waveguide instruments                              *
*   3 March 1996 John ffitch                                   *
\***************************************************************/

#include "cs.h"
#include "entry.h"
#include "repluck.h"
#include "auxfd.h"

void wgpsetin(WGPLUCK2 *);

void
wgpset(WGPLUCK2 * p)
{
	p->ain = NULL;
	wgpsetin(p);
}

void
wgpsetin(WGPLUCK2 * p)
{
	int npts;
	int pickpt;
	int rail_len;
	float upslope;
	float downslope;
	float *initial_shape;
	int i;
	int scale = 1;
	DelayLine *upper_rail;
	DelayLine *lower_rail;
	/* Initialize variables.... */
	npts = (int) (esr / *p->icps);	/* Length of full delay */
	while (npts < 512) {	/* Minimum rail length is 256 */
		npts += (int) (esr / *p->icps);
		scale++;
	}
	rail_len = npts / 2 + 1;	/* but only need half length */
	if (*p->plk >= 1.0f || *p->plk <= 0.0f) {
/*       info << "Pluck point " << *p->plk << " invalid, using 0.5\n" << endmsg;
 */
		*p->plk = (p->ain ? 0.0f : 0.5f);
	}
	pickpt = (int) (rail_len * *p->plk);

	/* Create upper rail */
	if (p->upper.auxp == NULL) {	/* get newspace    */
		auxalloc(sizeof (DelayLine), &p->upper);
	}
	upper_rail = (DelayLine *) p->upper.auxp;
	upper_rail->length = rail_len;
	if (rail_len > 0) {
		auxalloc(rail_len * sizeof (float), &p->up_data);
		upper_rail->data = (float *) p->up_data.auxp;
	} else
		upper_rail->data = NULL;
	upper_rail->pointer = upper_rail->data;
	upper_rail->end = upper_rail->data + rail_len - 1;

	/* Create lower rail */
	if (p->lower.auxp == NULL) {	/* get newspace    */
		auxalloc(sizeof (DelayLine), &p->lower);
	}
	lower_rail = (DelayLine *) p->lower.auxp;
	lower_rail->length = rail_len;
	if (rail_len > 0) {
		auxalloc(rail_len * sizeof (float), &p->down_data);
		lower_rail->data = (float *) p->down_data.auxp;
	} else
		lower_rail->data = NULL;
	lower_rail->pointer = lower_rail->data;
	lower_rail->end = lower_rail->data + rail_len - 1;

	/* Set initial shape */
	if (*p->plk != 0.0) {
		initial_shape = (float *) mmalloc(rail_len * sizeof (float));
		if (pickpt < 1)
			pickpt = 1;	/* Place for pluck, in range (0,1.0) 
					 */
		upslope = 1.0f / (float) pickpt;	/* Slightly faster to
							   precalculate */
		downslope = 1.0f / (float) (rail_len - pickpt - 1);
		for (i = 0; i < pickpt; i++)
			initial_shape[i] = upslope * i;
		for (i = pickpt; i < rail_len; i++)
			initial_shape[i] = downslope * (rail_len - 1 - i);
		for (i = 0; i < rail_len; i++)
			upper_rail->data[i] = 0.5f * initial_shape[i];
		for (i = 0; i < rail_len; i++)
			lower_rail->data[i] = 0.5f * initial_shape[i];
		mfree((char *) initial_shape);
	} else {
		for (i = 0; i < rail_len; i++)
			upper_rail->data[i] = lower_rail->data[i] = 0.0f;
	}
	/* Copy data into structure */
	p->state = 0.0f;	/* filter memory */
	p->rail_len = rail_len;
	p->scale = scale;
}				/* end wgpset(p) */



				/* Access a delay line with wrapping */
static float *
locate(DelayLine * dl, int position)
{
	float *outloc = dl->pointer + position;
	while (outloc < dl->data)
		outloc += dl->length;
	while (outloc > dl->end)
		outloc -= dl->length;
	return outloc;
}

static float
getvalue(DelayLine * dl, int position)
{
	float *outloc = dl->pointer + position;
	while (outloc < dl->data)
		outloc += dl->length;
	while (outloc > dl->end)
		outloc -= dl->length;
	return *outloc;
}

#define OVERCNT (256)
#define OVERSHT (8)
#define OVERMSK (0xFF)

void
wgpluck(WGPLUCK2 * p)
{
	float *ar, *ain;
	int nsmps;
	float yp0, ym0, ypM, ymM;
	DelayLine *upper_rail;
	DelayLine *lower_rail;
	int pickup, pickfrac;
	int i;
	int scale;
	float state = p->state;
	float reflect = *p->reflect;

	if (reflect <= 0.0f || reflect >= 1.0f) {
		info << "Reflection invalid (" << reflect << ")\n" << endmsg;

		reflect = 0.5f;
	}
	ar = p->ar;
	ain = p->ain;
	scale = p->scale;
	reflect = 1.0f - (1.0f - reflect) / (float) scale;	/* For over
								   sapling */
	nsmps = ksmps;		/* Number of points to calculate */
	upper_rail = (DelayLine *) p->upper.auxp;
	lower_rail = (DelayLine *) p->lower.auxp;
	pickup = (int) ((float) OVERCNT * *(p->pickup) * p->rail_len);
	/* fract delays */
	pickfrac = pickup & OVERMSK;
	pickup = pickup >> OVERSHT;
	if (pickup < 0 || pickup > p->rail_len) {
		info << "Pickup out of range (" << p->pickup << ")\n" << endmsg;

		pickup = p->rail_len * (OVERCNT / 2);
		pickfrac = pickup & OVERMSK;
		pickup = pickup >> OVERSHT;
	}
	/* *** Start the loop .... *** */
	do {			/* while (--nsmps) */
		float s, s1;
		s = getvalue(upper_rail, pickup) + getvalue(lower_rail,
							    pickup);
		s1 = getvalue(upper_rail, pickup + 1) + getvalue(lower_rail,
							     pickup + 1);
		*ar = s + (s1 - s) * (float) pickfrac / (float) OVERCNT;
/* Fractional delay */
		if (ain != NULL) {	/* Excite the string from input */
			float *loc = locate(lower_rail, 1);
			*loc += (0.5f * *ain) / (*p->xamp);
			loc = locate(upper_rail, 1);
			*loc += (0.5f * *ain++) / (*p->xamp);
		}
		*ar++ *= *p->xamp;	/* Increment and scale */
		for (i = 0; i < scale; i++) {	/* Loop for precision
						   figure */
			ym0 = getvalue(lower_rail, 1);	/* Sample
							   traveling into "bridge" */
			ypM = getvalue(upper_rail, upper_rail->length - 2);
			/* Sample to "nut" */
			ymM = -ypM;	/* Inverting reflection at rigid nut 
					 */
			/* reflection at yielding bridge */
			/* Implement a one-pole lowpass with feedback
			   coefficient from input */
			state = (state * reflect) + ym0 * (1.0f - reflect);
			yp0 = -state;	/* String state update */
			/* Decrement pointer and then update */
			{
				float *ptr = upper_rail->pointer;
				ptr--;
				if (ptr < upper_rail->data)
					ptr = upper_rail->end;
				*ptr = yp0;
				upper_rail->pointer = ptr;
			}
			/* Update and then increment pointer */
			{
				float *ptr = lower_rail->pointer;
				*ptr = ymM;
				ptr++;
				if (ptr > lower_rail->end)
					ptr = lower_rail->data;
				lower_rail->pointer = ptr;
			}
		}
	} while (--nsmps);
	p->state = state;	/* Remember last state sample */

}				/* end wgpluck(p) */




OENTRY opcodes[] =
{
{"repluck", S(WGPLUCK2), 5, "a", "ikikka", F(wgpsetin), NULL, F(wgpluck)},
 {"wgpluck2", S(WGPLUCK2), 5, "a", "ikikk", F(wgpset), NULL, F(wgpluck)},
	{NULL}
};
