#include <math.h>
#include "cs.h"
#include "entry.h"
#include "insert.h"
#include "window.h"
#include "disprep.h"
#include "fft.h"
#include "dsputil.h"
#include "auxfd.h"
#include "musmon.h"
#include "fgens.h"

static float *fftcoefs = NULL;	/* malloc for fourier coefs, mag or db */

void
printv(PRINTV * p)
{
	int nargs = p->INOCOUNT;
	char *txtp = p->STRARG;
	float **valp = p->iargs;

	info << "instr " << p->h.insdshead->insno << ":" << endmsg;

	while (nargs--) {
		info << "  " << txtp << " = " << **valp++ << endmsg;

		while (*txtp++);
	}
	info << "\n" << endmsg;

}

void
dspset(DSPLAY * p)
{
	long npts, nprds, bufpts, totpts;
	char *auxp;

	if (p->h.optext->t.intype == 'k')
		npts = (long) (*p->iprd * ekr);
	else
		npts = (long) (*p->iprd * esr);
	if (npts <= 0) {
		error << "INIT: illegal iprd" << endmsg;

		return;
	}
	if ((nprds = (long) *p->inprds) <= 1) {
		nprds = 0;
		bufpts = npts;
		totpts = npts;
	} else {
		bufpts = npts * nprds;
		totpts = bufpts * 2;
	}
	if ((auxp = p->auxch.auxp) == NULL || totpts != p->totpts) {
		auxalloc(totpts * sizeof (float), &p->auxch);
		auxp = p->auxch.auxp;
		p->begp = (float *) auxp;
		p->endp = p->begp + bufpts;
		p->npts = npts;
		p->nprds = nprds;
		p->bufpts = bufpts;
		p->totpts = totpts;
	}
	p->nxtp = (float *) auxp;
	p->pntcnt = npts;
	sprintf(strmsg, "instr %d, signal %s:", p->h.insdshead->insno,
		p->STRARG);
	dispset(&p->dwindow, (float *) auxp, bufpts, strmsg, (int) *p->iwtflg,
		"display");
}

void
kdsplay(DSPLAY * p)
{
	float *fp = p->nxtp;

	if (p->auxch.auxp == NULL) {	/* RWD fix */
		error << "INIT: display: not initialized" << endmsg;

		return;
	}
	if (!p->nprds) {
		*fp++ = *p->signal;
		if (fp >= p->endp) {
			fp = p->begp;
			display(&p->dwindow);
		}
	} else {
		float *fp2 = fp + p->bufpts;
		*fp++ = *p->signal;
		*fp2 = *p->signal;
		if (!(--p->pntcnt)) {
			p->pntcnt = p->npts;
			if (fp >= p->endp) {
				fp = p->begp;
				fp2 = fp + p->bufpts;
			}
			p->dwindow.fdata = fp;	/* display from fp */
			display(&p->dwindow);
		}
	}
	p->nxtp = fp;
}

void
dsplay(DSPLAY * p)
{
	float *fp = p->nxtp, *sp = p->signal, *endp = p->endp;
	int nsmps = ksmps;

	if (!p->nprds) {
		do {
			*fp++ = *sp++;
			if (fp >= endp) {
				fp = p->begp;
				display(&p->dwindow);
			}
		} while (--nsmps);
	} else {
		float *fp2 = fp + p->bufpts;
		do {
			*fp++ = *sp;
			*fp2++ = *sp++;
			if (!(--p->pntcnt)) {
				p->pntcnt = p->npts;
				if (fp >= endp) {
					fp = p->begp;
					fp2 = fp + p->bufpts;
				}
				p->dwindow.fdata = fp;	/* display from
							   fp */
				display(&p->dwindow);
			}
		} while (--nsmps);
	}
	p->nxtp = fp;
}

void
fftset(DSPFFT * p)
{				/* fftset, dspfft -- calc Fast Fourier */
	/* Transform of collected samples and  */
	/* displays coefficients (mag or db)   */
	long window_size, step_size;
	int hanning;

	window_size = (long) *p->inpts;
	if (window_size > WINDMAX)
		error << "INIT: too many points requested" << endmsg;

	if (window_size < WINDMIN)
		error << "INIT: too few points requested" << endmsg;

	if (!IsPowerOfTwo(window_size))
		error << "INIT: window size must be power of two" << endmsg;

	if (p->h.optext->t.intype == 'k')
		step_size = (long) (*p->iprd * ekr);
	else
		step_size = (long) (*p->iprd * esr);
	if (step_size <= 0)
		error << "INIT: illegal iprd" << endmsg;

	if (inerrcnt)
		return;
	hanning = (int) *p->ihann;
	p->dbout = (int) *p->idbout;
	p->overlap = window_size - step_size;
	if (window_size != p->windsize
	    || hanning != p->hanning) {		/* if windowing has
						   changed:  */
		long auxsiz;
		float *hWin;
		p->windsize = window_size;	/* set new parameter values
						 */
		p->hanning = hanning;
		p->bufp = p->sampbuf;
		p->endp = p->bufp + window_size;
		p->overN = 1.0f / (*p->inpts);
		p->ncoefs = window_size >> 1;
		auxsiz = (window_size / 2 + 1) * sizeof (float);	/* size 
									   for half window */
		auxalloc((long) auxsiz, &p->auxch);	/*  alloc or realloc 
							 */
		hWin = (float *) p->auxch.auxp;
		FillHalfWin(hWin, window_size, 1.0f, hanning);	/* fill
								   with proper values */
		p->fftlut = (float *) AssignBasis((complex *) NULL,
						  window_size);		/*lookup tbl */
		if (fftcoefs == NULL)	/* room for WINDMAX*2 floats (fft
					   size) */
			fftcoefs = (float *) mmalloc((long) WINDMAX * 2 *
						     sizeof (float));
		sprintf(strmsg, "instr %d, signal %s, fft (%s):",
			p->h.insdshead->insno,
			p->STRARG, p->dbout ? "db" : "mag");
		dispset(&p->dwindow, fftcoefs, p->ncoefs, strmsg, (int)
			*p->iwtflg, "fft");
	}
}

static void
d_fft(				/* perform an FFT as reqd below */
	     float *sce,	/* input array - pure packed real */
	     float *dst,	/* output array - packed magnitude, only
				   half-length */
	     long size,		/* number of points in input */
	     float *lut,	/* look up table for FFT - already set up */
	     float *hWin,	/* hanning window lookup table */
	     int dbq)
{				/* flag: 1-> convert output into db */
	CopySamps(sce, dst, size);	/* copy into scratch buffer */
	ApplyHalfWin(dst, hWin, size);
	UnpackReals(dst, size);	/* expand out size to re,0,re,0 etc */
	FFT2real((complex *) dst, size, 1, (complex *) lut);	/* perform
								   the FFT */
	Rect2Polar(dst, size);
	PackReals(dst, size);
	if (dbq)
		Lin2DB(dst, size);
}

void
kdspfft(DSPFFT * p)
{
	float *bufp = p->bufp, *endp = p->endp;

	if (p->auxch.auxp == NULL) {	/* RWD fix */
		error << "INIT: dispfft: not initialized" << endmsg;

		return;
	}
	if (bufp < p->sampbuf)	/* skip any spare samples */
		bufp++;
	else {			/* then start collecting  */
		*bufp++ = *p->signal;
		if (bufp >= endp) {	/* when full, do fft:     */
			float *tp, *tplim;
			float *hWin = (float *) p->auxch.auxp;
			d_fft(p->sampbuf, fftcoefs, p->windsize, p->fftlut,
			      hWin, p->dbout);
			tp = fftcoefs;
			tplim = tp + p->ncoefs;
			do
				*tp *= p->overN;	/* scale 1/N */
			while (++tp < tplim);
			display(&p->dwindow);	/* & display */
			if (p->overlap > 0) {
				bufp = p->sampbuf;
				tp = endp - p->overlap;
				do
					*bufp++ = *tp++;
				while (tp < endp);
			} else
				bufp = p->sampbuf + p->overlap;
		}
	}
	p->bufp = bufp;
}

void
dspfft(DSPFFT * p)
{
	float *sigp = p->signal, *bufp = p->bufp, *endp = p->endp;
	int nsmps = ksmps;

	if (p->auxch.auxp == NULL) {
		error << "INIT: dispfft: not initialized" << endmsg;

		return;
	}
	do {
		if (bufp < p->sampbuf) {	/* skip any spare samples */
			bufp++;
			sigp++;
		} else {	/* then start collecting  */
			*bufp++ = *sigp++;
			if (bufp >= endp) {	/* when full, do fft:     */
				float *tp, *tplim;
				float *hWin = (float *) p->auxch.auxp;
				d_fft(p->sampbuf, fftcoefs, p->windsize,
				      p->fftlut, hWin, p->dbout);
				tp = fftcoefs;
				tplim = tp + p->ncoefs;
				do
					*tp *= p->overN;	/* scale 1/N */
				while (++tp < tplim);
				display(&p->dwindow);	/* & display */
				if (p->overlap > 0) {
					bufp = p->sampbuf;
					tp = endp - p->overlap;
					do
						*bufp++ = *tp++;
					while (tp < endp);
				} else
					bufp = p->sampbuf + p->overlap;
			}
		}
	}
	while (--nsmps);
	p->bufp = bufp;
}

#define NTERMS  4
#define NCROSS  (NTERMS * (NTERMS-1))

void
tempeset(TEMPEST * p)
{
	int npts, nptsm1, minlam, maxlam, lamspan, auxsiz;
	float *fltp;
	FUNC *ftp;
	float b, iperiod = *p->iprd;

	if ((p->timcount = (int) (ekr * iperiod)) <= 0)
		error << "INIT: illegal iperiod" << endmsg;

	if ((p->dtimcnt = (int) (ekr * *p->idisprd)) < 0)
		error << "INIT: illegal idisprd" << endmsg;

	if ((p->tweek = *p->itweek) <= 0)
		error << "INIT: illegal itweek" << endmsg;

	if (iperiod != 0.) {
		if ((minlam = (int) (*p->imindur / iperiod)) <= 0)
			error << "INIT: illegal imindur" << endmsg;

		if ((npts = (int) (*p->imemdur / iperiod)) <= 0)
			error << "INIT: illegal imemdur" << endmsg;

	}
	if (*p->ihtim <= 0.)
		error << "INIT: illegal ihtim" << endmsg;

	if (*p->istartempo <= 0.)
		error << "INIT: illegal startempo" << endmsg;

	ftp = ftfind(p->ifn);
	if (ftp != NULL && *ftp->ftable == 0.)
		error << "INIT: ifn table begins with zero" << endmsg;

	if (inerrcnt)
		return;		/* if errors so far, return */
	nptsm1 = npts - 1;
	if (npts != p->npts || minlam != p->minlam) {
		p->npts = npts;
		p->minlam = minlam;
		p->maxlam = maxlam = nptsm1 / (NTERMS - 1);
		lamspan = maxlam - minlam + 1;	/* alloc 8 bufs: 2 circ,
						   6 lin */
		auxsiz = (npts * 5 + lamspan * 3) * sizeof (float);
		auxalloc((long) auxsiz, &p->auxch);
		fltp = (float *) p->auxch.auxp;
		p->hbeg = fltp;
		fltp += npts;
		p->hend = fltp;
		p->xbeg = fltp;
		fltp += npts;
		p->xend = fltp;
		p->stmemp = fltp;
		fltp += npts;
		p->linexp = fltp;
		fltp += npts;
		p->ftable = fltp;
		fltp += npts;
		p->xscale = fltp;
		fltp += lamspan;
		p->lmults = fltp;
		fltp += lamspan;
		p->lambdas = (short *) fltp;
		p->stmemnow = p->stmemp + nptsm1;
	}
	if (p->dtimcnt && !(p->dwindow.windid)) {	/* init to display
							   stmem & exp */
		sprintf(strmsg, "instr %d tempest:", p->h.insdshead->insno);
		dispset(&p->dwindow, p->stmemp, (long) npts * 2, strmsg, 0,
			"tempest");
		p->dwindow.danflag = 1;		/* for mid-scale axis */
	} {
		float *funp = ftp->ftable;
		long phs = 0;
		long inc = (long) PMASK / npts;
		long nn, lobits = ftp->lobits;
		for (fltp = p->hbeg, nn = npts * 4; nn--;)	/* clr 2 circ 
								   & 1st 2 lin bufs */
			*fltp++ = 0.0f;
		for (fltp = p->ftable + npts, nn = npts; nn--;) {	/* now 
									   sample the ftable  */
			*--fltp = *(funp + (phs >> lobits));	/*
								   backwards into tbl buf */
			phs += inc;
		}
	}
	{
		float *tblp, sumraw, sumsqr;	/* calc the CROSS prod
						   scalers */
		long terms;
		long lambda, maxlam;
		float crossprods, RMS, *endtable = p->ftable + nptsm1;
/*            float coef, log001 = -6.9078; */
		float *xscale = p->xscale;

		p->ncross = (float) NCROSS;
		for (lambda = p->minlam, maxlam = p->maxlam; lambda <= maxlam;
		     lambda++) {
			tblp = endtable;
			sumraw = *tblp;
			sumsqr = *tblp * *tblp;
			terms = NTERMS - 1;
			do {
				tblp -= lambda;
				sumraw += *tblp;
				sumsqr += *tblp * *tblp;
			} while (--terms);
			crossprods = sumraw * sumraw - sumsqr;
			RMS = (float) sqrt(crossprods / p->ncross);
/*              coef = exp(log001 * lambda / npts);
   *xscale++ = coef / RMS / (NTERMS - 1);  */
			*xscale++ = 0.05f / RMS / lambda;
		}
	}
	b = 2.0f - (float) cos((*p->ihp * 6.28318 / ekr));	/* calc input 
								   lo-pass filter coefs */
	p->coef1 = b - (float) sqrt(b * b - 1.0);
	p->coef0 = 1.0f - p->coef1;
	p->yt1 = 0.0f;
	p->fwdcoef = (float) pow(0.5, p->timcount / ekr / (*p->ihtim));
	p->fwdmask = 0.0f;
	info << "kin lopass coef1 " << p->coef1 << ", fwd mask coef1 " << p->fwdcoef << "\n" << endmsg;

	p->thresh = *p->ithresh;	/* record incoming loudness threshold
					 */
	p->xfdbak = *p->ixfdbak;	/*    & expectation feedback fraction
					 */
	p->tempscal = 60.0f * ekr / p->timcount;
	p->avglam = p->tempscal / *p->istartempo;	/* init the tempo
							   factors */
	p->tempo = 0.0f;
	p->hcur = p->hbeg;	/* init the circular ptrs */
	p->xcur = p->xbeg;
	p->countdown = p->timcount;	/* & prime the countdowns */
	p->dcntdown = p->dtimcnt;
}

#define NMULTS 5
static float lenmults[NMULTS] =
{3.0f, 2.0f, 1.0f, .5f, .333f};
static float lenfracs[NMULTS * 2] =
{.30f, .3667f, .45f, .55f, .92f, 1.08f, 1.88f, 2.12f, 2.85f, 3.15f};

void
tempest(TEMPEST * p)
{
	p->yt1 = p->coef0 * *p->kin + p->coef1 * p->yt1;	/* get lo-pass
								   of kinput */

	if (p->auxch.auxp == NULL) {	/* RWD fix */
		error << "INIT: tempest: not initialized" << endmsg;

		return;
	}
	if (!(--p->countdown)) {	/* then on countdown:    */
		float *memp;
		float kin, expect, *xcur = p->xcur;	/* xcur from prv
							   pass    */
		float lamtot = 0.0f, weightot = 0.0f;

		p->countdown = p->timcount;	/* reset the countdown       
						 */
		expect = *xcur;	/* get expected val from prv calc */
		*xcur++ = 0.0f;	/*    & clear the loc it occupied */
		if (xcur >= p->xend)
			xcur = p->xbeg;		/* xcur now points to
						   cur xarray  */
		p->xcur = xcur;
		if ((kin = *p->kin - p->yt1) < 0.0f)
			kin = 0.0f;	/* ignore input below lopass */
		{
			float *hcur = p->hcur;
			float *hend = p->hend;
			float *tblp = p->ftable;
			long wrap;
			*hcur++ = kin + expect * p->xfdbak;	/* join
								   insample & expect val */
			if (hcur < hend)
				p->hcur = hcur;		/* stor pntr for 
							   next insamp  */
			else
				p->hcur = p->hbeg;
			wrap = hcur - p->hbeg;
			memp = p->stmemp;
			while (hcur < hend)	/* now lineariz & envlp hbuf 
						 */
				*memp++ = *hcur++ * *tblp++;	/*  into
								   st_mem buf          */
			for (hcur = p->hbeg; wrap--;)
				*memp++ = *hcur++ * *tblp++;
		}
		if (p->yt1 > p->thresh	/* if lo-pass of kinput now
					   significant */
		    && kin > p->fwdmask) {	/*    & kin > masking due to
						   prev kin   */
			float sumraw, sumsqr;
			long lambda, minlam, maxlam;
			int terms, nn, npts = p->npts;
			float mult, crossprods, RMScross, RMStot, unilam,
			 rd;
			float *xend = p->xend;
/*                float *xscale = p->xscale; */
			float *mults, *fracs, *mulp;
			short minlen, maxlen, *lenp, *endlens;

			for (memp = p->stmemp, nn = npts, sumsqr = 0.0f; nn--;
			     memp++)
				sumsqr += *memp * *memp;
			RMStot = (float) sqrt(sumsqr / npts);
/*        info << "RMStot = " << RMStot << "\n" << endmsg;
    */
			mults = lenmults;	/* use the static lentables 
						 */
			fracs = lenfracs;
			mulp = p->lmults;
			lenp = p->lambdas;
			minlam = p->minlam;
			maxlam = p->maxlam;
			nn = NMULTS;
			do {
				mult = *mults++;
				minlen = (short) (p->avglam * *fracs++);
/* & the current avglam  */
				maxlen = (short) (p->avglam * *fracs++);
				if (minlen >= minlam && maxlen <= maxlam)
					do {
						*lenp++ = minlen++;	/*   
									   creat lst of lambda lens */
						*mulp++ = mult;
						/*   & their unit multipliers */
					} while (minlen <= maxlen);
			} while (--nn);
			endlens = lenp;		/* now for these lambda
						   lens: */
			for (lenp = p->lambdas, mulp = p->lmults; lenp <
			     endlens;) {
				lambda = *lenp++;
				mult = *mulp++;
				memp = p->stmemnow;
				sumraw = *memp;
				sumsqr = *memp * *memp;		/*
								   autocorrelate the st_mem buf */
				terms = NTERMS - 1;
				do {
					memp -= lambda;
					sumraw += *memp;
					sumsqr += *memp * *memp;
				} while (--terms);
				crossprods = sumraw * sumraw - sumsqr;
				RMScross = (float) sqrt(crossprods /
							p->ncross);
				if (RMScross < 1.4f * RMStot)	/* if
								   RMScross significant:   */
					continue;
/*   info << "RMScross = " << RMScross << ", lambda = " << lambda   << "\n" << endmsg;
  */
/*                  RMS *= *xscale++;     */
				unilam = lambda * mult;		/*   
								   get unit lambda implied */
				lamtot += unilam * RMScross;	/*    & add 
								   weighted to total */
				weightot += RMScross;
/*    info << "lambda " << lambda << ", unilam " << unilam << ", RMScross " << RMScross << "\n" << endmsg;
  */
				RMScross /= 5.0f;
				memp = xcur - 1;	/* multiply project
							   into expect buf */
				for (terms = 1; terms < NTERMS; ++terms) {
					if ((memp += (lambda - terms + 1)) >=
					    xend)
						memp -= npts;
					for (nn = terms, rd = RMScross / terms;
					     nn--;) {
						*memp++ += rd;
						if (memp >= xend)
							memp -= npts;
					}
				}
			}
		}
		if (weightot) {	/* if accumed weights, */
			p->avglam = (p->avglam + lamtot / weightot) / 2.0f;
			/*   update the avglam */
			p->avglam /= p->tweek;
			p->tempo = p->tempscal / p->avglam;	/*   & cvt
								   to tempo    */
/*       info << "lamtot " << lamtot << ", weightot " << weightot << ", newavglam " << p->avglam << ", tempo " << p->tempo << "\n" << endmsg;
    */
/*       info << "" << p->tempo << "\n" << endmsg;
  */
			fputc('.', stderr);
		} else
			p->tempo = 0.0f;	/* else tempo is 0     */
		p->fwdmask = p->fwdmask * p->fwdcoef + kin;
	}
	if (!(--p->dcntdown)) {	/* on display countdown    */
		float *linp = p->linexp;
		float *xcur = p->xcur;
		float *xend = p->xend;
		long wrap = xcur - p->xbeg;
		while (xcur < xend)	/* lineariz the circ xbuf */
			*linp++ = *xcur++;	/*  into linexp buf       */
		for (xcur = p->xbeg; wrap--;)
			*linp++ = *xcur++;
		display(&p->dwindow);	/* display double window  */
		p->dcntdown = p->dtimcnt;	/*   & reset the counter  */
	}
	*p->kout = p->tempo;	/* put current tempo */
}



OENTRY opcodes[] =
{
	{"tempest", S(TEMPEST), 5, "k", "kiiiiiiiiiop", F(tempeset), NULL,
	 F(tempest)},
	{"print", S(PRINTV), 1, "", "m", F(printv)},
 {"display", S(DSPLAY), 7, "", "sioo", F(dspset), F(kdsplay), F(dsplay)},
{"dispfft", S(DSPFFT), 7, "", "siiooo", F(fftset), F(kdspfft), F(dspfft)},
	{NULL}
};
