/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holder's names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to 
    education and research.

    $Id: ugens6.cc,v 1.2 1999/11/17 12:22:07 stef Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/function_tables.h>
#include "ugens6.h"
#include <quasimodo/reverb.h>

static Number log001 = -6.9078f;	/* log(.001) */

void
downset(DOWNSAMP * p)
{
	if ((p->len = (size_t) *p->ilen) > ksmps)
		p->error ("ilen > ksmps");
}

void
downsamp(DOWNSAMP * p)
{
	Number *asig, sum;
	int16 len;

	if (p->len <= 1)
		*p->kr = *p->asig;
	else {
		asig = p->asig;
		sum = 0.0f;
		len = p->len;
		do
			sum += *asig++;
		while (--len);
		*p->kr = sum / p->len;
	}
}

void
upsamp(UPSAMP * p)
{
	Number *ar, kval;
	int16 nsmps = ksmps;

	ar = p->ar;
	kval = *p->ksig;
	do
		*ar++ = kval;
	while (--nsmps);
}

void
indfset(INDIFF * p)
{
	if (*p->istor == 0.0f)
		p->prev = 0.0f;
}

void
interp(INDIFF * p)
{
	Number *ar, val, incr;
	int16 nsmps = ksmps;

	ar = p->rslt;
	val = p->prev;
	incr = (*p->xsig - val) / ensmps;
	do
		*ar++ = val += incr;
	while (--nsmps);
	p->prev = val;
}

void
kntegrate(INDIFF * p)
{
	*p->rslt = p->prev += *p->xsig;
}

void
integrate(INDIFF * p)
{
	Number *rslt, *asig, sum;
	int16 nsmps = ksmps;

	rslt = p->rslt;
	asig = p->xsig;
	sum = p->prev;
	do
		*rslt++ = sum += *asig++;
	while (--nsmps);
	p->prev = sum;
}

void
kdiff(INDIFF * p)
{
	*p->rslt = *p->xsig - p->prev;
	p->prev = *p->xsig;
}

void
diff(INDIFF * p)
{
	Number *ar, *asig, prev;
	int16 nsmps = ksmps;

	ar = p->rslt;
	asig = p->xsig;
	prev = p->prev;
	do {
		*ar++ = *asig - prev;
		prev = *asig++;
	}
	while (--nsmps);
	p->prev = prev;
}

void
samphset(SAMPHOLD * p)
{
	if (!(*p->istor))
		p->state = *p->ival;
	p->audiogate = p->nth_arg_is_audio (1);
}

void
ksmphold(SAMPHOLD * p)
{
	if (*p->xgate > 0.)
		p->state = *p->xsig;
	*p->xr = p->state;
}

void
samphold(SAMPHOLD * p)
{
	Number *ar, *asig, *agate, state;
	int16 nsmps = ksmps;

	ar = p->xr;
	asig = p->xsig;
	state = p->state;
	if (p->audiogate) {
		agate = p->xgate;
		do {
			if (*agate++ > 0.)
				state = *asig;
			*ar++ = state;
			asig++;
		}
		while (--nsmps);
	} else {
		if (*p->xgate > 0.) {
			do
				*ar++ = state = *asig++;
			while (--nsmps);
		} else {
			do
				*ar++ = state;
			while (--nsmps);
		}
	}
	p->state = state;
}

void
delset(DELAY * p)
{
	size_t npts;
	char *auxp;

	if (*p->istor && p->auxch.auxp != NULL)
		return;
	if ((npts = (long) (*p->idlt * esr)) <= 0) {
		p->error ("illegal delay time");
	}
	if ((auxp = p->auxch.auxp) == NULL || npts != p->npts) {
		/* new space if reqd */
		p->auxch.alloc ((size_t) npts * sizeof (Number));
		auxp = p->auxch.auxp;
		p->npts = npts;
	} else if (!(*p->istor)) {	/* else if requested */
		int32 *lp = (int32 *) auxp;
		do
			*lp++ = 0;	/*   clr old to zero */
		while (--npts);
	}
	p->curp = (Number *) auxp;
}

static DELAYR *dlrdadr;

void
delrset(DELAYR * p)
{
	size_t npts;
	char *auxp;

	dlrdadr = p;		/* stor structadr for delayw */
	if (*p->istor && p->auxch.auxp != NULL)
		return;
	if ((npts = (size_t) (*p->idlt * esr)) < ksmps) { /* ksmps 
							     is min dely */
		p->error ("illegal delay time");
	}
	if ((auxp = p->auxch.auxp) == NULL || npts != p->npts) {
		/* new space if reqd */
		p->auxch.alloc ((size_t) npts * sizeof (Number));
		auxp = p->auxch.auxp;
		p->npts = npts;
	} else if (!(*p->istor)) {	/* else if requested */
		int32 *lp = (int32 *) auxp;
		do
			*lp++ = 0;	/*   clr old to zero */
		while (--npts);
	}
	p->curp = (Number *) auxp;
}

void
delwset(DELAYW * p)
{
	p->delayr = dlrdadr;	/* adr delayr struct */
}

void
tapset(DELTAP * p)
{
	p->delayr = dlrdadr;	/* adr delayr struct */
}

void
delay(DELAY * p)
{
	Number *ar, *asig, *curp, *endp;
	int16 nsmps = ksmps;

	if (p->auxch.auxp == NULL) {	/* RWD fix */
		p->error ("delay: not initialized");
	}
	ar = p->ar;
	asig = p->asig;
	curp = p->curp;
	endp = (Number *) p->auxch.endp;
	do {
		*ar++ = *curp;
		*curp = *asig++;
		if (++curp >= endp)
			curp = (Number *) p->auxch.auxp;
	}
	while (--nsmps);
	p->curp = curp;		/* sav the new curp */
}

void
delayr(DELAYR * p)
{
	Number *ar, *curp, *endp;
	int16 nsmps = ksmps;

	if (p->auxch.auxp == NULL) {	/* RWD fix */
		p->error ("delayr: not initialized");
	}
	ar = p->ar;
	curp = p->curp;
	endp = (Number *) p->auxch.endp;
	do {
		*ar++ = *curp++;
		if (curp >= endp)
			curp = (Number *) p->auxch.auxp;
	}
	while (--nsmps);
}				/* leave old curp */

void
delayw(DELAYW * p)
{
	DELAYR *q = p->delayr;
	Number *asig, *curp, *endp;
	int16 nsmps = ksmps;

	if (q->auxch.auxp == NULL) {	/* RWD fix */
		p->error ("delayw: not initialized");
	}
	asig = p->asig;
	curp = q->curp;
	endp = (Number *) q->auxch.endp;
	do {
		*curp = *asig++;
		if (++curp >= endp)
			curp = (Number *) q->auxch.auxp;
	}
	while (--nsmps);
	q->curp = curp;		/* now sav new curp */
}

void
deltap(DELTAP * p)
{
	DELAYR *q = p->delayr;
	Number *ar, *tap, *endp;
	int16 nsmps = ksmps;

	if (q->auxch.auxp == NULL) {	/* RWD fix */
		p->error ("deltap: not initialized");
	}
	ar = p->ar;
	if ((tap = q->curp - (long) (*p->xdlt * esr)) < (Number *)
	    q->auxch.auxp)
		tap += q->npts;
	endp = (Number *) q->auxch.endp;
	do {
		if (tap >= endp)
			tap -= q->npts;
		*ar++ = *tap++;
	}
	while (--nsmps);
}

void
deltapi(DELTAP * p)
{
	DELAYR *q = p->delayr;
	Number *ar, *tap, *prv, *begp, *endp;
	int16 nsmps = ksmps;
	int32 idelsmps;
	Number delsmps, delfrac;

	if (q->auxch.auxp == NULL) {
		p->error ("deltapi: not initialized");
	}
	ar = p->ar;
	begp = (Number *) q->auxch.auxp;
	endp = (Number *) q->auxch.endp;
	if (!p->has_audio_arguments()) {
		delsmps = *p->xdlt * esr;
		idelsmps = (long) delsmps;
		delfrac = delsmps - idelsmps;
		if ((tap = q->curp - idelsmps) < begp)
			tap += q->npts;
		do {
			if (tap >= endp)
				tap -= q->npts;
			if ((prv = tap - 1) < begp)
				prv += q->npts;
			*ar++ = *tap + (*prv - *tap) * delfrac;
			tap++;
		}
		while (--nsmps);
	} else {
		Number *timp = p->xdlt, *curq = q->curp;
		do {
			delsmps = *timp++ * esr;
			idelsmps = (long) delsmps;
			delfrac = delsmps - idelsmps;
			if ((tap = curq++ - idelsmps) < begp)
				tap += q->npts;
			else if (tap >= endp)
				tap -= q->npts;
			if ((prv = tap - 1) < begp)
				prv += q->npts;
			*ar++ = *tap + (*prv - *tap) * delfrac;
		}
		while (--nsmps);
	}
}

void
del1set(DELAY1 * p)
{
	if (!(*p->istor))
		p->sav1 = 0.0f;
}

void
delay1(DELAY1 * p)
{
	Number *ar, *asig;
	int16 nsmps = ksmps - 1;

	ar = p->ar;
	asig = p->asig;
	*ar++ = p->sav1;
	if (nsmps) {
		do
			*ar++ = *asig++;
		while (--nsmps);
	}
	p->sav1 = *asig;
}

void
cmbset(COMB * p)
{
	int32 lpsiz;
	size_t nbytes;

	if ((lpsiz = (int32) (*p->ilpt * esr)) <= 0) {
		p->error ("illegal loop time");
	}
	nbytes = lpsiz * sizeof (Number);
	if (p->auxch.auxp == NULL || nbytes != p->auxch.size) {
		p->auxch.alloc ((size_t) nbytes);
		p->pntr = (Number *) p->auxch.auxp;
		p->prvt = 0.0f;
	} else if (!(*p->istor)) {
		int32 *fp = (int32 *) p->auxch.auxp;
		p->pntr = (Number *) fp;
		do
			*fp++ = 0;
		while (--lpsiz);
		p->prvt = 0.0f;
	}
}

void
comb(COMB * p)
{
	int16 nsmps = ksmps;
	Number *ar, *asig, *xp, *endp;

	if (p->auxch.auxp == NULL) {	/* RWD fix */
		p->error ("comb: not initialized");
	}
	if (p->prvt != *p->krvt) {
		p->coef = (Number) exp((double) (log001 * *p->ilpt / *p->krvt));
		p->prvt = *p->krvt;
	}
	xp = p->pntr;
	endp = (Number *) p->auxch.endp;
	ar = p->ar;
	asig = p->asig;
	do {
		*ar++ = *xp;
		*xp *= p->coef;
		*xp += *asig++;
		if (++xp >= endp)
			xp = (Number *) p->auxch.auxp;
	}
	while (--nsmps);
	p->pntr = xp;
}

void
alpass(COMB * p)
{
	int16 nsmps = ksmps;
	Number *ar, *asig, *xp, *endp;
	Number y, z;

	if (p->auxch.auxp == NULL) {	/* RWD fix */
		p->error ("allpadd: not initialized");
	}
	if (p->prvt != *p->krvt) {
		p->coef = (Number) exp((double) (log001 * *p->ilpt / *p->krvt));
		p->prvt = *p->krvt;
	}
	xp = p->pntr;
	endp = (Number *) p->auxch.endp;
	ar = p->ar;
	asig = p->asig;
	do {
		y = *xp;
		*xp++ = z = p->coef * y + *asig++;
		*ar++ = y - p->coef * z;
		if (xp >= endp)
			xp = (Number *) p->auxch.auxp;
	}
	while (--nsmps);
	p->pntr = xp;
}

void
rvbset(REVERB * p)
{
	if (p->auxch.auxp == NULL) {	/* if no space yet, */
		int32 *sizp = revlpsiz;
		p->auxch.alloc (revlpsum * sizeof (Number));
		/*    allocate it   */
		p->adr1 = p->p1 = (Number *) p->auxch.auxp;
		p->adr2 = p->p2 = p->adr1 + *sizp++;
		p->adr3 = p->p3 = p->adr2 + *sizp++;	/*    & init ptrs   
							 */
		p->adr4 = p->p4 = p->adr3 + *sizp++;
		p->adr5 = p->p5 = p->adr4 + *sizp++;
		p->adr6 = p->p6 = p->adr5 + *sizp++;
		if (p->adr6 + *sizp != (Number *) p->auxch.endp) {
			fatal << "revlpsiz inconsistent\n" << endmsg;

		}
		p->prvt = 0.0f;
	}
	if (!(*p->istor)) {	/* else if istor = 0 */
		Number *fp = p->adr1;
		int32 nn = revlpsum;
		do
			*fp++ = 0.0f;	/*  clr existing spc */
		while (--nn);
		p->p1 = p->adr1;	/*  and reset   */
		p->p2 = p->adr2;
		p->p3 = p->adr3;
		p->p4 = p->adr4;
		p->p5 = p->adr5;
		p->p6 = p->adr6;
		p->prvt = 0.0f;
	}
}

void
reverb(REVERB * p)
{
	Number *asig, *p1, *p2, *p3, *p4, *p5, *p6, *ar, *endp;
	int16 nsmps = ksmps;

	if (p->auxch.auxp == NULL) {	/* RWD fix */
		p->error ("reverb: not intialized");
	}
	if (p->prvt != *p->krvt) {
		Number *lptimp = revlptimes;
		Number logdrvt = log001 / *p->krvt;
		p->c1 = (Number) exp(logdrvt * *lptimp++);
		p->c2 = (Number) exp(logdrvt * *lptimp++);
		p->c3 = (Number) exp(logdrvt * *lptimp++);
		p->c4 = (Number) exp(logdrvt * *lptimp++);
		p->c5 = (Number) exp(logdrvt * *lptimp++);
		p->c6 = (Number) exp(logdrvt * *lptimp++);
	}
	p1 = p->p1;
	p2 = p->p2;
	p3 = p->p3;
	p4 = p->p4;
	p5 = p->p5;
	p6 = p->p6;
	endp = (Number *) p->auxch.endp;

	ar = p->ar;
	asig = p->asig;
	do {
		Number cmbsum, y1, y2, z;
		cmbsum = *p1 + *p2 + *p3 + *p4;
		*p1 = p->c1 * *p1 + *asig;
		*p2 = p->c2 * *p2 + *asig;
		*p3 = p->c3 * *p3 + *asig;
		*p4 = p->c4 * *p4 + *asig++;
		p1++;
		p2++;
		p3++;
		p4++;
		y1 = *p5;
		*p5++ = z = p->c5 * y1 + cmbsum;
		y1 -= p->c5 * z;
		y2 = *p6;
		*p6++ = z = p->c6 * y2 + y1;
		*ar++ = y2 - p->c6 * z;
		if (p1 >= p->adr2)
			p1 = p->adr1;
		if (p2 >= p->adr3)
			p2 = p->adr2;
		if (p3 >= p->adr4)
			p3 = p->adr3;
		if (p4 >= p->adr5)
			p4 = p->adr4;
		if (p5 >= p->adr6)
			p5 = p->adr5;
		if (p6 >= endp)
			p6 = p->adr6;
	}
	while (--nsmps);
	p->p1 = p1;
	p->p2 = p2;
	p->p3 = p3;
	p->p4 = p4;
	p->p5 = p5;
	p->p6 = p6;
}

void
panset(PAN * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind(p->ifn)) == 0) {
		p->error ("no table for pan");
		return;
	}

	p->ftp = ftp;

	if (*p->imode) {
		p->xmul = (Number) ftp->flen;
	} else {
		p->xmul = 1.0f;
	}

	if (*p->ioffset) {
		p->xoff = ftp->flen >> 1;
	} else {
		p->xoff = 0;
	}
}

void
pan(PAN * p)
{
	Number *r1, *r2, *r3, *r4, *sigp, ch1, ch2, ch3, ch4;
	int32 xndx, yndx, flen;
	int16 nsmps = ksmps;
	RCPointer<FunctionTable> ftp;

	ftp = p->ftp;
	flen = ftp->flen;
	xndx = (long) (*p->kx * p->xmul) - p->xoff;
	yndx = (long) (*p->ky * p->xmul) - p->xoff;
	if (xndx < 0L || xndx > flen
	    || yndx < 0L || yndx > flen) {
		int32 xt, yt, off = flen >> 1;
		xt = xndx - off;
		yt = yndx - off;
		if (xt * xt > yt * yt) {
			if (xt < 0)
				xt = -xt;
			yndx = yt * off / xt + off;
		} else {
			if (yt < 0)
				yt = -yt;
			xndx = xt * off / yt + off;
		}
		if (xndx < 0)
			xndx = 0;
		else if (xndx > flen)
			xndx = flen;
		if (yndx < 0)
			yndx = 0;
		else if (yndx > flen)
			yndx = flen;
	}
	ch2 = *(ftp->ftable + xndx) * *(ftp->ftable + yndx);
	ch4 = *(ftp->ftable + xndx) * *(ftp->ftable + flen - yndx);
	ch1 = *(ftp->ftable + flen - xndx) * *(ftp->ftable + yndx);
	ch3 = *(ftp->ftable + flen - xndx) * *(ftp->ftable + flen - yndx);
	r1 = p->r1;
	r2 = p->r2;
	r3 = p->r3;
	r4 = p->r4;
	sigp = p->asig;
	do {
		*r1++ = *sigp * ch1;
		*r2++ = *sigp * ch2;
		*r3++ = *sigp * ch3;
		*r4++ = *sigp * ch4;
		sigp++;
	}
	while (--nsmps);
}


Opcode opcodes[] =
{
	UGENS6_OPCODE_LIST,
	{NULL}
};
