/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: ugens1.cc,v 1.2 1999/11/29 18:49:28 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dsp.h>
#include <quasimodo/process.h>

#include "ugens1.h"

void
linset(LINE * p)
{
	Number dur;

	if ((dur = *p->idur) > FZERO) {
		p->incr = (*p->ib - *p->ia) / dur * onedkr;
		p->val = *p->ia;
	}
}

void
kline(LINE * p)
{
	*p->xr = p->val;	/* rslt = val   */
	p->val += p->incr;	/* val += incr  */
}

void
aline(LINE * p)
{
	Number val, inc, *ar;
	int nsmps = ksmps;

	val = p->val;
	inc = p->incr;
	p->val += inc;		/* nxtval = val + inc */
	inc /= ensmps;
	ar = p->xr;
	do {
		*ar++ = val;
		val += inc;	/* interp val for ksmps */
	}
	while (--nsmps);
}

void
expset(EXPON * p)
{
	Number dur, a, b;

	if ((dur = *p->idur) > FZERO) {
		a = *p->ia;
		b = *p->ib;
		if ((a * b) > FZERO) {
			p->mlt = (Number) pow((double) (b / a), (onedkr /
							  (double) dur));
			p->val = a;
		} else if (a == FZERO)
			p->error ("arg1 is zero");
		else if (b == FZERO)
			p->error ("arg2 is zero");
		else
			p->error ("unlike signs");
	}
}

void
kexpon(EXPON * p)
{
	*p->xr = p->val;	/* rslt = val   */
	p->val *= p->mlt;	/* val *= mlt  */
}

void
expon(EXPON * p)
{
	Number val, mlt, inc, *ar, nxtval;
	int nsmps = ksmps;

	val = p->val;
	mlt = p->mlt;
	nxtval = val * mlt;
	inc = nxtval - val;
	inc /= ensmps;		/* increment per sample */
	ar = p->xr;
	do {
		*ar++ = val;
		val += inc;	/* interp val for ksmps */
	}
	while (--nsmps);
	p->val = nxtval;	/*store next value */
}

void
lsgset(LINSEG * p)
{
	SEG *segp;
	int nsegs;
	Number **argp, val;

	nsegs = (p->INOCOUNT) >> 1;	/* count segs & alloc if nec */
	if ((segp = (SEG *) p->auxch.auxp) == NULL ||
	    nsegs * sizeof (SEG) < (size_t) p->auxch.size) {
		p->auxch.alloc ((size_t) nsegs * sizeof (SEG));
		p->cursegp = segp = (SEG *) p->auxch.auxp;
		segp[nsegs - 1].cnt = MAXPOS;	/* set endcount for safety 
						 */
	}
	argp = p->argums;
	val = **argp++;
	if (**argp <= FZERO)
		return;		/* if idur1 <= 0, skip init  */
	p->curval = val;
	p->curcnt = 0;
	p->cursegp = segp - 1;	/* else setup null seg0 */
	p->segsrem = nsegs + 1;
	do {			/* init each seg ..  */
		Number dur = **argp++;
		segp->nxtpt = **argp++;
		if ((segp->cnt = (int32) (dur * ekr + 0.5f)) < 0)
			segp->cnt = 0;
		segp++;
	} while (--nsegs);
}

void
klnseg(LINSEG * p)
{
	*p->rslt = p->curval;	/* put the cur value    */
	if (p->auxch.auxp == NULL) {	/* RWD fix */
		fatal << "Error: linseg not initialized (krate)" << endmsg;

	}
	if (p->segsrem) {	/* done if no more segs */
		while (--p->curcnt < 0	/* if done cur segment  */
		       && p->segsrem) {
			SEG *segp = ++p->cursegp;	/*   find the next     
							 */
			if (!(--p->segsrem))
				return;
			if (!(p->curcnt = segp->cnt))	/*   nonlen =
							   discontin */
				p->curval = segp->nxtpt;	/*   poslen =
								   new slope */
			else
				p->curinc = (segp->nxtpt - p->curval) /
				    segp->cnt;
		}
		p->curval += p->curinc;		/* advance the cur val 
						 */
	}
}

void
linseg(LINSEG * p)
{
	Number val, ainc, *rs = p->rslt;
	int nsmps = ksmps;

	val = p->curval;	/* sav the cur value    */
	if (p->segsrem) {	/* if no more segs putk */
		if (--p->curcnt < 0) {	/*  if done cur segment */
			SEG *segp = p->cursegp;
		      chk1:
			if (!--p->segsrem) {	/*   if none left       */
				val = p->curval = segp->nxtpt;
				goto putk;	/*      put endval      */
			}
			p->cursegp = ++segp;	/*   else find the next */
			if (!(p->curcnt = segp->cnt)) {
				val = p->curval = segp->nxtpt;	/* nonlen 
								   = discontin */
				goto chk1;
			}	/*   poslen = new slope */
			p->curinc = (segp->nxtpt - val) / segp->cnt;
			p->curainc = p->curinc * dvensmps;
		}
		p->curval = val + p->curinc;	/* advance the cur val  */
		if ((ainc = p->curainc) == FZERO)
			goto putk;
		do {
			*rs++ = val;
			val += ainc;
		} while (--nsmps);
	} else {
	      putk:
		do
			*rs++ = val;
		while (--nsmps);
	}
}

/* **** ADSR is just a construction and use of linseg */


void
adsrset(LINSEG * p)
{
	SEG *segp;
	int nsegs;
	Number **argp;
	Number dur;

	nsegs = 6;		/* DADSR */
	if ((segp = (SEG *) p->auxch.auxp) == NULL ||
	    nsegs * sizeof (SEG) < (unsigned int) p->auxch.size) {
		p->auxch.alloc ((size_t) nsegs * sizeof (SEG));
		p->cursegp = segp = (SEG *) p->auxch.auxp;
		segp[nsegs - 1].cnt = MAXPOS;	/* set endcount for safety 
						 */
	}
	argp = p->argums;
	if (**argp <= 0.0f)
		return;		/* if idur1 <= 0, skip init  */
	p->curval = 0.0f;
	p->curcnt = 0;
	p->cursegp = segp - 1;	/* else setup null seg0 */
	p->segsrem = nsegs + 1;
	/* Delay */
	dur = *argp[4];
	segp->nxtpt = 0.0f;
	if ((segp->cnt = (int32) (dur * ekr + 0.5f)) < 0)
		segp->cnt = 0;
	segp++;
	/* Attack */
	dur = *argp[0];
	segp->nxtpt = 1.0f;
	if ((segp->cnt = (int32) (dur * ekr + 0.5f)) < 0)
		segp->cnt = 0;
	segp++;
	/* Decay */
	dur = *argp[1];
	segp->nxtpt = *argp[2];
	if ((segp->cnt = (int32) (dur * ekr + 0.5f)) < 0)
		segp->cnt = 0;
	segp++;

	/* Sustain */
	/* Should use p3 from score, but how.... PBD: this is how !!!*/

	if (p->PROCESS->score_duration() <= 0.0f) {
		dur = 10000.0f;	/* Just in case! */
	} else {
		dur = p->PROCESS->score_duration() - 
		          *argp[4] - *argp[0] - *argp[1] - *argp[3];
	}

	segp->nxtpt = *argp[2];
	if ((segp->cnt = (int32) (dur * ekr + 0.5f)) < 0)
		segp->cnt = 0;
	segp++;
	/* Release */
	dur = *argp[3];
	segp->nxtpt = 0.0f;
	if ((segp->cnt = (int32) (dur * ekr + 0.5f)) < 0)
		segp->cnt = 0;
}

/* End of ADSR */

void
lsgrset(LINSEG * p)
{
	uint32 relestim;
	lsgset(p);
	relestim = (p->cursegp + p->segsrem - 1)->cnt;
	if (relestim > p->PROCESS->extra_time)
		p->PROCESS->extra_time = (short) relestim;
}

void
klnsegr(LINSEG * p)
{
	*p->rslt = p->curval;	/* put the cur value    */
	if (p->segsrem) {	/* done if no more segs */
		SEG *segp;
		if (p->PROCESS->releasing() && p->segsrem > 1) {
			while (p->segsrem > 1) {	/* reles flag new:      
							 */
				segp = ++p->cursegp;	/*   go to last
							   segment */
				p->segsrem--;
			}	/*   get univ relestim  */
			segp->cnt = p->PROCESS->extra_time;
			goto newi;	/*   and set new curinc */
		}
		if (--p->curcnt < 0) {	/* if done cur seg      */
		      chk2:
			if (p->segsrem == 2)
				return;		/*   seg Y rpts lastval
						 */
			if (!(--p->segsrem))
				return;		/*   seg Z now done all
						 */
			segp = ++p->cursegp;	/*   else find nextseg  */
		      newi:
			if (!(p->curcnt = segp->cnt)) {		/*  
								   nonlen = discontin */
				p->curval = segp->nxtpt;	/*     reload & 
								   rechk   */
				goto chk2;
			}	/*   else get new slope */
			p->curinc = (segp->nxtpt - p->curval) / segp->cnt;
		}
		p->curval += p->curinc;		/* advance the cur val 
						 */
	}
}

void
linsegr(LINSEG * p)
{
	Number val, ainc, *rs = p->rslt;
	int nsmps = ksmps;

	val = p->curval;	/* sav the cur value    */
	if (p->segsrem) {	/* if no more segs putk */
		SEG *segp;
		if (p->PROCESS->releasing() && p->segsrem > 1) {
			while (p->segsrem > 1) {	/* reles flag new:      
							 */
				segp = ++p->cursegp;	/*   go to last
							   segment */
				p->segsrem--;
			}	/*   get univ relestim  */
			segp->cnt = p->PROCESS->extra_time;
			goto newi;	/*   and set new curinc */
		}
		if (--p->curcnt < 0) {	/* if done cur seg      */
		      chk2:
			if (p->segsrem == 2)
				goto putk;	/*   seg Y rpts lastval */
			if (!(--p->segsrem))
				goto putk;	/*   seg Z now done all */
			segp = ++p->cursegp;	/*   else find nextseg  */
		      newi:
			if (!(p->curcnt = segp->cnt)) {		/*  
								   nonlen = discontin */
				val = p->curval = segp->nxtpt;	/*  
								   reload & rechk  */
				goto chk2;
			}	/*   else get new slope */
			p->curinc = (segp->nxtpt - val) / segp->cnt;
			p->curainc = p->curinc * dvensmps;
		}
		p->curval = val + p->curinc;	/* advance the cur val  */
		if ((ainc = p->curainc) == FZERO)
			goto putk;
		do {
			*rs++ = val;
			val += ainc;
		} while (--nsmps);
	} else {
	      putk:
		do
			*rs++ = val;
		while (--nsmps);
	}
}

void
xsgset(EXXPSEG * p)
{
	XSEG *segp;
	int nsegs;
	Number d, **argp, val, dur, nxtval;
	int n;

	nsegs = p->INOCOUNT >> 1;	/* count segs & alloc if nec */
	if ((segp = (XSEG *) p->auxch.auxp) == NULL) {
		p->auxch.alloc ((size_t) nsegs * sizeof (XSEG));
		p->cursegp = segp = (XSEG *) p->auxch.auxp;
		(segp + nsegs - 1)->cnt = MAXPOS;	/* set endcount for
							   safety */
	}
	argp = p->argums;
	nxtval = **argp++;
	if (**argp <= FZERO)
		return;		/* if idur1 <= 0, skip init  */
	p->cursegp = segp;	/* else proceed from 1st seg */
	segp--;
	do {
		segp++;		/* init each seg ..  */
		val = nxtval;
		dur = **argp++;
		nxtval = **argp++;
		if (dur > FZERO) {
			if (val * nxtval <= FZERO)
				goto experr;
			d = dur * ekr;
			segp->val = val;
			segp->mlt = (Number) pow((double) (nxtval / val), (1.0 /
							    (double) d));
			segp->cnt = (int32) (d + 0.5f);
		} else
			break;	/*  .. til 0 dur or done */
	} while (--nsegs);
	segp->cnt = MAXPOS;	/* set last cntr to infin */
	return;

      experr:
	n = segp - p->cursegp + 1;
	if (val == FZERO) {
		p->error ("ival%d is zero", n);
	} else {
		p->error ("ival%d sign conflict", n + 1);
	}
}

void 
xdsrset(EXXPSEG *p)

{
        XSEG	*segp;
        int16	nsegs;
        Number	**argp = p->argums;
        Number	len = p->PROCESS->duration ();
        Number  delay = *argp[4], attack = *argp[0], decay = *argp[1];
        Number  sus, dur;
        Number  release = *argp[3];

        if (len<0.0f) len = 100000.0f; /* MIDI case set int32 */
        len -= release;         /* len is time remaining */
        if (len<0.0f) {         /* Odd case of release time greater than dur */
		release = p->PROCESS->duration (); len = 0.0f;
        }
	nsegs = 5;		/* DXDSR */
	if ((segp = (XSEG *) p->auxch.auxp) == NULL ||
            nsegs*sizeof(XSEG) < (unsigned int)p->auxch.size) {
		p->auxch.alloc ((int32)nsegs*sizeof(XSEG));
		segp = (XSEG *) p->auxch.auxp;
		segp[nsegs-1].cnt = MAXPOS; /* set endcount for safety */
	}
	if (**argp <= 0.0f)  return;		/* if idur1 <= 0, skip init  */
	p->cursegp = segp;          /* else setup null seg0 */
	p->segsrem = nsegs;
        delay += 0.001f;
        if (delay > len) delay = len; len -= delay;
        attack -= 0.001f;
	if (attack > len) attack = len; len -= attack;
  	if (decay > len) decay = len; len -= decay;
	sus = len;
        segp[0].val = 0.001f;   /* Like zero start, but exponential */
        segp[0].mlt = 1.0f;
        segp[0].cnt = (int32) (delay*ekr + 0.5f);
        dur = attack*ekr;
        segp[1].val = 0.001f;
        segp[1].mlt = (Number) pow(1000.0, 1.0/(double)dur);
        segp[1].cnt = (int32) (dur + 0.5f);
        dur = decay*ekr;
        segp[2].val = 1.0f;
        segp[2].mlt = (Number) pow((double)*argp[2], 1.0/(double)dur);
        segp[2].cnt = (int32) (dur + 0.5f);
        segp[3].val = *argp[2];
        segp[3].mlt = 1.0f;
        segp[3].cnt = (int32) (sus*ekr + 0.5f);
        dur = release*ekr;
        segp[4].val = *argp[2];
        segp[4].mlt = (Number) pow(0.001/(double)*argp[2], 1.0/(double)dur);
        segp[4].cnt = (int32) (dur + 0.5f);
}

void
kxpseg(EXXPSEG * p)
{
	XSEG *segp;

	segp = p->cursegp;
	while (--segp->cnt < 0)
		p->cursegp = ++segp;
	*p->rslt = segp->val;
	segp->val *= segp->mlt;
}

void
expseg(EXXPSEG * p)
{
	XSEG *segp;
	int nsmps = ksmps;
	Number li, val, *rs;
	Number nxtval;

	segp = p->cursegp;
	while (--segp->cnt < 0)
		p->cursegp = ++segp;
	val = segp->val;
	nxtval = val * segp->mlt;
	li = (nxtval - val) / ensmps;
	rs = p->rslt;
	do {
		*rs++ = val;
		val += li;
	} while (--nsmps);
	segp->val = nxtval;
}

void
xsgrset(EXPSEG * p)
{
	unsigned short relestim;
	SEG *segp;
	int nsegs, n;
	Number **argp, prvpt;

	nsegs = p->INOCOUNT >> 1;	/* count segs & alloc if nec */
	if ((segp = (SEG *) p->auxch.auxp) == NULL) {
		p->auxch.alloc ((size_t) nsegs * sizeof (SEG));
		p->cursegp = segp = (SEG *) p->auxch.auxp;
	}
	argp = p->argums;
	prvpt = **argp++;
	if (**argp < FZERO)
		return;		/* if idur1 < 0, skip init      */
	p->curval = prvpt;
	p->curcnt = 0;		/* else setup null seg0         */
	p->cursegp = segp - 1;
	p->segsrem = nsegs + 1;
	do {			/* init & chk each real seg ..  */
		Number dur = **argp++;
		segp->nxtpt = **argp++;
		if ((segp->cnt = (int32) (dur * ekr + 0.5f)) <= 0)
			segp->cnt = 0;
		else if (segp->nxtpt * prvpt <= FZERO)
			goto experr;
		prvpt = segp->nxtpt;
		segp++;
	} while (--nsegs);
	relestim = (unsigned short) (p->cursegp + p->segsrem - 1)->cnt;
	if (relestim > p->PROCESS->extra_time)
		p->PROCESS->extra_time = relestim;
	return;

      experr:
	n = segp - p->cursegp + 2;
	if (prvpt == FZERO)
		p->error ("ival%d is zero", n);
	else
		p->error ("ival%d sign conflict", n + 1);
}

void mxdsrset(EXPSEG *p)
{
	uint16  relestim;
	SEG    *segp;
	int32	nsegs;
	Number	**argp = p->argums;
	Number  delay = *argp[4], attack = *argp[0], decay = *argp[1];
	Number	rel = *argp[3];

	nsegs = 4;		/* DXDSR */
	if ((segp = (SEG *) p->auxch.auxp) == NULL ||
	    nsegs*sizeof(SEG) < (unsigned int)p->auxch.size) {
		p->auxch.alloc ((int32)nsegs*sizeof(SEG));
		segp = (SEG *) p->auxch.auxp;
	}
	if (**argp <= 0.0f)  return;		/* if idur1 <= 0, skip init  */
	p->cursegp = segp-1;          /* else setup null seg0 */
	p->segsrem = nsegs+1;
	p->curval = 0.001f;
	p->curcnt = 0;                  /* else setup null seg0         */
	delay += 0.001f;
	attack -= 0.001f;
	segp[0].nxtpt = 0.001f;
	segp[0].cnt = (int32) (delay*ekr + 0.5f);
	segp[1].nxtpt = 1.0f;
	segp[1].cnt = (int32) (attack*ekr + 0.5f);
	segp[2].nxtpt = *argp[2];
	segp[2].cnt = (int32) (decay*ekr + 0.5f);
	segp[3].nxtpt = 0.001f;
	segp[3].cnt = (int32) (rel*ekr + 0.5f);
	relestim = (uint16)(p->cursegp + p->segsrem - 1)->cnt;

	if (relestim > p->PROCESS->extra_time) {
		p->PROCESS->extra_time = relestim;
	}
}

void
kxpsegr(EXPSEG * p)
{
	*p->rslt = p->curval;	/* put the cur value    */
	if (p->segsrem) {	/* done if no more segs */
		SEG *segp;
		if (p->PROCESS->releasing() && p->segsrem > 1) {
			while (p->segsrem > 1) {	/* reles flag new:      
							 */
				segp = ++p->cursegp;	/*   go to last
							   segment */
				p->segsrem--;
			}	/*   get univ relestim  */
			segp->cnt = p->PROCESS->extra_time;
			goto newm;	/*   and set new curmlt */
		}
		if (--p->curcnt < 0) {	/* if done cur seg      */
		      chk2:
			if (p->segsrem == 2)
				return;		/*   seg Y rpts lastval
						 */
			if (!(--p->segsrem))
				return;		/*   seg Z now done all
						 */
			segp = ++p->cursegp;	/*   else find nextseg  */
		      newm:
			if (!(p->curcnt = segp->cnt)) {		/*  
								   nonlen = discontin */
				p->curval = segp->nxtpt;	/*     reload & 
								   rechk   */
				goto chk2;
			}
			if (segp->nxtpt == p->curval)	/*   else get new
							   mlt   */
				p->curmlt = FONE;
			else
				p->curmlt = (Number) pow(segp->nxtpt /
					      p->curval, 1. / segp->cnt);
		}
		p->curval *= p->curmlt;		/* advance the cur val 
						 */
	}
}

void
expsegr(EXPSEG * p)
{
	Number val, amlt, *rs = p->rslt;
	int nsmps = ksmps;

	val = p->curval;	/* sav the cur value    */
	if (p->segsrem) {	/* if no more segs putk */
		SEG *segp;
		if (p->PROCESS->releasing() && p->segsrem > 1) {
			while (p->segsrem > 1) {	/* if reles flag new    
							 */
				segp = ++p->cursegp;	/*   go to last
							   segment */
				p->segsrem--;
			}	/*   get univ relestim  */
			segp->cnt = p->PROCESS->extra_time;
			goto newm;	/*   and set new curmlt */
		}
		if (--p->curcnt < 0) {	/* if done cur seg      */
		      chk2:
			if (p->segsrem == 2)
				goto putk;	/*   seg Y rpts lastval */
			if (!(--p->segsrem))
				goto putk;	/*   seg Z now done all */
			segp = ++p->cursegp;	/*   else find nextseg  */
		      newm:
			if (!(p->curcnt = segp->cnt)) {		/*  
								   nonlen = discontin */
				val = p->curval = segp->nxtpt;	/*  
								   reload & rechk  */
				goto chk2;
			}	/*   else get new mlts  */
			if (segp->nxtpt == val) {
				p->curmlt = p->curamlt = FONE;
				p->curval = val;
				goto putk;
			} else {
				p->curmlt = (Number) pow((double) (segp->nxtpt /
					 val), 1.0 / (double) segp->cnt);
				p->curamlt = (Number) pow(p->curmlt, dvensmps);
			}
		}
		p->curval = val * p->curmlt;	/* advance the cur val  */
		if ((amlt = p->curamlt) == FONE)
			goto putk;
		do {
			*rs++ = val;
			val *= amlt;
		} while (--nsmps);
	} else {
	      putk:
		do
			*rs++ = val;
		while (--nsmps);
	}
}

void
lnnset(LINEN * p)
{
	Number a, b, dur;

	if ((dur = *p->idur) > FZERO) {
		p->cnt1 = (int32) (*p->iris * ekr + 0.5f);
		if (p->cnt1 > 0L) {
			p->inc1 = FONE / (Number) p->cnt1;
			p->val = FZERO;
		} else
			p->inc1 = p->val = FONE;
		a = dur * ekr + 0.5f;
		b = *p->idec * ekr + 0.5f;
		if ((int32) b > 0L) {
			p->cnt2 = (int32) (a - b);
			p->inc2 = FONE / b;
		} else {
			p->inc2 = FONE;
			p->cnt2 = (int32) a;
		}
		p->lin1 = FZERO;
		p->lin2 = FONE;
	}
}

void
klinen(LINEN * p)
{
	Number fact = FONE;

	if (p->cnt1 > 0L) {
		fact = p->lin1;
		p->lin1 += p->inc1;
		p->cnt1--;
	}
	if (p->cnt2)
		p->cnt2--;
	else {
		fact *= p->lin2;
		p->lin2 -= p->inc2;
	}
	*p->rslt = *p->sig * fact;
}

void
linen(LINEN * p)
{
	int flag = 0, nsmps = ksmps;
	Number *rs, *sg, li, val, nxtval = 1.0f;

	val = p->val;
	rs = p->rslt;
	sg = p->sig;
	if (p->cnt1 > 0L) {
		flag = 1;
		p->lin1 += p->inc1;
		p->cnt1--;
		nxtval = p->lin1;
	}
	if (p->cnt2 <= 0L) {
		flag = 1;
		p->lin2 -= p->inc2;
		nxtval *= p->lin2;
	} else
		p->cnt2--;
	p->val = nxtval;
	if (flag) {
		li = (nxtval - val) / ensmps;
		if (p->has_audio_arguments()) {
			do {
				*rs++ = *sg++ * val;
				val += li;
			}
			while (--nsmps);
		} else {
			do {
				*rs++ = *sg * val;
				val += li;
			}
			while (--nsmps);
		}
	} else {
		if (p->has_audio_arguments()) {
			do
				*rs++ = *sg++;
			while (--nsmps);
		} else {
			do
				*rs++ = *sg;
			while (--nsmps);
		}
	}
}

void
lnrset(LINENR * p)
{
	p->cnt1 = (int32) (*p->iris * ekr + 0.5f);
	if (p->cnt1 > 0L) {
		p->inc1 = FONE / (Number) p->cnt1;
		p->val = FZERO;
	} else
		p->inc1 = p->val = FONE;
	if (*p->idec > FZERO) {
		unsigned short relestim = (unsigned short) (*p->idec * ekr +
							    0.5f);
		if (relestim > p->PROCESS->extra_time)
			p->PROCESS->extra_time = relestim;
		if (*p->iatdec <= FZERO)
			p->error ("non-positive iatdec.");
		else
			p->mlt2 = (Number) pow((double) *p->iatdec, ((double)
						     onedkr / *p->idec));
	} else
		p->mlt2 = FONE;
	p->lin1 = FZERO;
	p->val2 = FONE;
}

void
klinenr(LINENR * p)
{
	Number fact = FONE;

	if (p->cnt1 > 0L) {
		fact = p->lin1;
		p->lin1 += p->inc1;
		p->cnt1--;
	}
	if (p->PROCESS->releasing()) {
		fact *= p->val2;
		p->val2 *= p->mlt2;
	}
	*p->rslt = *p->sig * fact;
}

void
linenr(LINENR * p)
{
	int flag = 0, nsmps = ksmps;
	Number *rs, *sg, li, val, nxtval = FONE;

	val = p->val;
	rs = p->rslt;
	sg = p->sig;
	if (p->cnt1 > 0L) {
		flag = 1;
		p->lin1 += p->inc1;
		p->cnt1--;
		nxtval = p->lin1;
	}
	if (p->PROCESS->releasing()) {
		flag = 1;
		p->val2 *= p->mlt2;
		nxtval *= p->val2;
	}
	p->val = nxtval;
	if (flag) {
		li = (nxtval - val) / ensmps;
		if (p->has_audio_arguments()) {
			do {
				*rs++ = *sg++ * val;
				val += li;
			}
			while (--nsmps);
		} else {
			do {
				*rs++ = *sg * val;
				val += li;
			}
			while (--nsmps);
		}
	} else {
		if (p->has_audio_arguments()) {
			do
				*rs++ = *sg++;
			while (--nsmps);
		} else {
			do
				*rs++ = *sg;
			while (--nsmps);
		}
	}
}

void
evxset(ENVLPX * p)
{
	RCPointer<FunctionTable> ftp;
	Number ixmod, iatss, idur, prod, diff, asym, nk, denom, irise;
	int32 cnt1;

	if ((ftp = ftfind(p->ifn)) == NULL) {
	        p->error ("evxset: no such table %.2f", p->ifn);
		return;
	}
	p->ftp = ftp;
	if ((idur = *p->idur) > FZERO) {
		if ((iatss = (Number) fabs(*p->iatss)) == FZERO) {
			p->error ("iatss = 0");
			return;
		}
		if (iatss != FONE && (ixmod = *p->ixmod) != FZERO) {
			if (fabs(ixmod) > .95) {
				p->error ("ixmod out of range.");
				return;
			}
			ixmod = -(Number) sin(sin(ixmod));
			prod = ixmod * iatss;
			diff = ixmod - iatss;
			denom = diff + prod + 1.0f;
			if (denom == FZERO)
				asym = FHUND;
			else {
				asym = 2 * prod / denom;
				if (fabs(asym) > FHUND)
					asym = FHUND;
			}
			iatss = (iatss - asym) / (1.0f - asym);
			asym = asym * *(ftp->ftable + ftp->flen);	/* +1
									 */
		} else
			asym = FZERO;
		if ((irise = *p->irise) > FZERO) {
			p->phs = 0;
			p->ki = (int32) (kicvt / irise);
			p->val = *ftp->ftable;
		} else {
			p->phs = -1;
			p->val = *(ftp->ftable + ftp->flen) - asym;
			irise = FZERO;	/* in case irise < 0 */
		}
		if (!(*(ftp->ftable + ftp->flen)))
			p->error ("rise func ends with zero");
		cnt1 = (int32) ((idur - irise - *p->idec) * ekr + 0.5f);
		if (cnt1 < 0L) {
			cnt1 = 0L;
			nk = ekr;
		} else {
			if (*p->iatss < FZERO || cnt1 <= 4L)
				nk = ekr;
			else
				nk = (Number) cnt1;
		}
		p->mlt1 = (Number) pow((double) iatss, (FONE / nk));
		if (*p->idec > FZERO) {
			if (*p->iatdec <= FZERO)
				p->error ("non-positive iatdec.");
			else
				p->mlt2 = (Number) pow((double) *p->iatdec,
					   ((double) onedkr / *p->idec));
		}
		p->cnt1 = cnt1;
		p->asym = asym;
	}
}

void
knvlpx(ENVLPX * p)
{
	RCPointer<FunctionTable> ftp;
	int32 phs;
	Number fact, v1, fract, *ftab;

	ftp = p->ftp;

	if ((phs = p->phs) >= 0) {
		fract = (Number) PFRAC(phs);
		ftab = ftp->ftable + (phs >> ftp->lobits);
		v1 = *ftab++;
		fact = (v1 + (*ftab - v1) * fract);
		phs += p->ki;
		if (phs >= MAXLEN) {	/* check that 2**N+1th pnt is good
					 */
			p->val = *(ftp->ftable + ftp->flen);
			if (!p->val)
				p->error ("envlpx rise func ends with zero");
			p->val -= p->asym;
			phs = -1L;
		}
		p->phs = phs;
	} else {
		fact = p->val;
		if (p->cnt1 > 0L) {
			p->val *= p->mlt1;
			fact += p->asym;
			p->cnt1--;
			if (p->cnt1 == 0L)
				p->val += p->asym;
		} else
			p->val *= p->mlt2;
	}
	*p->rslt = *p->xamp * fact;
}

void
envlpx(ENVLPX * p)
{
	RCPointer<FunctionTable> ftp;
	int32 phs;
	int nsmps = ksmps;
	Number *xamp, *rslt, val, nxtval, li, v1, fract, *ftab;

	xamp = p->xamp;
	rslt = p->rslt;
	val = p->val;
	if ((phs = p->phs) >= 0L) {
		ftp = p->ftp;
		fract = (Number) PFRAC(phs);
		ftab = ftp->ftable + (phs >> ftp->lobits);
		v1 = *ftab++;
		nxtval = (v1 + (*ftab - v1) * fract);
		phs += p->ki;
		if (phs >= MAXLEN) {	/* check that 2**N+1th pnt is good
					 */
			nxtval = *(ftp->ftable + ftp->flen);
			if (!nxtval)
				p->error ("envlpx rise func ends with zero");
			nxtval -= p->asym;
			phs = -1;
		}
		p->phs = phs;
	} else {
		nxtval = val;
		if (p->cnt1 > 0L) {
			nxtval *= p->mlt1;
			nxtval += p->asym;
			p->cnt1--;
		} else
			nxtval *= p->mlt2;
	}
	p->val = nxtval;
	li = (nxtval - val) / ensmps;	/* linear interpolation factor */
	if (p->has_audio_arguments()) {	/* for audio rate amplitude: */
		do {
			*rslt++ = *xamp++ * val;
			val += li;
		}
		while (--nsmps);
	} else {
		do {
			*rslt++ = *xamp * val;
			val += li;
		}
		while (--nsmps);
	}
}

void
xsgset2(EXPSEG2 * p)
{				/*gab-A1 (G.Maldonado) */
	register XSEG *segp;
	register int nsegs;
	register Number d, **argp, val, dur, nxtval;
	int n;

	nsegs = p->INOCOUNT >> 1;	/* count segs & alloc if nec */
	if ((segp = (XSEG *) p->auxch.auxp) == NULL) {
		p->auxch.alloc ((size_t) nsegs * sizeof (XSEG));
		p->cursegp = segp = (XSEG *) p->auxch.auxp;
		(segp + nsegs - 1)->cnt = MAXPOS;	/* set endcount for
							   safety */
	}
	argp = p->argums;
	nxtval = **argp++;
	if (**argp <= 0.0f) {
		return;		/* if idur1 <= 0, skip init  */
	}
	p->cursegp = segp;	/* else proceed from 1st seg */
	segp--;
	do {
		segp++;		/* init each seg ..  */
		val = nxtval;
		dur = **argp++;
		nxtval = **argp++;
		if (dur > 0.0f) {
			if (val * nxtval <= 0.0f)
				goto experr;
			d = dur * esr;	/* ekr;  (gab) */
			segp->val = val;
			segp->mlt = (Number) pow((nxtval / val), (1. / d));
			segp->cnt = (int32) (d + .5);
		} else
			break;	/*  .. til 0 dur or done */
	} while (--nsegs);
	segp->cnt = MAXPOS;	/* set last cntr to infin */
	return;

      experr:n = segp - p->cursegp + 1;
	if (val == 0.0f) {
		p->error ("ival%d is zero", n);
	} else {
		p->error ("ival%d sign conflict", n + 1);
	}
}



void
expseg2(EXPSEG2 * p)
{				/*gab-A1 (G.Maldonado) */
	register XSEG *segp;
	register int nsmps = ksmps;
	register Number val, *rs;
	segp = p->cursegp;
	val = segp->val;
	rs = p->rslt;
	do {
		while (--segp->cnt < 0) {
			p->cursegp = ++segp;
			val = segp->val;
		}
		*rs++ = val;
		val *= segp->mlt;
	} while (--nsmps);
	segp->val = val;
}

void
evrset(ENVLPR * p)
{
	RCPointer<FunctionTable> ftp;
	Number ixmod, iatss, prod, diff, asym, denom, irise;

	if ((ftp = ftfind(p->ifn)) == NULL) {
		p->error ("evrset: no such table %.2f", p->ifn);
		return;
	}
	p->ftp = ftp;
	if ((iatss = (Number) fabs((double) *p->iatss)) == 0.0f) {
		p->error ("iatss = 0");
		return;
	}
	if (iatss != FONE && (ixmod = *p->ixmod) != 0.0f) {
		if (fabs(ixmod) > .95) {
			p->error ("ixmod out of range.");
			return;
		}
		ixmod = -(Number) sin(sin((double) ixmod));
		prod = ixmod * iatss;
		diff = ixmod - iatss;
		denom = diff + prod + 1.0f;
		if (denom == 0.0f)
			asym = FHUND;
		else {
			asym = 2 * prod / denom;
			if (fabs(asym) > FHUND)
				asym = FHUND;
		}
		iatss = (iatss - asym) / (1.0f - asym);
		asym = asym * *(ftp->ftable + ftp->flen);	/* +1 */
	} else
		asym = 0.0f;
	if ((irise = *p->irise) > 0.0f) {
		p->phs = 0L;
		p->ki = (int32) (kicvt / irise);
		p->val = *ftp->ftable;
	} else {
		p->phs = -1L;
		p->val = *(ftp->ftable + ftp->flen) - asym;
		irise = 0.0f;	/* in case irise < 0 */
	}
	if (!(*(ftp->ftable + ftp->flen)))
		p->error ("rise func ends with zero");
	p->mlt1 = (Number) pow((double) iatss, (double) onedkr);
	if (*p->idec > 0.0f) {
		int32 rlscnt = (int32) (*p->idec * ekr + .5);
		if ((p->rindep = (int32) *p->irind))
			p->rlscnt = rlscnt;
		else if (rlscnt > (int32) p->PROCESS->extra_time)
			p->PROCESS->extra_time = (qm_time_t) rlscnt;
		if ((p->atdec = *p->iatdec) <= 0.0f) {
			p->error ("non-positive iatdec");
		}
	}
	p->asym = asym;
	p->rlsing = 0;

}

void
knvlpxr(ENVLPR * p)
{
	Number fact;
	int32 rlscnt;

	if (!p->rlsing) {	/* if not in reles seg  */
		if (p->PROCESS->releasing()) {
			p->rlsing = 1;	/*   if new flag, set mlt2 */
			rlscnt = (p->rindep) ? p->rlscnt :
			    p->PROCESS->extra_time;
			if (rlscnt)
				p->mlt2 = (Number) pow((double) p->atdec, 1.0 /
						      (double) rlscnt);
			else
				p->mlt2 = 1.0f;
		}
		if (p->phs >= 0) {	/* do fn rise for seg 1 */
			RCPointer<FunctionTable> ftp = p->ftp;
			int32 phs = p->phs;
			Number fract = PFRAC(phs);
			Number *ftab = ftp->ftable + (phs >> ftp->lobits);
			Number v1 = *ftab++;
			fact = (v1 + (*ftab - v1) * fract);
			phs += p->ki;
			if (phs < MAXLEN || p->rlsing)	/* if more fn or
							   beg rls */
				p->val = fact;	/*      save cur val    
						 */
			else {	/* else prep for seg 2  */
				p->val = *(ftp->ftable + ftp->flen) - p->asym;
				phs = -1L;
			}
			p->phs = phs;
		} else {
			fact = p->val + p->asym;	/* do seg 2 with asym
							 */
			p->val *= p->mlt1;
			if (p->rlsing)	/* if ending, rm asym */
				p->val += p->asym;
		}
	} else
		fact = p->val *= p->mlt2;	/* else do seg 3 decay */
	*p->rslt = *p->xamp * fact;
}

void
envlpxr(ENVLPR * p)
{
	int nsmps = ksmps;
	int32 rlscnt;
	Number *xamp, *rslt, val, nxtval, li;

	xamp = p->xamp;
	rslt = p->rslt;
	val = p->val;
	if (!p->rlsing) {	/* if not in reles seg  */
		if (p->PROCESS->releasing()) {
			p->rlsing = 1;	/*   if new flag, set mlt2 */
			rlscnt = (p->rindep) ? p->rlscnt : p->PROCESS->extra_time;
			if (rlscnt) {
				p->mlt2 = (Number) pow((double) p->atdec, 1.0 /
						      (double) rlscnt);
			} else {
				p->mlt2 = 1.0f;
			}
		}
		if (p->phs >= 0) {	/* do fn rise for seg 1 */
			RCPointer<FunctionTable> ftp = p->ftp;
			int32 phs = p->phs;
			Number fract = PFRAC(phs);
			Number *ftab = ftp->ftable + (phs >> ftp->lobits);
			Number v1 = *ftab++;
			ftp = p->ftp;
			fract = PFRAC(phs);
			ftab = ftp->ftable + (phs >> ftp->lobits);
			v1 = *ftab++;
			nxtval = (v1 + (*ftab - v1) * fract);
			phs += p->ki;
			if (phs < MAXLEN || p->rlsing)	/* if more fn or
							   beg rls */
				p->val = nxtval;	/*      save 2nd brkpnt 
							 */
			else {	/* else prep for seg 2  */
				p->val = *(ftp->ftable + ftp->flen) - p->asym;
				phs = -1;
			}
			p->phs = phs;
		} else {
			nxtval = p->val *= p->mlt1;	/* do seg 2 with
							   asym   */
			val += p->asym;
			nxtval += p->asym;
			if (p->rlsing)	/* if ending, rm asym   */
				p->val += p->asym;
		}
	} else
		p->val = nxtval = val * p->mlt2;	/* else do seg 3 decay  
							 */
	li = (nxtval - val) * dvensmps;		/* all segs use interp 
						 */
	if (p->has_audio_arguments()) {
		do {
			*rslt++ = *xamp++ * val;
			val += li;
		} while (--nsmps);
	} else {
		do {
			*rslt++ = *xamp * val;
			val += li;
		} while (--nsmps);
	}
}

Opcode opcodes[] =
{
	UGENS1_OPCODE_LIST,
	{NULL}
};

