/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holder's names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to 
    education and research.

    $Id: streson.cc,v 1.1 1999/11/01 04:29:07 pbd Exp $
*/

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>

#include "streson.h"

/*******************************************************/
/* streson.c : string resonator opcode                 */
/*             takes one input and passes it through   */
/*             emulates the resonance                  */
/*             of a string tuned to a kfun fundamental */
/*          Victor Lazzarini, 1998                     */
/*******************************************************/

void
stresonset(STRES *p)
{
	int32 n;
	p->size = (int32) (esr/20);   /* size of delay line */
	p->aux.alloc (p->size*sizeof(Number));
	p->Cdelay = (Number*) p->aux.auxp; /* delay line */
	p->LPdelay = p->APdelay = 0.0f; /* reset the All-pass and Low-pass delays */
	p->wpointer = p->rpointer = 0; /* reset the read/write pointers */
	for (n = 0; n < p->size; n++)
	    p->Cdelay[n] = 0.0f;
}

void
streson(STRES *p)
{
	Number *out = p->result;
	Number *in = p->ainput;
	Number g = *p->ifdbgain;
	Number freq, a, s, w, sample, tdelay, fracdelay;
	int32 delay, nn = ksmps;
	int32	rp = p->rpointer, wp = p->wpointer;
	int32 size = p->size;
	Number	APdelay = p->APdelay;
	Number	LPdelay = p->LPdelay;
	int32		vdt = p->vdtime;

	freq = *p->afr;
	if (freq < 20) freq = 20.0f;   /* lowest freq is 20 Hz */
	tdelay = esr/freq;
	delay = (int32) (tdelay - 0.5f); /* comb delay */
	fracdelay = tdelay - (delay + 0.5f); /* fractional delay */
	p->vdtime = size - delay;       /* set the var delay */
	a = (1-fracdelay)/(1+fracdelay);   /* set the all-pass gain */
	do {
		/* GetSample(p); */
		Number tmpo;
		tmpo = p->Cdelay[rp];
		rp = (vdt + wp)%size;
		w = *in++ + tmpo;
		s = LPdelay*0.5f + w*0.5f;
		LPdelay = w;
		*out++ = sample = APdelay + s*(-a);
		APdelay = s + (sample*a);
		/* PutSample(sample*g, p); */
		p->Cdelay[wp] = sample*g;
		wp++;
		if (wp == size) wp=0;
	} while (--nn);
	p->rpointer = rp; p->wpointer = wp;
	p->LPdelay = LPdelay; p->APdelay = APdelay;
}

Opcode opcodes[] = {
	STRESON_OPCODE_LIST,
	{ NULL }
};

