/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holder's names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to 
    education and research.

    $Id: random.cc,v 1.1 1999/11/01 04:29:07 pbd Exp $
*/

#include <math.h>
#include <quasimodo/qm.h>

#include "random.h"


/*=========================================================================
 *
 * seedrand()  and the PRNG seed.
 *
 * Driven by opcode "seed".  Takes a float input parm and seeds the 
 * 31 bit state of the PRNG as follows:
 *
 * Input  = 0  Use time of day to generate seed.
 *
 * Input != 0  Make it positive, multiply by 1,000, mod it by
 *             (2^31 - 1) turn it into an integer using floor,
 *             so it can never be 2^31, make it 1 if the result
 *             was 0, and use this as the seed.
 *             The PRNG will put out 0s if its state is 0!
 *
 * The seed is a 32 bit integer. Initialise it with an 
 * auspicious number - pi to the 17th power.
 *
 * The seed opcode uses the same data structure PRAND as the generators
 * - its sole parameter, an input, is in the first position, which is
 * called "out".
 */ 

#define rim RANDINT_MAX
#define ria RANDINT_MULT
#define riq (RANDINT_MAX/RANDINT_MULT)
#define rir (RANDINT_MAX%RANDINT_MULT)

/* Add one in case y2k voodoo causes time()
 * to return 0. XXX say what ??? -- pbd
 */

long seed31 =  282844563L; 

/*=========================================================================
 *
 * randint31() 
 *
 * 31 bit Park Miller PRNG using Linus Schrage's method for doing it all
 * with 32 bit variables.
 *
 * Code adapted from Ray Garder's public domain code of July 1997 at:
 * http://www.snippets.org/RG_RAND.C     Thanks!
 * 
 *  Based on "Random Number Generators: Good Ones Are Hard to Find",
 *  S.K. Park and K.W. Miller, Communications of the ACM 31:10 (Oct 1988),
 *  and "Two Fast Implementations of the 'Minimal Standard' Random
 *  Number Generator", David G. Carta, Comm. ACM 33, 1 (Jan 1990), p. 87-88
 *
 *  Linear congruential generator: f(z) = (16807 * z) mod (2 ** 31 - 1)
 *
 *  Uses L. Schrage's method to avoid overflow problems.
 */

long randint31()
{
      unsigned long rilo, rihi;

      rilo = ria * (long)(seed31 & 0xFFFF);
      rihi = ria * (long)((unsigned long)seed31 >> 16);
      rilo += (rihi & 0x7FFF) << 16;
      if (rilo > rim)
      {
            rilo &= rim;
            ++rilo;
      }
      rilo += rihi >> 15;
      if (rilo > rim)
      {
            rilo &= rim;
            ++rilo;
      }
      return ( seed31 = (long)rilo );
}

float unifrand(float range)
{
    return range*unirand();
}

float linrand(float range)	/*      linear distribution routine     */
{
    float r1, r2;

    r1 = unirand();
    r2 = unirand();

    if (r1 > r2)
	r1 = r2;

    return (r1 * range);
}

float trirand(float range)	/*      triangle distribution routine   */
{
    float r1, r2;

    r1 = unirand();
    r2 = unirand();

    return (((r1 + r2) - 1.0f) * range);
}
		
float exprand(float l)		/*      exponential distribution routine */
{
    float r1;

    if (l < 0.0f) return (0.0f);

    do {
	r1 = unirand();
    } while (r1 == 0.0f);

    return (-(float)log(r1 *l));
}

float biexprand(float l)	/* bilateral exponential distribution routine */
{
    float r1;

    if (l < 0.0f) return (0.0f);

    do {
	r1 = 2.0f * unirand();
    } while (r1 == 0.0f || r1 == 2.0f); 

    if (r1 > 1.0f)     {
	r1 = 2.0f - r1;
	return (-(float)log(r1 * l));
    }
    return ((float)log(r1 * l));
}

float gaussrand(float s)	/*      gaussian distribution routine   */
{
    float r1 = 0.0f;
    int n = 12;
    s /= 3.83f;

    do {
	r1 += unirand();
    } while (--n);

    return (s * (r1 - 6.0f));
}

float cauchrand(float a)	/*      cauchy distribution routine     */
{
    float r1;
    a /= 318.3f;

    do {
      do {
	r1 = unirand();
      } while (r1 == 0.5f);

      r1 = a * (float)tan(pi*(double)r1);
    } while (r1>1.0f || r1<-1.0f); /* Limit range artificially */
    return r1;
}

float pcauchrand(float a)	/*      positive cauchy distribution routine */
{
    float r1;
    a /= 318.3f;

    do {
      do {
	r1 = unirand();
      } while (r1 == 1);

      r1 = a * (float)tan( pi * 0.5 * r1);
    } while (r1>1.0f);
    return r1;
}

float betarand(float range, float a, float b) /* beta distribution routine  */
{
    float r1, r2;

    if (a < 0.0f || b < 0.0f ) return (0.0f); 

    do {
	do {
	    r1 = unirand();
	} while (r1 == 0.0f);
	
	do {
	    r2 = unirand();
	} while (r2 == 0.0f);
	
	r1 = (float)pow(r1, 1.0 / (double)a);
	r2 = (float)pow(r2, 1.0 / (double)b);
    } while ((r1 + r2) > 1.0f);

    return ((r1 / (r1 +r2)) * range);
}

float weibrand(float s, float t) /*      weibull distribution routine    */
{
    float r1, r2;

    if (t < 0.0f ) return (0.0f);

    do {
	r1 = unirand();
    } while (r1 == 0.0f || r1 == 1.0f);

    r2 = 1.0f /  (1.0f - r1);

    return (s * (float)pow (log((double)r2),  (1.0 /(double)t)));
}

float poissrand(float l)	/*      Poisson distribution routine    */
{
    float r1, r2, r3;

    if (l < 0.0f ) return (0.0f);

    r1 = unirand();
    r2 = (float)exp(-l);
    r3 = 0.0f;

    while (r1 >= r2) {
	r3++;
	r1 *= unirand();
    }

    return (r3);
}

/* Opcode entries in mathops.cc */

