/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: arith_ops.cc,v 1.4 1999/11/16 05:31:40 pbd Exp $
*/


#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/function_tables.h>
#include <quasimodo/defaults.h>
#include <quasimodo/audio.h>
#include <quasimodo/opcode.h>
#include <quasimodo/opcode_enums.h>
#include <quasimodo/process.h>
#include <quasimodo/module.h>

#include "arith_ops.h"

static Dumber octave;

void
initScalar (ASSIGN *p)

{
	Number *r;
	size_t v;

	r = p->r;

	/* DO NOT initialize patched data locations if there is
	   already a voice playing.
	*/

	if (p->MODULE->is_patched (r) &&
	    (v = p->MODULE->voices()) > 1) {
		return;
	}

	*r = *p->a;
}

void
initVector (ASSIGN *p)

{
	Number *r;
	Number a;
	int nsmps = ksmps;
	int v;

	/* DO NOT initialize patched data locations if there is
	   already a voice playing.
	*/

	r = p->r;

	if (p->MODULE->is_patched (r) &&
	    (v = p->MODULE->voices()) > 1) {
		return;
	}

	a = *p->a;

	do {
		*r++ = a;
	} while (--nsmps);
}

void
int1(EVAL * p)
{				/* returns signed whole no. */
	double intpart;
	modf((double) *p->a, &intpart);
	*p->r = (float) intpart;
}

void
frac1(EVAL * p)
{				/* returns positive frac part */
	double intpart, fracpart;
	fracpart = modf((double) *p->a, &intpart);
	*p->r = (float) fracpart;
}

static double rndfrac = .5, rndmlt = 105.947;

void
rnd1(EVAL * p)
{				/* returns unipolar rand(x) */
	double intpart;
	rndfrac = modf(rndfrac * rndmlt, &intpart);
	*p->r = *p->a * (float) rndfrac;
}

void
birnd1(EVAL * p)
{				/* returns bipolar rand(x) */
	double intpart;
	rndfrac = modf(rndfrac * rndmlt, &intpart);
	*p->r = *p->a * (2.0f * (float) rndfrac - 1.0f);
}

#define LIB1(OPNAME,LIBNAME)  \
              void OPNAME(EVAL *p) { \
		*p->r = (float)LIBNAME((double)*p->a); \
              }

LIB1(abs1, fabs)
LIB1(exp01, exp)
LIB1(log01, log)
LIB1(sqrt1, sqrt)
LIB1(sin1, sin)
LIB1(cos1, cos)
LIB1(tan1, tan)
LIB1(asin1, asin)
LIB1(acos1, acos)
LIB1(atan1, atan)
LIB1(sinh1, sinh)
LIB1(cosh1, cosh)
LIB1(tanh1, tanh)
LIB1(log101, log10)
void
atan21(AOP * p)
{
	*p->r = (float) atan2((double) *p->a, (double) *p->b);
}

#define LIBA(OPNAME,LIBNAME)                      \
       void OPNAME(EVAL *p) {                     \
         int        nsmps = ksmps;                \
         float *r, *a;                            \
                                                  \
         r = p->r;                                \
         a = p->a;                                \
         do {                                     \
	   *r++ = (float) LIBNAME((double) *a++); \
         } while (--nsmps);                       \
       }

LIBA(absa, fabs)
LIBA(expa, exp)
LIBA(loga, log)
LIBA(sqrta, sqrt)
LIBA(sina, sin)
LIBA(cosa, cos)
LIB1(tana, tan)
LIB1(asina, asin)
LIB1(acosa, acos)
LIB1(atana, atan)
LIB1(sinha, sinh)
LIB1(cosha, cosh)
LIB1(tanha, tanh)
LIB1(log10a, log10)
void
atan2aa(AOP * p)
{
	int nsmps = ksmps;
	float *r, *a, *b;
	r = p->r;
	a = p->a;
	b = p->b;
	do
		*r++ = (float) atan2((double) *a++, (double) *b++);
	while (--nsmps);
}

void
dbamp(EVAL * p)
{
	*p->r = (float) (log(fabs((double) *p->a)) / log10d20);
}

void
ampdb(EVAL * p)
{
	*p->r = (float) exp((double) *p->a * log10d20);
}

void
aampdb(EVAL * p)
{
	int nsmps = ksmps;
	float *r, *a;
	r = p->r;
	a = p->a;
	do
		*r++ = (float) exp((double) *a++ * log10d20);
	while (--nsmps);
}

void
ftlen(EVAL * p)
{
	RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind(p->a)) != 0)
		*p->r = (float) ftp->flen;
	else
		*p->r = -1.0f;	/* Return something */
}

void
ftlptim(EVAL * p)
{
	RCPointer<FunctionTable> ftp;
	if ((ftp = ftfind(p->a, true, true)) == NULL)
		return;
	if (ftp->loopmode1)
		*p->r = ftp->begin1 * onedsr;
	else {
		*p->r = 0.0f;
		warning << "ftlptim: non-looping sample" << endmsg;

	}
}

/***** nsamp by G.Maldonado ****//* gab-A1
*/

void
numsamp(EVAL * p)
{
	register RCPointer<FunctionTable> ftp;
	if ((ftp = ftfind(p->a)) != 0)
		*p->r = (float) ftp->soundend;
}

/**** ftsr by G.Maldonado ****//* gab-A1 */

void
ftsr(EVAL * p)
{
	register RCPointer<FunctionTable> ftp;
	if ((ftp = ftfind(p->a)) != 0)
		*p->r = ftp->frame_rate;
}

/**** ftlen2 by G.Maldonado ****/

RCPointer<FunctionTable> ftfind2(float *);		/*declaration */
/* SEE FGENS.C for ftfind2() function */
void
ftlen2(EVAL * p)
{
	register RCPointer<FunctionTable> ftp;

	if ((ftp = ftfind (p->a, true, false)) != 0)
		*p->r = (float) ftp->flen;
}

/********************************/

void
octpch(EVAL * p)
{
	double fract;
	fract = modf((double) *p->a, &octave);
	fract *= eipt3;
	*p->r = (float) (octave + fract);
}

void
pchoct(EVAL * p)
{
	double fract;
	fract = modf((double) *p->a, &octave);
	fract *= 0.12;
	*p->r = (float) (octave + fract);
}


void
cpsoct(EVAL * p)
{
	long loct = (long) (*p->a * octresol);
	*p->r = (float) CPSOCTL(loct);
}

/* gab-A1 */
void
powoftwo(EVAL * p)
{				/* by G.Maldonado */
	*p->r = powerof2[(int) (*p->a * (STEPS / (OCTAVES * 2)) + .5)];
}

void
logbasetwo(EVAL * p)
{				/* by G.Maldonado */
	*p->r = logbase2[(int) ((*p->a - (1 / INTERVAL)) / (INTERVAL - 1 /
						INTERVAL) * STEPS + .5)];
}

/* end gab-A1 */

void
acpsoct(EVAL * p)
{
	float *r, *a;
	long loct, nsmps = ksmps;
	a = p->a;
	r = p->r;
	do {
		loct = (long) (*a++ * octresol);
		*r++ = CPSOCTL(loct);
	} while (--nsmps);
}

void
octcps(EVAL * p)
{
	*p->r = (float) (log((double) *p->a / onept) / logtwo);
}

void
cpspch(EVAL * p)
{
	double fract;
	long loct;

	fract = modf((double) *p->a, &octave);
	fract *= eipt3;
	loct = (long) ((octave + fract) * octresol);
	*p->r = (float) CPSOCTL(loct);
/*      error << "" << *p->a << " -> " << *p->r << "\n" << endmsg;
 */
}

void
cpsxpch(XENH * p)
{				/* This may be too expensive */
	double fract;
	double loct;

	info << "pc,et,cy,ref=" << *p->pc << "," << *p->et << "," << *p->cy << "," << *p->ref << "\n" << endmsg;


	fract = modf((double) *p->pc, &loct);	/* Get octave */
	if (*p->et > 0) {
		fract = pow((double) *p->cy, loct + (100.0 * fract) / ((double)
								*p->et));
		*p->r = (float) fract **p->ref;
	} else {		/* Values in a table */
		float t = -*p->et;
		RCPointer<FunctionTable> ftp = ftfind(&t);
		long len;
		if (ftp == NULL) {
			info << "No tuning table " << (int) (-*p->et) << "\n" << endmsg;

			exit(1);
		}
		len = ftp->flen;
		while (fract > len) {
			fract -= len;
			loct++;
		}
		fract += 0.005;
		*p->r = *p->ref * *(ftp->ftable + (int) (100.0 * fract)) *
		    (float) pow((double) *p->cy, loct);
	}
	error << "....-> " << *p->r << "\n" << endmsg;

}

void
cps2pch(XENH * p)
{
	double fract;
	double loct;

	fract = modf((double) *p->pc, &loct);	/* Get octave */
	if (*p->et > 0) {
		fract = pow(2.0, loct + (100.0 * fract) / ((double) *p->et));
		*p->r = (float) (fract * 1.02197503906);	/* Refer to
								   base frequency */
	} else {
		float t = -*p->et;
		RCPointer<FunctionTable> ftp = ftfind(&t);
		long len;
		if (ftp == NULL) {
			info << "No tuning table " << (int) (-*p->et) << "\n" << endmsg;

			exit(1);
		}
		len = ftp->flen;
		while (fract > len) {
			fract -= len;
			loct++;
		}
		fract += 0.005;
		*p->r = (float) (1.02197503906 * *(ftp->ftable + (int) (100.0 *
								fract)) *
				 pow(2.0, loct));
	}

/*       double ref = 261.62561 / pow(2.0, 8.0); */
}

Opcode opcodes[] =
{
	ARITHOPS_OPCODE_LIST,
	{ NULL }
};

