/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: vibraphn.cc,v 1.2 1999/12/16 03:53:35 pbd Exp $
*/

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/function_tables.h>
#include <quasimodo/process.h>

#include "physutil.h"
#include "vibraphn.h"

/*******************************************/
/*  Vibraphone SubClass of Modal4          */
/*  Module, by Perry R. Cook, 1995-96  */
/*                                         */
/*   Controls:    CONTROL1 = stickHardness */
/*                CONTROL2 = strikePosition */
/*                CONTROL3 = vibFreq       */
/*                MOD_WHEEL= vibAmt        */
/*******************************************/

void
vibraphnset(VIBRAPHN * p)
{
	Modal4 *m = &(p->m4);
	float temp;
	RCPointer<FunctionTable>	ftp;

	if ((ftp = ftfind(p->ifn)) != 0)
		p->m4.wave = ftp;
	else
	    p->error ("No table for Vibraphone strike");

	/* Expect an impulslything */

	make_Modal4(m, p->ivfn, *p->vibAmt, *p->vibFreq);

	p->m4.w_phaseOffset = 0.0f;

	OnePole_setPole(&p->m4.onepole, 0.2f);
	Modal4_setRatioAndReson(m, 0, 1.0f, 0.99995f);	/*  Set        
							 */
	Modal4_setRatioAndReson(m, 1, 2.01f, 0.99991f);		/*  our  
								 */
	Modal4_setRatioAndReson(m, 2, 3.9f, 0.99992f);	/*  resonance  
							 */
	Modal4_setRatioAndReson(m, 3, 14.37f, 0.99990f);	/*  values here 
								 */
	Modal4_setFiltGain(m, 0, 0.025f);
	Modal4_setFiltGain(m, 1, 0.015f);
	Modal4_setFiltGain(m, 2, 0.015f);
	Modal4_setFiltGain(m, 3, 0.015f);
	p->m4.directGain = 0.0f;

	p->m4.w_rate = 2.0f + (22.66f * *p->hardness);
	p->m4.masterGain = 0.2f + (*p->hardness * 1.6f);

	/* Set Strike position */
	temp = p->strikePosition * pi;
	BiQuad_setGain(p->m4.filters[0], 0.025f * (float) sin((double) temp));
	BiQuad_setGain(p->m4.filters[1], 0.015f * (float) sin(0.1 + (2.01 *
							(double) temp)));
	BiQuad_setGain(p->m4.filters[2], 0.015f * (float) sin(3.95 * (double)
							      temp));
	/* Strike */
	Modal4_strike(m, *p->amplitude * AMP_RSCALE);
	Modal4_setFreq(m, *p->frequency);

	{
		uint32 relestim = (uint32) (esr * *p->dettack);

		if (relestim > p->PROCESS->extra_time)
			p->PROCESS->extra_time = relestim;
	}

	/* XXX this is wrong */

	p->kloop = (int32) ((p->PROCESS->time_off() * ekr) - 
			    (int32) (ekr * *p->dettack));

}

void
vibraphn(VIBRAPHN * p)
{
	Modal4 *m = &(p->m4);
	float *ar = p->ar;
	long nsmps = ksmps;
	float amp = (*p->amplitude) * AMP_RSCALE;	/* Normalise */

	if (p->kloop >= 0 && p->PROCESS->releasing())
		p->kloop = 1;
	if (p->kloop && ((--p->kloop) == 0)) {
		Modal4_damp(m, 1.0f - (amp * 0.03f));
	}
	Modal4_setFreq(m, *p->frequency);
	p->m4.v_rate = *p->vibFreq;
	p->m4.vibrGain = *p->vibAmt;
	do {
		float lastOutput;
		lastOutput = Modal4_tick(m);
		*ar++ = lastOutput * 8.0f * AMP_SCALE;	/* Times 8 as
							   seems too quiet */
	} while (--nsmps);
}

/*******************************************/
/*  AgogoBell SubClass of Modal4 Module */
/*  by Perry R. Cook, 1995-96              */
/*                                         */
/*   Controls:    CONTROL1 = stickHardness */
/*                CONTROL2 = strikePosition */
/*                CONTROL3 = vibFreq       */
/*                MOD_WHEEL= vibAmt        */
/*******************************************/


/*   Modes measured from my Agogo Bell by FFT:  */
/*   360, 1470, 2401, 4600                      */


void
agogobelset(VIBRAPHN * p)
{
	Modal4 *m = &(p->m4);
	RCPointer<FunctionTable>	ftp;

	if ((ftp = ftfind(p->ifn)) != 0)
		p->m4.wave = ftp;
	else
	    p->error ("No table for Agogobell strike");

	/* Expect an impulslything */

	make_Modal4(m, p->ivfn, *p->vibAmt, *p->vibFreq);

	p->m4.w_phaseOffset = 0.0f;

	OnePole_setPole(&p->m4.onepole, 0.2f);
	Modal4_setRatioAndReson(m, 0, 1.00f, 0.999f);	/*  Set         */
	Modal4_setRatioAndReson(m, 1, 4.08f, 0.999f);	/*  our         */
	Modal4_setRatioAndReson(m, 2, 6.669f, 0.999f);	/*  resonance  
							 */
	Modal4_setRatioAndReson(m, 3, -3725.0f, 0.999f);	/*  values here 
								 */
	Modal4_setFiltGain(m, 0, 0.07f);
	Modal4_setFiltGain(m, 1, 0.06f);
	Modal4_setFiltGain(m, 2, 0.04f);
	Modal4_setFiltGain(m, 3, 0.02f);
	p->m4.directGain = 0.3f;

	p->m4.w_rate = 3.0f + (8.0f * *p->hardness);
	p->m4.masterGain = 1.0f;

	/* Set Strike position */
	/* Strike */
	Modal4_strike(m, *p->amplitude * AMP_RSCALE);
	Modal4_setFreq(m, *p->frequency);
}

void
agogobel(VIBRAPHN * p)
{
	Modal4 *m = &(p->m4);
	float *ar = p->ar;
	long nsmps = ksmps;

	p->m4.v_rate = *p->vibFreq;
	p->m4.vibrGain = *p->vibAmt;
	do {
		float lastOutput;
		lastOutput = Modal4_tick(m);
		*ar++ = lastOutput * AMP_SCALE;
	} while (--nsmps);
}

Opcode opcodes[] = {
	VIBRAPHN_OPCODE_LIST,
	{ NULL }
};
