/* 
    Copyright 1999, by Sean M. Costello

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: svfilter.cc,v 1.1 1999/11/01 04:25:03 pbd Exp $

    svfilter is an implementation of Hal Chamberlin's state variable filter 
    algorithm, from "Musical Applications of Microprocessors" (Hayden Books,
    Indianapolis, Indiana, 1985), 2nd. edition, pp. 489-492. It implements
    a second-order resonant filter, with lowpass, highpass and bandpass
    outputs.
 */

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>

#include "svfilter.h"


void svfset(SVF *p)
{
	/* set initial delay states to 0 */
	p->ynm1 = p->ynm2 = 0.0f;
}       

void svf(SVF *p)
{
	Number f1, q1, scale;
	Number *low, *high, *band, *in, *ynm1, *ynm2;
	Number low2, high2, band2;
	Number kfco = *p->kfco, kq = *p->kq;
	int32 nsmps = ksmps;
        
	/* calculate frequency and Q coefficients */
	f1 = 2.0f * (Number)sin(PI * (double)(kfco / esr));
	q1 = 1.0f / kq;

	in = p->in;
	low = p->low;
	band = p->band;
	high = p->high;
	ynm1 = &(p->ynm1);
	ynm2 = &(p->ynm2);
    
	/* if there is a non-zero value for iscl, set scale to be
	 * equal to the Q coefficient.
	 */
	if (*p->iscl) 
		scale = q1;
	else 
		scale = 1.0f;
        
	/* equations derived from Hal Chamberlin, "Musical Applications
	 * of Microprocessors.
	 */
	do{
		*low++ = low2 = *ynm2 + f1 * *ynm1;
		*high++ = high2 = scale * *in++ - low2 - q1 * *ynm1;
		*band++ = band2 = f1 * high2 + *ynm1;
		*ynm1 = band2;
		*ynm2 = low2;
	} while (--nsmps);
}

Opcode opcodes[] = {
	SVFILTER_OPCODE_LIST,
	{ NULL }
};
