/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: shaker.cc,v 1.2 1999/12/16 03:53:35 pbd Exp $
*/

#include <stdlib.h>
#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/process.h>

#include "physutil.h"
#include "shaker.h"

/********************************************************/
/*    Maracha SImulation by Perry R. Cook, 1996         */
/********************************************************/
/********************************************************/
/*    In real life, each grain has an individual 
   sound and envelope, but if you buy the 
   notion that each sound is independent       
   noise, the sum of a bunch of independent 
   exponentially decaying enveloped noises 
   is a single exponentially decaying enveloped 
   noise.  shakeEnergy is an exponentially      
   decaying, but reexcitable by shaking, energy
   expressing how loud a single collision will be.

   This code would implement individual grain envelopes            

   if (random(8) < 1) {  
   noises[which] = 1024 * shakeEnergy;
   which += 1;
   if (which==MAX) which = 0;
   }
   input = 0.0;
   for (i=0;i<MAX;i++)     {
   input += noises[i] * noise_tick();
   noises[i] *= COLL_DECAY;
   }

   But we're smarter than that!!!  See below
 */


void
shakerset(SHAKER * p)
{
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */

	p->shake_speed = 0.0008f + (amp * 0.0004f);
	make_BiQuad(&p->filter);
	make_ADSR(&p->envelope);
	p->res_freq = 3200.0f;
	BiQuad_setFreqAndReson(p->filter, p->res_freq, 0.96f);
	BiQuad_setEqualGainZeroes(p->filter);
	BiQuad_setGain(p->filter, 1.0f);
	p->shakeEnergy = 0.0f;
	p->noiseGain = 0.0f;
	p->coll_damp = 0.95f;
	ADSR_setAll(&p->envelope,
		    p->shake_speed, p->shake_speed, 0.0f, p->shake_speed);
	p->num_beans = (int) *p->beancount;
	p->wait_time = RAND_MAX / p->num_beans;
	p->gain_norm = 0.0005f;
	p->shake_num = (int) *p->times;
	ADSR_keyOn(&p->envelope);
	p->kloop = (int) (p->PROCESS->time_off() * ekr) - (int) (ekr *
							    *p->dettack);
	p->freq = -1;		/* So will get changed */
}

void
shaker(SHAKER * p)
{
	float *ar = p->ar;
	long nsmps = ksmps;
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */
	float shake = amp + amp;
	float damp = *p->shake_damp;
	float gain = p->gain_norm;
	float ngain = p->noiseGain;
	float sEnergy = p->shakeEnergy;
	float shake_speed = 0.0008f + amp * 0.0004f;

	if (p->freq != *p->kfreq)
		BiQuad_setFreqAndReson(p->filter, p->freq = *p->kfreq, 0.96f);
	if (p->num_beans != (int) *p->beancount) {	/* Bean Count */
		p->num_beans = (int) *p->beancount;
		p->wait_time = RAND_MAX / p->num_beans;
	}
	if (shake_speed != p->shake_speed) {
		p->shake_speed = shake_speed;
		ADSR_setAll(&p->envelope, shake_speed, shake_speed, 0.0f,
			    shake_speed);
	}
	if (p->kloop > 0 && p->PROCESS->releasing())
		p->kloop = 1;
	if ((--p->kloop) == 0) {
		p->shake_num = 0;
	}
	gain *= p->num_beans;	/* Save work in loop */
	do {
		float lastOutput;
		float temp;

		ADSR_tick(&p->envelope);
		temp = p->envelope.value * shake;
		if (p->shake_num > 0) {
			if (p->envelope.state == SUSTAIN) {
				if (p->shake_num < 64)
					p->shake_num -= 1;
				ADSR_keyOn(&p->envelope);
			}
		}
		if (temp > sEnergy)
			sEnergy = temp;
		sEnergy *= damp;	/*   Exponential System Decay */

		/* There's Roughly Constant Probablity of a Collision, and */
		/* Energy of Each Collision is Weighted by Exponentially   */
		/* Decaying System Energy.  All add together for total     */
		/* exponentially decaying sound energy.                    */

		if (rand() < p->wait_time) {
			ngain += gain * sEnergy;
		}
		lastOutput = ngain *
		    ((float) rand() - (float) (RAND_MAX / 2)) /
		    (float) (RAND_MAX / 2);	/* Actual Sound is Random  
						 */
		ngain *= p->coll_damp;	/* Each (all) event(s)      */
		/* decay(s) exponentially   */
		lastOutput = BiQuad_tick(&p->filter, lastOutput);
		*ar++ = lastOutput * AMP_SCALE * 7.0f;	/* As too quiet
							 */
	} while (--nsmps);
	p->noiseGain = ngain;
	p->shakeEnergy = sEnergy;
}

Opcode opcodes[] = {
	SHAKER_OPCODE_LIST,
	{ NULL }
};

