/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: marimba.cc,v 1.2 1999/12/16 03:53:35 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/process.h>

#include "physutil.h"
#include "marimba.h"


/*******************************************/
/*  Marimba SubClass of Modal4 Module, */
/*  by Perry R. Cook, 1995-96              */
/*                                         */
/*   Controls:    stickHardness            */
/*                strikePosition           */
/*                vibFreq                  */
/*                vibAmt                   */
/*******************************************/

void
marimbaset(MARIMBA * p)
{
	Modal4 *m = &(p->m4);
	Number temp, temp2;
	int32 itemp;
	RCPointer<FunctionTable>	ftp;

	if ((ftp = ftfind(p->ifn)) != 0) {
		p->m4.wave = ftp;
	} else {
		p->error ("No table for Marimba strike");
	}

	/* Expect an impulslything */

	make_Modal4(m, p->ivfn, *p->vibAmt, *p->vibFreq);
	p->m4.w_phaseOffset = 0.0f;

	Modal4_setRatioAndReson(m, 0, 1.00f, 0.9996f);	/*  Set all       
							   132.0  */
	Modal4_setRatioAndReson(m, 1, 3.99f, 0.9994f);	/*  of our        
							   523.0  */
	Modal4_setRatioAndReson(m, 2, 10.65f, 0.9994f);		/* 
								   default       1405.0 */
	Modal4_setRatioAndReson(m, 3, -18.50f, 0.999f);		/* 
								   resonances    2443.0 */
	Modal4_setFiltGain(m, 0, 0.08f);	/*  and        */
	Modal4_setFiltGain(m, 1, 0.02f);	/*  gains      */
	Modal4_setFiltGain(m, 2, 0.02f);	/*  for each   */
	Modal4_setFiltGain(m, 3, 0.015f);	/*  resonance  */
	p->m4.directGain = 0.1f;
	p->multiStrike = 0;
	p->strikePosition = *p->spos;
	/* Set Stick hardness stuff */
	p->stickHardness = *p->hardness;

	p->m4.w_rate = (0.25f * (Number) pow(4.0, (double) p->stickHardness));
	p->m4.masterGain = (0.1f + (1.8f * p->stickHardness));

	/* Set Strike position */
	temp2 = p->strikePosition * pi;
	temp = (Number) sin((double) temp2);
	BiQuad_setGain(p->m4.filters[0], 0.12f * temp);		/* 1st
								   mode function of pos. */
	temp = (Number) sin(0.05 + (3.9 * (double) temp2));
	BiQuad_setGain(p->m4.filters[1], -0.03f * temp);	/* 2nd mode
								   function of pos. */
	temp = (Number) sin(-0.05 + (11.0 * (double) temp2));
	BiQuad_setGain(p->m4.filters[2], 0.11f * temp);		/* 3rd
								   mode function of pos. */
	/* Strike */
	itemp = rand() % 5;
	if (itemp < 2) {
		p->multiStrike = 1;
	} else if (itemp < 1) {
		p->multiStrike = 2;
	} else {
		p->multiStrike = 0;
	}

	Modal4_strike(m, *p->amplitude * AMP_RSCALE);
	Modal4_setFreq(m, *p->frequency);

	{
		uint32 relestim = (uint32 ) (esr * *p->dettack);

		if (relestim > p->PROCESS->extra_time)
			p->PROCESS->extra_time = relestim;
	}

	/* XXX this is wrong */

	p->kloop = (int32) ((p->PROCESS->time_off() * ekr) - 
			    (int32) (ekr * *p->dettack));
}

void
marimba(MARIMBA * p)
{
	Modal4 *m = &(p->m4);
	Number *ar = p->ar;
	long nsmps = ksmps;
	Number amp = (*p->amplitude) * AMP_RSCALE;	/* Normalise */

	if (p->kloop >= 0 && p->PROCESS->releasing()) {
		p->kloop = 1;
	}

	if (p->kloop && ((--p->kloop) == 0)) {
		Modal4_damp(m, 1.0f - (amp * 0.03f));
	}

	p->m4.v_rate = *p->vibFreq;	/* 6.0; */
	p->m4.vibrGain = *p->vibAmt;	/* 0.05; */

	do {
		Number lastOutput;
		if (p->multiStrike > 0)
			if (p->m4.w_allDone) {
				p->m4.w_time = 0.0f;
				p->m4.w_lastOutput = 0.0f;
				p->m4.w_allDone = 0;
				p->multiStrike -= 1;
			}
		lastOutput = Modal4_tick(m);
		*ar++ = lastOutput * AMP_SCALE * 0.6f;
	} while (--nsmps);
}

Opcode opcodes[] = 
{
	MARIMBA_OPCODE_LIST,
	{ NULL }
};


