/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: mandolin.cc,v 1.2 1999/12/16 03:53:35 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/function_tables.h>
#include <quasimodo/process.h>

#include "physutil.h"
#include "mandolin.h"


/********************************************/
/*  Commuted Mandolin Subclass of enhanced  */
/*  dual plucked-string model               */
/*  by Perry Cook, 1995-96                  */
/*   Controls:    CONTROL1 = bodySize       */
/*                CONTROL2 = pluckPosition  */
/*                CONTROL3 = loopGain       */
/*                MOD_WHEEL= deTuning       */
/*                                          */
/*  Note: Commuted Synthesis, as with many  */
/*  other WaveGuide techniques, is covered  */
/*  by patents, granted, pending, and/or    */
/*  applied-for.  Many are assigned to the  */
/*  Board of Trustees, Stanford University. */
/*  For information, contact the Office of  */
/*  Technology Licensing, Stanford U.       */
/********************************************/


static int
infoTick(MANDOL * p)
{
	long temp;
	float temp_time, alpha;
	int allDone = 0;

	p->s_time += *p->s_rate;	/*  Update current time          */

	if (p->s_time >= (float) p->soundfile->flen) {	/*  Check for end 
							   of sound     */
		p->s_time = (float) (p->soundfile->flen - 1L);	/*  stick 
								   at end      */
		allDone = 1;	/* Information for one-shot use  */
	} else if (p->s_time < 0.0f)	/*  Check for end of sound       */
		p->s_time = 0.0f;	/*  stick at beg                 */

	temp_time = p->s_time;

	temp = (long) temp_time;	/*  Integer part of time address */
	alpha = temp_time - (float) temp;	/*  fractional part of time
						   address */
	p->s_lastOutput =	/* Do linear interpolation         */
	    ((float *) (p->soundfile->ftable))[temp];
	p->s_lastOutput = p->s_lastOutput +	/*  same as
						   alpha*data[temp+1]      */
	    (alpha * (((float *) (p->soundfile->ftable))[temp + 1] -
		      p->s_lastOutput));	/*  + (1-alpha)data[temp]       
						 */
	p->s_lastOutput *= 0.51f;	/* Scaling hack; see below, and
					   changed */
	return allDone;
}

void
mandolinset(MANDOL * p)
{
	RCPointer<FunctionTable>	ftp;

	if ((ftp = ftfind(p->ifn)) != 0)
		p->soundfile = ftp;
	else
	    p->error ("No table for Mandolin");

	/* Expect pluck wave */

	if (*p->lowestFreq) {
		p->length = (long) (esr / (*p->lowestFreq * 0.9) + 1);
		p->lastFreq = *p->lowestFreq * 2.0f;
	} else {
		p->length = (long) (esr / *p->frequency + 1);
		p->lastFreq = *p->frequency * 2.0f;
	}
	make_DLineA(&p->delayLine1, p->length);
	make_DLineA(&p->delayLine2, p->length);
	make_DLineL(&p->combDelay, p->length);
	make_OneZero(&p->filter1);
	make_OneZero(&p->filter2);
	p->lastLength = p->length * 0.5f;
	p->waveDone = 1;
	p->lastFreq = *p->frequency;
	p->lastLength = (esr / p->lastFreq);	/* length - delays */
	DLineA_setDelay(&p->delayLine1, (p->lastLength / *p->detuning) - 0.5f);
	DLineA_setDelay(&p->delayLine2, (p->lastLength * *p->detuning) - 0.5f);

	/* this function gets interesting here, */
	p->s_time = 0.0f;	/* because pluck may be longer than     */
	/* string length, so we just reset the  */
	/* soundfile and add in the pluck in    */
	/* the tick method.                     */
	/* Set Pick Position                    */
	DLineL_setDelay(&p->combDelay, 0.5f * *p->pluckPos * p->lastLength);
	/*   which puts zeroes at pos*length    */
	p->dampTime = (long) p->lastLength;	/* See tick method below     
						 */
	p->waveDone = 0;
	{
		unsigned short relestim = (unsigned short) (ekr * 0.1f);
                /* 1/10th second decay extention */
		if (relestim > p->PROCESS->extra_time)
			p->PROCESS->extra_time = relestim;
	}
	p->kloop = (int) (p->PROCESS->time_off() * ekr);
}

void
mandolin(MANDOL * p)
{
	float *ar = p->ar;
	long nsmps = ksmps;
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */
	float lastOutput;
	float loopGain;

	loopGain = *p->baseLoopGain + (p->lastFreq * 0.000005f);
	if (loopGain > 1.0f)
		loopGain = 0.99999f;

	if (p->kloop > 0 && p->PROCESS->releasing())
		p->kloop = 1;

	if (p->lastFreq != *p->frequency) {
		p->lastFreq = *p->frequency;
		p->lastLength = (esr / p->lastFreq);	/* length - delays
							 */
		DLineA_setDelay(&p->delayLine1, (p->lastLength / *p->detuning)
				- 0.5f);
		DLineA_setDelay(&p->delayLine2, (p->lastLength * *p->detuning)
				- 0.5f);
	}
	if ((--p->kloop) == 0) {
		loopGain = (1.0f - amp) * 0.5f;
	}
	do {
		float temp = 0.0f;
		if (!p->waveDone) {
			p->waveDone = infoTick(p);	/* as long as it goes 
							   . . .   */
			temp = p->s_lastOutput * amp;	/* scaled pluck
							   excitation    */
			temp = temp - DLineL_tick(&p->combDelay, temp);
			/* with comb filtering */
		}
		if (p->dampTime >= 0) {		/* Damping hack to help
						   avoid */
			p->dampTime -= 1;	/*   overflow on replucking  
						 */
			lastOutput =
			    DLineA_tick(&p->delayLine1,		/*
								   Calculate 1st delay */
					OneZero_tick(&p->filter1,	/* 
									   filterered reflection */
						     temp +	/*  plus
								   pluck excitation      */
				     (p->delayLine1.lastOutput * 0.7f)));
			lastOutput +=
			    DLineA_tick(&p->delayLine2,		/* and
								   2nd delay just like the 1st   */
					OneZero_tick(&p->filter2,
			      temp + (p->delayLine2.lastOutput * 0.7f)));
			/* that's the whole thing!!     */
		} else {	/*  No damping hack after 1 period */
			lastOutput =
			    DLineA_tick(&p->delayLine1,		/*
								   Calculate 1st delay */
					OneZero_tick(&p->filter1,	/* 
									   filtered reflection  */
						     temp +	/*  plus
								   pluck excitation     */
				 (p->delayLine1.lastOutput * loopGain)));
			lastOutput +=
			    DLineA_tick(&p->delayLine2,		/* and
								   2nd delay      */
					OneZero_tick(&p->filter2,	/* 
									   just like the 1st */
						     temp +
				 (p->delayLine2.lastOutput * loopGain)));
		}
		lastOutput *= 1.8f;
		*ar++ = lastOutput * AMP_SCALE;
	} while (--nsmps);
}

Opcode opcodes[]= {
	MANDOLIN_OPCODE_LIST,
	{ NULL }
};
