/* hilbert.c
 *
 * Copyright 1999, by Sean M. Costello
 *
 * hilbert is an implementation of an IIR Hilbert transformer.
 * The structure is based on two 6th-order allpass filters in 
 * parallel, with a constant phase difference of 90 degrees
 * (+- some small amount of error) between the two outputs.
 * Allpass coefficients are calculated at i-time.
 */

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>

#include "hilbert.h"

void hilbertset(HILBERT *p)
{
	int j;  /* used to increment for loop */

	/* pole values taken from Bernie Hutchins, "Musical Engineer's Handbook" */
	static double poles[12] = {.3609, 2.7412, 11.1573, 44.7581, 179.6242, 798.4578, 
				   1.2524, 5.5671, 22.3423, 89.6271, 364.7914, 2770.1114};
	double polefreq[12], rc[12], alpha[12], beta[12];
	/* calculate coefficients for allpass filters, based on sampling rate */
	for(j=0; j<12; j++) {
		/*      p->coef[j] = (1 - (15 * PI * pole[j]) / esr) /
			(1 + (15 * PI * pole[j]) / esr); */
		polefreq[j] = poles[j] * 15.0;
		rc[j] = 1.0 / (2.0 * PI * polefreq[j]);
		alpha[j] = 1.0 / rc[j];
		beta[j] = (1.0 - (alpha[j] / (2.0 * (double)esr))) /
			(1.0 + (alpha[j] / (2.0 * (double)esr)));
		p->xnm1[j] = p->ynm1[j] = 0.0f;
		p->coef[j] = -(Number)beta[j]; 
	}
}

void hilbert(HILBERT *p)
{
	Number xn1 = 0.0f, yn1 = 0.0f, xn2 = 0.0f, yn2 = 0.0f;
	Number *out1, *out2, *in;
	Number *xnm1, *ynm1, *coef;
	int nsmps = ksmps;
	int j;

	xnm1 = p->xnm1;
	ynm1 = p->ynm1;
	coef = p->coef;
	out1 = p->out1;
	out2 = p->out2;
	in = p->in;

	do {
		xn1 = *in;
		/* 6th order allpass filter for cosine output. Structure is
		 * 6 first-order allpass sections in series. Coefficients
		 * taken from arrays calculated at i-time.
		 */
		for (j=0; j < 6; j++) {
			yn1 = coef[j] * (xn1 - p->ynm1[j]) + p->xnm1[j];
			p->xnm1[j] = xn1;
			p->ynm1[j] = yn1;
			xn1 = yn1;
		}
		xn2 = *in++;
		/* 6th order allpass filter for sine output. Structure is
		 * 6 first-order allpass sections in series. Coefficients
		 * taken from arrays calculated at i-time.
		 */
		for (j=6; j < 12; j++) {
			yn2 = coef[j] * (xn2 - p->ynm1[j]) + p->xnm1[j];
			p->xnm1[j] = xn2;
			p->ynm1[j] = yn2;
			xn2 = yn2;
		}
		*out1++ = yn1;
		*out2++ = yn2;
	} while (--nsmps);
}

Opcode opcodes[] = {
	HILBERT_OPCODE_LIST,
	{ NULL }
};
