/*
   Original Work:
    Copyright (C) 1993, 1994 Michel Beaudouin-Lafon, http://www-ihm.lri.fr/~mbl
    					William W. Gaver, http://www-crd.rca.ac.uk/~bill				
    Rewrite for quasimodo:
    Copyright (C) 1999 Stephane Conversy
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: gaver.cc,v 1.3 1999/11/17 13:20:20 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dsp.h>
#include <quasimodo/process.h>

#include "gaver.h"

int
compute_internal_data(GAVER_FILTER *p)
{
	Number bwidth [MAXFILTERS];
	Number freq [MAXFILTERS];
	Number bbw;
	Number bwslope;
	
	/* first set freq and bwidth to log scales */
	/* 0 < size < 100 => 40 < freq[0] < 3000 */
	/* well, size=100 -> big object, so we do 101-size. that's it. */
	Number tmp = (101-*p->size)/100.;
	freq[0] = 40. + tmp * tmp * 2960.;

	/* basic bandwidth */
	/* 0 < material < 100 => .005 < bbw < .1 */
	tmp = *p->material/300.;
	bbw = .005 + tmp * tmp;

	/* slope for bandwidth */
	/* 0 < material < 100 => -.01 < bwslope < .99 */
	bwslope = (*p->material - 1)/100.;

	/* bandwidth of the fundamental */
	bwidth[0] = freq[0] * bbw * bwslope;
	
	/* now set other freq's, bwidths as function of fund*/
	/* freq corresponds to bar resonances, */
	/* bwidth slopes up depending on initial bw */
	// [MAC] 
	int i=1;
	int shape = (int) *p->shape;
	for (; i < MAXFILTERS; ++i) {
		switch (shape) {
			case 0: { // string
				freq[i] = freq[0] * (i+1);
				break;
			}
			default:
			case 1: { // bar
				int n = (i+1)*2 + 1;
				n *= n;
				freq[i] = freq[0] * n / 9.0;
				break;
			}
			
			case 2: { // box
			}
		}
		
		bwidth[i] = (freq[i] - freq[0]) * bbw * bwslope;

		
		/* limit to 1/3 of sample freq (Nyquist is 1/2) */
		//if (freq[i] > _play_frequency/3.)
		if (freq[i] > esr / 2.2)
			break;
   	}

	int numFilters = i;

	// right.  First set up filter coefficients for numflt filters...
	float twopidivsrate = (2.0 * M_PI) / esr;
	
	for (i = 0; i < numFilters; i++) {
		p->filters[i][2] = (exp (bwidth[i] * - twopidivsrate));
		p->filters[i][1] = ((4. * p->filters[i][2]) *
			cos (freq[i] * twopidivsrate) / (p->filters[i][2] + 1));
		p->filters[i][0] = ( (1. - p->filters[i][2]) *
			sqrt (1. - ((p->filters[i][1] * p->filters[i][1]) /
				     (p->filters[i][2] * 4.))) );
	}


	return numFilters;

}


void
gaver_filter_set(GAVER_FILTER *p)
{
	for(int i=0; i<MAXFILTERS; ++i)
		p->xnm1[i]=p->xnm2[i]=p->ynm1[i]=p->ynm2[i]=0;		
}


void
gaver_filter_audio(GAVER_FILTER *p)
{
	int numFilters = compute_internal_data(p);	

	Number *out = p->out;
	Number *in = p->in;
	int nsamps = ksmps;

	if(!numFilters) {
		do {
			*out++=0;
		} while(--nsamps);
		return;
	}
	
	
	Number insamp=0, outsamp=0, osamp[MAXFILTERS];
	
	do {
		// get input sample 
		// divide by number of filters to ensure constant energy through the bank 
//		insamp = GenInput (phase);	      
#if 0
		if (*p->speed == 0) {
			insamp =  0.0;	// nothing when steady
		}
			else		
				if (phase % (101 - (int)*p->speed) == 0)
					insamp = *p->roughness * drand48();
#endif
		insamp = *in++;

		insamp /= numFilters;
		
		// pass it through the filter bank 
		outsamp = 0.0;
		for (int j = 0; j < numFilters; ++j) {
			osamp[j] = ((p->filters[j][0] * insamp) +
				(p->filters[j][1] * p->ynm1[j]) -
				(p->filters[j][2] * p->ynm2[j]));

			p->xnm2[j] = p->xnm1[j];
			p->xnm1[j] = insamp;
			p->ynm2[j] = p->ynm1[j];
			p->ynm1[j] = osamp[j];
			outsamp +=  osamp[j];
			osamp[j] = 0.0;
		}
		
		// scale is the product of the amp resulting from the gain
		// and a normalization constant defined by the subclass
//		outsamp *= scale;
		*out++ = outsamp;
	} while(--nsamps);
}

void
gaver_scrape_set(GAVER_SCRAPE *p)
{
	p->phase = 0;
	p->lastin=0;
}


void
gaver_scrape_audio(GAVER_SCRAPE *p)
{
	Number *out = p->out;
	int nbsamps = ksmps;
	Number amp = *p->amp;
	Number lastin = p->lastin;
	
	do {
	
		if (*p->speed == 0) {
			*out++=0.0;// nothing when steady
		}
		else		
			if (p->phase % (101 - (int)*p->speed) == 0)
				*out++ = lastin = *p->roughness * drand48() * amp;
			else
				*out++ = lastin * amp;
		p->phase++;

	} while(--nbsamps);
	p->lastin = lastin;

}


void
gaver_impact_set(GAVER_IMPACT *p)
{
	p->impulse_length = (int) ((1 + (100 - *p->hardness)) * esr / 10000.);

#ifdef WITHRTSIN
	p->thesin.reset();
#else
	p->a=0;
	p->b=1;
#endif
}


void
gaver_impact_audio(GAVER_IMPACT *p)
{

	Number *out = p->out;
	Number amp = *p->amp * 30000000;
	Number a = p->a, b = p->b;
	
//	p->thesin.set_frequency( (*p->hardness) * 10);

	const Number reald = 1/10.0;
	Number d = ((2 * M_PI) + reald) * (*p->hardness) / esr;

/*
	int nbsamps = ksmps;
	do {
		*out++ = 0.0;
	} while(--nbsamps);
*/

	if(p->impulse_length<=0) {
		int nbsamps = ksmps;
	
		do {
			*out++ = 0.0;
		} while(--nbsamps);
		return;
	}

	int tmp = p->impulse_length - ksmps;
	
	int nbsamps = 0;
	if(tmp<=0) {
		size_t i;
		for(i=0; i< (size_t) p->impulse_length; ++i) {
//			*out++ = p->thesin.sin() * amp;
//			p->thesin.next_sample();
			*out++ = a * amp;
			a = a + d * b;
			b = b - d * a;
		}
		for(;i<ksmps; ++i)
			*out++ = 0.0;
		p->a = a;
		p->b = b;
		p->PROCESS->turnoff ();
		p->impulse_length = tmp;
		return;
	}

	nbsamps=ksmps;
	do {
//		*out++ = p->thesin.sin() * amp;
//		p->thesin.next_sample();
		*out++ = a * amp;
		a = a + d * b;
		b = b - d * a;
	} while(--nbsamps);

	p->a = a;
	p->b = b;
	p->impulse_length = tmp;

}

Opcode opcodes[] = 

{
	GAVER_OPCODE_LIST,
	{ NULL }
};
