/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: fm4op.cc,v 1.2 1999/12/16 03:53:34 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
//#include <quasimodo/dspcode.h>
#include <quasimodo/function_tables.h>
#include "physutil.h"
#include "fm4op.h"


/*********************************************************/
/*  Master Class for 4 Operator FM Synth                 */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97/98 */
/*  This module contains 4 waves, 4 adsr, and        */
/*  various state vars.                                  */
/*                                                       */
/*  The basic Chowning/Stanford FM patent expired April  */
/*  1995, but there exist follow-on patents, mostly      */
/*  assigned to Yamaha.  If you are of the type who      */
/*  should worry about this (making money) worry away.   */
/*                                                       */
/*********************************************************/

static int32 FM_tabs_built = 0;
static Number FM4Op_gains[100];
static Number FM4Op_susLevels[16];
static Number FM4Op_attTimes[32];

void
build_FM(void)
{				/* The following tables are pre-built */
	Number temp = 1.0f;
	int32 i;

	for (i = 99; i >= 0; i--) {
		FM4Op_gains[i] = temp;
		temp *= 0.933033f;
	}
	temp = 1.0f;
	for (i = 15; i >= 0; i--) {
		FM4Op_susLevels[i] = temp;
		temp *= 0.707101f;
	}
	temp = 8.498186f;
	for (i = 0; i < 32; i++) {
		FM4Op_attTimes[i] = temp;
		temp *= 0.707101f;
	}
	FM_tabs_built = 1;
}

void
make_FM4Op(FM4OP * p)
{
	Number tempCoeffs[2] =
	{0.0f, -1.0f};
	RCPointer<FunctionTable>	ftp;

	if (!FM_tabs_built)
		build_FM();	/* Ensure tables exist */

	make_ADSR(&p->adsr[0]);
	make_ADSR(&p->adsr[1]);
	make_ADSR(&p->adsr[2]);
	make_ADSR(&p->adsr[3]);
	make_TwoZero(&p->twozero);
	if ((ftp = ftfind(p->vifn)) != 0)
		p->vibWave = ftp;
	else
	    p->error ("No table for VibWaveato");

	/* Expect sine wave */
	p->baseFreq = 440.0f;
	p->ratios[0] = 1.0f;
	p->ratios[1] = 1.0f;
	p->ratios[2] = 1.0f;
	p->ratios[3] = 1.0f;
	p->gains[0] = 1.0f;
	p->gains[1] = 1.0f;
	p->gains[2] = 1.0f;
	p->gains[3] = 1.0f;
	TwoZero_setZeroCoeffs(&p->twozero, tempCoeffs);
	p->twozero.gain = 0.0f;
}

void
FM4Op_loadWaves(FM4OP * p)
{
	RCPointer<FunctionTable>	ftp;

	if ((ftp = ftfind(p->ifn0)) != 0)
		p->waves[0] = ftp;
	else
		p->error ("No table for FM4Op");

	/* Expect sine wave */
	if ((ftp = ftfind(p->ifn1)) != 0)
		p->waves[1] = ftp;
	else
	    p->error("No table for  FM4Op");

	/* Expect sine wave */

	if ((ftp = ftfind(p->ifn2)) != 0)
		p->waves[2] = ftp;
	else
		p->error ("No table for FM4Op");

	/* Expect sine wave */

	if ((ftp = ftfind(p->ifn3)) != 0)
		p->waves[3] = ftp;
	else
		p->error ("No table for  FM4Op");

	/* Expect sine wave */
}

void
FM4Op_setRatio(FM4OP * p, int32 whichOne, Number ratio)
{
	p->ratios[whichOne] = ratio;
	if (ratio > 0.0f)
		p->w_rate[whichOne] = p->baseFreq * ratio;
	else
		p->w_rate[whichOne] = ratio;
}

void
FM4Op_keyOff(FM4OP * p)
{
	ADSR_keyOff(&p->adsr[0]);
	ADSR_keyOff(&p->adsr[1]);
	ADSR_keyOff(&p->adsr[2]);
	ADSR_keyOff(&p->adsr[3]);
}

/*********************************************************/
/*  Algorithm 5 (TX81Z) Subclass of 4 Operator FM Synth  */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97/98 */
/*  This connection topology is 2 simple FM Pairs summed */
/*  together, like:                                      */
/*                                                       */
/*   Alg 5 is :      4->3--\                             */
/*                          + --> Out                    */
/*                   2->1--/                             */
/*                                                       */
/*  Controls: control1 = mod index 1                     */
/*            control2 = crossfade of two outputs        */
/*                                                       */
/*********************************************************/


Number
FM4Alg5_tick(FM4OP * p, Number c1, Number c2)
{
	Number temp, temp2;
	Number lastOutput;

	temp = p->gains[1] * ADSR_tick(&p->adsr[1]) *
	    Wave_tick(&p->w_time[1], (int32) p->waves[1]->flen,
		      p->waves[1]->ftable,
		      p->w_rate[1], p->w_phase[1]);
	temp = temp * c1;
	p->w_phase[0] = p->waves[0]->flen * temp;	/* addPhaseOffset */
	p->w_phase[3] = p->waves[0]->flen * p->twozero.lastOutput;
	temp = p->gains[3] * ADSR_tick(&p->adsr[3]) *
	    Wave_tick(&p->w_time[3], (int32) p->waves[3]->flen,
		      p->waves[3]->ftable,
		      p->w_rate[3], p->w_phase[3]);
	TwoZero_tick(&p->twozero, temp);
	p->w_phase[2] = p->waves[2]->flen * temp;	/* addPhaseOffset */
	temp = (1.0f - (c2 * 0.5f)) * p->gains[0] *
	    ADSR_tick(&p->adsr[0]) *
	    Wave_tick(&p->w_time[0], (int32) p->waves[0]->flen,
		      p->waves[0]->ftable,
		      p->w_rate[0], p->w_phase[0]);
	temp += c2 * 0.5f * p->gains[2] * ADSR_tick(&p->adsr[2]) *
	    Wave_tick(&p->w_time[2], (int32) p->waves[2]->flen,
		      p->waves[2]->ftable,
		      p->w_rate[2], p->w_phase[2]);

	temp2 = Wave_tick(&p->v_time, (int32) p->vibWave->flen,
			  p->vibWave->ftable, p->v_rate, 0.0f) *
	    *p->modDepth;	/* Calculate amplitude mod */
	temp = temp * (1.0f + temp2);	/*  and apply it to output */

	lastOutput = temp * 0.5f;
	return lastOutput;
}

/***************************************************************/
/*  Tubular Bell (Orch. Chime) Subclass of Algorithm 5 (TX81Z) */
/*  Subclass of 4 Operator FM Synth by Perry R. Cook, 1995-96  */
/*  Recoded in C by John ffitch 1997-98                        */
/***************************************************************/

void
tubebellset(FM4OP * p)
{
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */

	make_FM4Op(p);
	FM4Op_loadWaves(p);	/* 4 times  "rawwaves/sinewave.raw" */

	FM4Op_setRatio(p, 0, 1.0f * 0.995f);
	FM4Op_setRatio(p, 1, 1.414f * 0.995f);
	FM4Op_setRatio(p, 2, 1.0f * 1.005f);
	FM4Op_setRatio(p, 3, 1.414f);
	p->gains[0] = amp * FM4Op_gains[94];
	p->gains[1] = amp * FM4Op_gains[76];
	p->gains[2] = amp * FM4Op_gains[99];
	p->gains[3] = amp * FM4Op_gains[71];
	ADSR_setAll(&p->adsr[0], 0.03f, 0.00001f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[1], 0.03f, 0.00001f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[2], 0.07f, 0.00002f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[3], 0.04f, 0.00001f, 0.0f, 0.02f);
	p->twozero.gain = 0.5f;
	p->v_rate = 2.0f * p->vibWave->flen / esr;	/* Vib rate */
	/* Set Freq */
	p->baseFreq = *p->frequency;
	p->w_rate[0] = p->baseFreq * p->ratios[0] * p->waves[0]->flen / esr;
	p->w_rate[1] = p->baseFreq * p->ratios[1] * p->waves[1]->flen / esr;
	p->w_rate[2] = p->baseFreq * p->ratios[2] * p->waves[2]->flen / esr;
	p->w_rate[3] = p->baseFreq * p->ratios[3] * p->waves[3]->flen / esr;
	ADSR_keyOn(&p->adsr[0]);
	ADSR_keyOn(&p->adsr[1]);
	ADSR_keyOn(&p->adsr[2]);
	ADSR_keyOn(&p->adsr[3]);
}

void
tubebell(FM4OP * p)
{
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */
	Number *ar = p->ar;
	int32 nsmps = ksmps;
	Number c1 = *p->control1;
	Number c2 = *p->control2;

	/* Set Freq */
	p->baseFreq = *p->frequency;
	p->gains[0] = amp * FM4Op_gains[94];
	p->gains[1] = amp * FM4Op_gains[76];
	p->gains[2] = amp * FM4Op_gains[99];
	p->gains[3] = amp * FM4Op_gains[71];
	p->w_rate[0] = p->baseFreq * p->ratios[0] * p->waves[0]->flen / esr;
	p->w_rate[1] = p->baseFreq * p->ratios[1] * p->waves[1]->flen / esr;
	p->w_rate[2] = p->baseFreq * p->ratios[2] * p->waves[2]->flen / esr;
	p->w_rate[3] = p->baseFreq * p->ratios[3] * p->waves[3]->flen / esr;
	p->v_rate = *p->vibFreq * p->vibWave->flen / esr;

	do {
		Number lastOutput;
		lastOutput = FM4Alg5_tick(p, c1, c2);
		*ar++ = lastOutput * AMP_SCALE * 1.8f;
	} while (--nsmps);
}

/*****************************************************************/
/*  Fender Rhodes Electric Piano Subclass of Algorithm 5 (TX81Z) */
/*  Subclass of 4 Operator FM Synth by Perry R. Cook, 1995-96    */
/*  Recoded in C by John ffitch 1997-98                          */
/*****************************************************************/

void
rhodeset(FM4OP * p)
{
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */

	make_FM4Op(p);
	FM4Op_loadWaves(p);	/* 3 times  "sinewave.raw"; 1 x fwavblnk.raw 
				 */

	FM4Op_setRatio(p, 0, 1.0f);
	FM4Op_setRatio(p, 1, 0.5f);
	FM4Op_setRatio(p, 2, 1.0f);
	FM4Op_setRatio(p, 3, 15.0f);
	p->gains[0] = amp * FM4Op_gains[99];
	p->gains[1] = amp * FM4Op_gains[90];
	p->gains[2] = amp * FM4Op_gains[99];
	p->gains[3] = amp * FM4Op_gains[67];
	ADSR_setAll(&p->adsr[0], 0.05f, 0.00003f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[1], 0.05f, 0.00003f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[2], 0.05f, 0.00005f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[3], 0.05f, 0.0002f, 0.0f, 0.02f);
	p->twozero.gain = 1.0f;
	p->v_rate = 2.0f * p->vibWave->flen / esr;	/* Vib rate */
	/* Set Freq */
	p->baseFreq = *p->frequency;
	p->w_rate[0] = p->baseFreq * p->ratios[0] * p->waves[0]->flen / esr;
	p->w_rate[1] = p->baseFreq * p->ratios[1] * p->waves[1]->flen / esr;
	p->w_rate[2] = p->baseFreq * p->ratios[2] * p->waves[2]->flen / esr;
	p->w_rate[3] = p->baseFreq * p->ratios[3] * p->waves[3]->flen / esr;
	ADSR_keyOn(&p->adsr[0]);
	ADSR_keyOn(&p->adsr[1]);
	ADSR_keyOn(&p->adsr[2]);
	ADSR_keyOn(&p->adsr[3]);
}

/***************************************************************/
/*  Wurlitzer Electric Piano Subclass of Algorithm 5 (TX81Z)   */
/*  Subclass of 4 Operator FM Synth by Perry R. Cook, 1995-96  */
/*  Recoded in C by John ffitch 1997-98                        */
/***************************************************************/


void
wurleyset(FM4OP * p)
{
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */

	make_FM4Op(p);
	FM4Op_loadWaves(p);	/* 3 times  "sinewave.raw"; 1 x fwavblnk.raw 
				 */

	FM4Op_setRatio(p, 0, 1.0f);
	FM4Op_setRatio(p, 1, 4.05f);
	FM4Op_setRatio(p, 2, -510.0f);
	FM4Op_setRatio(p, 3, -510.0f);
	p->gains[0] = amp * FM4Op_gains[99];
	p->gains[1] = amp * FM4Op_gains[82];
	p->gains[2] = amp * FM4Op_gains[82];
	p->gains[3] = amp * FM4Op_gains[68];
	ADSR_setAll(&p->adsr[0], 0.05f, 0.00003f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[1], 0.05f, 0.00003f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[2], 0.05f, 0.0002f, 0.0f, 0.02f);
	ADSR_setAll(&p->adsr[3], 0.05f, 0.0003f, 0.0f, 0.02f);
	p->twozero.gain = 2.0f;
	/* Set Freq */
	p->baseFreq = *p->frequency;
	p->w_rate[0] = p->baseFreq * p->ratios[0] * p->waves[0]->flen / esr;
	p->w_rate[1] = p->baseFreq * p->ratios[1] * p->waves[1]->flen / esr;
	p->w_rate[2] = p->ratios[2] * p->waves[2]->flen / esr;
	p->w_rate[3] = p->ratios[3] * p->waves[3]->flen / esr;
	ADSR_keyOn(&p->adsr[0]);
	ADSR_keyOn(&p->adsr[1]);
	ADSR_keyOn(&p->adsr[2]);
	ADSR_keyOn(&p->adsr[3]);
}

void
wurley(FM4OP * p)
{
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */
	Number *ar = p->ar;
	int32 nsmps = ksmps;
	Number c1 = *p->control1;
	Number c2 = *p->control2;

	/* Set Freq */
	p->baseFreq = *p->frequency;
	p->gains[0] = amp * FM4Op_gains[99];
	p->gains[1] = amp * FM4Op_gains[82];
	p->gains[2] = amp * FM4Op_gains[82];
	p->gains[3] = amp * FM4Op_gains[68];
	p->w_rate[0] = p->baseFreq * p->ratios[0] * p->waves[0]->flen / esr;
	p->w_rate[1] = p->baseFreq * p->ratios[1] * p->waves[1]->flen / esr;
	p->w_rate[2] = p->ratios[2] * p->waves[2]->flen / esr;
	p->w_rate[3] = p->ratios[3] * p->waves[3]->flen / esr;
	p->v_rate = *p->vibFreq * p->vibWave->flen / esr;

	do {
		Number lastOutput;
		lastOutput = FM4Alg5_tick(p, c1, c2);
		*ar++ = lastOutput * AMP_SCALE * 1.9f;
	} while (--nsmps);
}

/*********************************************************/
/*  Algorithm 3 (TX81Z) Subclass of 4 Operator FM Synth  */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97/98 */
/*                                                       */
/*  Alg 3 is :          4--\                             */
/*                  3-->2-- + -->1-->Out                 */
/*                                                       */
/*  Controls: control1 = total mod index                 */
/*            control2 = crossfade of two modulators     */
/*                                                       */
/*********************************************************/

Number
FM4Alg3_tick(FM4OP * p, Number c1, Number c2)
{
	Number temp;
	Number lastOutput;

	temp = *p->modDepth * 0.2f * Wave_tick(&p->v_time, (int32)
					       p->vibWave->flen,
				    p->vibWave->ftable, p->v_rate, 0.0f);
	p->w_rate[0] = p->baseFreq * (1.0f + temp) * p->ratios[0];
	p->w_rate[1] = p->baseFreq * (1.0f + temp) * p->ratios[1];
	p->w_rate[2] = p->baseFreq * (1.0f + temp) * p->ratios[2];
	p->w_rate[3] = p->baseFreq * (1.0f + temp) * p->ratios[3];

	temp = p->gains[2] * ADSR_tick(&p->adsr[2]) *
	    Wave_tick(&p->w_time[2], (int32) p->waves[2]->flen,
		      p->waves[2]->ftable,
		      p->w_rate[2], p->w_phase[2]);
	p->w_phase[1] = p->waves[1]->flen * temp;
	p->w_phase[3] = p->waves[3]->flen * p->twozero.lastOutput;
	temp = (1.0f - (c2 * 0.5f)) * p->gains[3] * ADSR_tick(&p->adsr[3]) *
	    Wave_tick(&p->w_time[3], (int32) p->waves[3]->flen,
		      p->waves[3]->ftable,
		      p->w_rate[3], p->w_phase[3]);
	TwoZero_tick(&p->twozero, temp);

	temp += c2 * 0.5f * p->gains[1] * ADSR_tick(&p->adsr[1]) *
	    Wave_tick(&p->w_time[1], (int32) p->waves[1]->flen,
		      p->waves[1]->ftable,
		      p->w_rate[1], p->w_phase[1]);
	temp = temp * c1;

	p->w_phase[0] = p->waves[0]->flen * temp;
	temp = p->gains[0] * ADSR_tick(&p->adsr[0]) *
	    Wave_tick(&p->w_time[0], (int32) p->waves[0]->flen,
		      p->waves[0]->ftable,
		      p->w_rate[0], p->w_phase[0]);

	lastOutput = temp * 0.5f;
	return lastOutput;
}

void
heavymetset(FM4OP * p)
{
	make_FM4Op(p);
	FM4Op_loadWaves(p);	/* Mixed */
	FM4Op_setRatio(p, 0, 1.00f);
	FM4Op_setRatio(p, 1, 4.00f * 0.999f);
	FM4Op_setRatio(p, 2, 3.00f * 1.001f);
	FM4Op_setRatio(p, 3, 0.50f * 1.002f);
	ADSR_setAll(&p->adsr[0], 0.050f, 0.0100f, 1.0f, 0.001f);
	ADSR_setAll(&p->adsr[1], 0.050f, 0.0010f, 1.0f, 0.0001f);
	ADSR_setAll(&p->adsr[2], 0.001f, 0.0020f, 1.0f, 0.0002f);
	ADSR_setAll(&p->adsr[3], 0.050f, 0.0010f, 0.2f, 0.0002f);
	p->twozero.gain = 2.0f;
	ADSR_keyOn(&p->adsr[0]);
	ADSR_keyOn(&p->adsr[1]);
	ADSR_keyOn(&p->adsr[2]);
	ADSR_keyOn(&p->adsr[3]);
}

void
heavymet(FM4OP * p)
{
	Number *ar = p->ar;
	int32 nsmps = ksmps;
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */
	Number c1 = *p->control1;
	Number c2 = *p->control2;

	p->baseFreq = *p->frequency;
	p->gains[0] = amp * FM4Op_gains[92];
	p->gains[1] = amp * FM4Op_gains[76];
	p->gains[2] = amp * FM4Op_gains[91];
	p->gains[3] = amp * FM4Op_gains[68];

	p->w_rate[0] = p->baseFreq * p->ratios[0] * p->waves[0]->flen / esr;
	p->w_rate[1] = p->baseFreq * p->ratios[1] * p->waves[1]->flen / esr;
	p->w_rate[2] = p->baseFreq * p->ratios[2] * p->waves[2]->flen / esr;
	p->w_rate[3] = p->baseFreq * p->ratios[3] * p->waves[3]->flen / esr;
	p->v_rate = *p->vibFreq * p->vibWave->flen / esr;
	do {
		Number lastOutput;
		lastOutput = FM4Alg3_tick(p, c1, c2);
		*ar++ = lastOutput * AMP_SCALE * 2.0f;
	} while (--nsmps);
}

/**********************************************************/
/*  Algorithm 8 (TX81Z) Subclass of 4 Operator FM Synth   */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97-98  */
/*  This connection topology is simple Additive Synthesis */
/*                                                        */
/*                   1 --.                                */
/*                   2 -\|                                */
/*                       +-> Out                          */
/*                   3 -/|                                */
/*                   4 --                                 */
/*                                                        */
/*  Controls: control1 = op4 (fb) gain                    */
/*            control2 = op3 gain                         */
/*                                                        */
/**********************************************************/

Number
FM4Alg8_tick(FM4OP * p, Number c1, Number c2)
{
	Number temp;
	Number lastOutput;

	p->w_phase[3] = p->waves[3]->flen * p->twozero.lastOutput;

	temp = c1 * 2.0f * p->gains[3] * ADSR_tick(&p->adsr[3]) *
	    Wave_tick(&p->w_time[3], (int32) p->waves[3]->flen,
		      p->waves[3]->ftable,
		      p->w_rate[3], p->w_phase[3]);
	TwoZero_tick(&p->twozero, temp);
	temp += c2 * 2.0f * p->gains[2] * ADSR_tick(&p->adsr[2]) *
	    Wave_tick(&p->w_time[2], (int32) p->waves[2]->flen,
		      p->waves[2]->ftable,
		      p->w_rate[2], p->w_phase[2]);
	temp += p->gains[1] * ADSR_tick(&p->adsr[1]) *
	    Wave_tick(&p->w_time[1], (int32) p->waves[1]->flen,
		      p->waves[1]->ftable,
		      p->w_rate[1], p->w_phase[1]);
	temp += p->gains[0] * ADSR_tick(&p->adsr[0]) *
	    Wave_tick(&p->w_time[0], (int32) p->waves[0]->flen,
		      p->waves[0]->ftable,
		      p->w_rate[0], p->w_phase[0]);

	lastOutput = temp * 0.125f;
	return lastOutput;
}


/**************************************************************/
/*  Hammond(OID) Organ Subclass of Algorithm 8 (TX81Z)        */
/*  Subclass of 4 Operator FM Synth by Perry R. Cook, 1995-96 */
/*  Recoded in C by John ffitch 1997-98                       */
/**************************************************************/


void
b3set(FM4OP * p)
{
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */

	make_FM4Op(p);
	FM4Op_loadWaves(p);	/* sines */
	FM4Op_setRatio(p, 0, 0.999f);
	FM4Op_setRatio(p, 1, 1.997f);
	FM4Op_setRatio(p, 2, 3.006f);
	FM4Op_setRatio(p, 3, 6.009f);

	p->gains[0] = amp * FM4Op_gains[95];
	p->gains[1] = amp * FM4Op_gains[95];
	p->gains[2] = amp * FM4Op_gains[99];
	p->gains[3] = amp * FM4Op_gains[95];
	ADSR_setAll(&p->adsr[0], 0.05f, 0.03f, 1.0f, 0.04f);
	ADSR_setAll(&p->adsr[1], 0.05f, 0.03f, 1.0f, 0.04f);
	ADSR_setAll(&p->adsr[2], 0.05f, 0.03f, 1.0f, 0.04f);
	ADSR_setAll(&p->adsr[3], 0.05f, 0.001f, 0.4f, 0.06f);
	p->twozero.gain = 0.1f;
	ADSR_keyOn(&p->adsr[0]);
	ADSR_keyOn(&p->adsr[1]);
	ADSR_keyOn(&p->adsr[2]);
	ADSR_keyOn(&p->adsr[3]);
}

void
hammondB3(FM4OP * p)
{
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */
	Number *ar = p->ar;
	int32 nsmps = ksmps;
	Number c1 = *p->control1;
	Number c2 = *p->control2;
	Number temp;

	p->baseFreq = *p->frequency;
	p->gains[0] = amp * FM4Op_gains[95];
	p->gains[1] = amp * FM4Op_gains[95];
	p->gains[2] = amp * FM4Op_gains[99];
	p->gains[3] = amp * FM4Op_gains[95];
	do {
		Number lastOutput;
		if (*p->modDepth > 0.0f) {
			p->v_rate = *p->vibFreq * p->vibWave->flen / esr;
			temp = 1.0f + (*p->modDepth * 0.1f *
			    Wave_tick(&p->v_time, (int32) p->vibWave->flen,
				   p->vibWave->ftable, p->v_rate, 0.0f));
			temp *= p->baseFreq;
			p->w_rate[0] = p->ratios[0] * temp * p->waves[0]->flen
			    / esr;
			p->w_rate[1] = p->ratios[1] * temp * p->waves[1]->flen
			    / esr;
			p->w_rate[2] = p->ratios[2] * temp * p->waves[2]->flen
			    / esr;
			p->w_rate[3] = p->ratios[3] * temp * p->waves[3]->flen
			    / esr;
		}
		lastOutput = FM4Alg8_tick(p, c1, c2);
		*ar++ = lastOutput * AMP_SCALE;
	} while (--nsmps);
}

/************************************************************/
/*  Algorithm 6 (TX81Z) Subclass of 4 Operator FM Synth     */
/*  by Perry R. Cook, 1995-96; recoded John ffitch 97-98    */
/*  This connection topology is three Carriers and a common */
/*  Modulator     /->1 -\                                   */
/*             4-|-->2 - +-> Out                            */
/*                \->3 -/                                   */
/*                                                          */
/*  Controls: control1 = vowel                              */
/*            control2 = spectral tilt                      */
/*                                                          */
/************************************************************/

Number
FM4Alg6_tick(FM4OPV * q)
{
	Number temp, temp2;
	FM4OP *p = (FM4OP *) q;

	temp = p->gains[3] * ADSR_tick(&p->adsr[3]) *
	    Wave_tick(&p->w_time[3], (int32) p->waves[3]->flen,
		      p->waves[3]->ftable,
		      p->w_rate[3], p->w_phase[3]);
	/*  Calculate frequency mod  */
	temp2 = Wave_tick(&p->v_time, (int32) p->vibWave->flen,
			  p->vibWave->ftable,
			  p->v_rate, 0.0f) * *p->modDepth * 0.1f;

	temp2 = (1.0f + temp2) * p->baseFreq / esr;
	p->w_rate[0] = temp2 * p->ratios[0] * p->waves[0]->flen;
	p->w_rate[1] = temp2 * p->ratios[1] * p->waves[1]->flen;
	p->w_rate[2] = temp2 * p->ratios[2] * p->waves[2]->flen;
	p->w_rate[3] = temp2 * p->ratios[3] * p->waves[3]->flen;

	p->w_phase[0] = p->waves[0]->flen * temp * q->mods[0];
	p->w_phase[1] = p->waves[1]->flen * temp * q->mods[1];
	p->w_phase[2] = p->waves[2]->flen * temp * q->mods[2];
	p->w_phase[3] = p->waves[3]->flen * p->twozero.lastOutput;

	TwoZero_tick(&p->twozero, temp);

	temp = p->gains[0] * q->tilt[0] * ADSR_tick(&p->adsr[0]) *
	    Wave_tick(&p->w_time[0], (int32) p->waves[0]->flen,
		      p->waves[0]->ftable,
		      p->w_rate[0], p->w_phase[0]);
	temp += p->gains[1] * q->tilt[1] * ADSR_tick(&p->adsr[1]) *
	    Wave_tick(&p->w_time[1], (int32) p->waves[1]->flen,
		      p->waves[1]->ftable,
		      p->w_rate[1], p->w_phase[1]);
	temp += p->gains[2] * q->tilt[2] * ADSR_tick(&p->adsr[2]) *
	    Wave_tick(&p->w_time[2], (int32) p->waves[2]->flen,
		      p->waves[2]->ftable,
		      p->w_rate[2], p->w_phase[2]);

	return temp * 0.33f;
}

#define currentVowel (*q->control1)

void
FMVoices_setFreq(FM4OPV * q, Number frequency)
{
	Number temp;
	Number temp2 = 0;
	int32 tempi;
	int tempi2 = 0;

	if (currentVowel < 16) {
		tempi2 = (int32) currentVowel;
		temp2 = 0.9f;
	} else if (currentVowel < 32) {
		tempi2 = (int32) currentVowel - 16;
		temp2 = 1.0f;
	} else if (currentVowel < 48) {
		tempi2 = (int32) currentVowel - 32;
		temp2 = 1.1f;
	} else if (currentVowel < 64) {
		tempi2 = (int32) currentVowel - 48;
		temp2 = 1.2f;
	}
	q->baseFreq = frequency;
	temp = (temp2 * phonParams[tempi2][0][0] / q->baseFreq) + 0.5f;
	tempi = (int32) temp;
	FM4Op_setRatio((FM4OP *) q, 0, (Number) tempi);
	temp = (temp2 * phonParams[tempi2][1][0] / q->baseFreq) + 0.5f;
	tempi = (int32) temp;
	FM4Op_setRatio((FM4OP *) q, 1, (Number) tempi);
	temp = (temp2 * phonParams[tempi2][2][0] / q->baseFreq) + 0.5f;
	tempi = (int32) temp;
	FM4Op_setRatio((FM4OP *) q, 2, (Number) tempi);
	q->gains[0] = 1.0f;	/* pow(10.0f,phonParams[tempi2][0][2] *
				   0.05f); */
	q->gains[1] = 1.0f;	/* pow(10.0f,phonParams[tempi2][1][2] *
				   0.05f); */
	q->gains[2] = 1.0f;	/* pow(10.0f,phonParams[tempi2][2][2] *
				   0.05f); */
}

void
FMVoiceset(FM4OPV * q)
{
	FM4OP *p = (FM4OP *) q;
	Number amp = *q->amp * AMP_RSCALE;

	make_FM4Op(p);
	FM4Op_loadWaves(p);
	FM4Op_setRatio(p, 0, 2.00f);
	FM4Op_setRatio(p, 1, 4.00f);
	FM4Op_setRatio(p, 2, 12.0f);
	FM4Op_setRatio(p, 3, 1.00f);
	p->gains[3] = FM4Op_gains[80];
	ADSR_setAll(&p->adsr[0], 0.001f, 0.001f, FM4Op_susLevels[15], 0.001f);
	ADSR_setAll(&p->adsr[1], 0.001f, 0.001f, FM4Op_susLevels[15], 0.001f);
	ADSR_setAll(&p->adsr[2], 0.001f, 0.001f, FM4Op_susLevels[15], 0.001f);
	ADSR_setAll(&p->adsr[3], 0.05f, 0.05f, FM4Op_susLevels[15], 0.0001f);
	p->twozero.gain = 0.0f;
	q->tilt[0] = 1.0f;
	q->tilt[1] = 0.5f;
	q->tilt[2] = 0.2f;
	q->mods[0] = 1.0f;
	q->mods[1] = 1.1f;
	q->mods[2] = 1.1f;
	p->baseFreq = 110.0f;
	FMVoices_setFreq(q, 110.0f);
	q->tilt[0] = amp;
	q->tilt[1] = amp * amp;
	q->tilt[2] = amp * amp * amp;
	ADSR_keyOn(&p->adsr[0]);
	ADSR_keyOn(&p->adsr[1]);
	ADSR_keyOn(&p->adsr[2]);
	ADSR_keyOn(&p->adsr[3]);
	q->last_control = -1.0f;
}

void
FMVoice(FM4OPV * q)
{
	FM4OP *p = (FM4OP *) q;
	Number amp = *q->amp * AMP_RSCALE;
	Number *ar = q->ar;
	int32 nsmps = ksmps;

	if (p->baseFreq != *q->frequency || *q->control1 != q->last_control) {
		q->last_control = *q->control1;
		p->baseFreq = *q->frequency;
		FMVoices_setFreq(q, p->baseFreq);
	}
	q->tilt[0] = amp;
	q->tilt[1] = amp * amp;
	q->tilt[2] = amp * amp * amp;
	p->gains[3] = FM4Op_gains[(int32) (*p->control2 * 0.78125f)];

	do {
		Number lastOutput;
		lastOutput = FM4Alg6_tick(q);
		*ar++ = lastOutput * AMP_SCALE * 0.8f;
	} while (--nsmps);

}

/* ********************************************************************** */

/*********************************************************/
/*  Algorithm 4 (TX81Z) Subclass of 4 Operator FM Synth  */
/*  by Perry R. Cook, 1995-96  Recoded John ffitch 97/98 */
/*                                                       */
/*  Alg 4 is :      4->3--\                              */
/*                     2-- + -->1-->Out                  */
/*                                                       */
/*  Controls: control1 = total mod index                 */
/*            control2 = crossfade of two                */
/*                          modulators                   */
/*                                                       */
/*********************************************************/

Number
FM4Alg4_tick(FM4OP * p, Number c1, Number c2)
{
	Number temp;
	Number lastOutput;

	temp = Wave_tick(&p->v_time, (int32) p->vibWave->flen,
			 p->vibWave->ftable, p->v_rate, 0.0f) *
	    *p->modDepth * 0.2f;
	temp = p->baseFreq * (1.0f + temp) / esr;
	p->w_rate[0] = p->ratios[0] * temp * p->waves[0]->flen;
	p->w_rate[1] = p->ratios[1] * temp * p->waves[1]->flen;
	p->w_rate[2] = p->ratios[2] * temp * p->waves[2]->flen;
	p->w_rate[3] = p->ratios[3] * temp * p->waves[3]->flen;

	p->w_phase[3] = p->waves[3]->flen * p->twozero.lastOutput;
	temp = p->gains[3] * ADSR_tick(&p->adsr[3]) *
	    Wave_tick(&p->w_time[3], (int32) p->waves[3]->flen,
		      p->waves[3]->ftable,
		      p->w_rate[3], p->w_phase[3]);
	TwoZero_tick(&p->twozero, temp);
	p->w_phase[2] = p->waves[2]->flen * temp;
	temp = (1.0f - (c2 * 0.5f)) * p->gains[2] * ADSR_tick(&p->adsr[2]) *
	    Wave_tick(&p->w_time[2], (int32) p->waves[2]->flen,
		      p->waves[2]->ftable,
		      p->w_rate[2], p->w_phase[2]);
	temp += c2 * 0.5f * p->gains[1] * ADSR_tick(&p->adsr[1]) *
	    Wave_tick(&p->w_time[1], (int32) p->waves[1]->flen,
		      p->waves[1]->ftable,
		      p->w_rate[1], p->w_phase[1]);
	temp = temp * c1;
	p->w_phase[0] = p->waves[0]->flen * temp;
	temp = p->gains[0] * ADSR_tick(&p->adsr[0]) *
	    Wave_tick(&p->w_time[0], (int32) p->waves[0]->flen,
		      p->waves[0]->ftable,
		      p->w_rate[0], p->w_phase[0]);

	lastOutput = temp * 0.5f;
	return lastOutput;
}


void
percfluteset(FM4OP * p)
{
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */

	make_FM4Op(p);
	FM4Op_loadWaves(p);	/* sines */

	FM4Op_setRatio(p, 0, 1.50f);
	FM4Op_setRatio(p, 1, 3.00f * 0.995f);
	FM4Op_setRatio(p, 2, 2.99f * 1.005f);
	FM4Op_setRatio(p, 3, 6.00f * 0.997f);

	p->gains[0] = amp * FM4Op_gains[99];
	p->gains[1] = amp * FM4Op_gains[71];
	p->gains[2] = amp * FM4Op_gains[93];
	p->gains[3] = amp * FM4Op_gains[85];
	ADSR_setAll(&p->adsr[0], 0.001f, 0.001f, FM4Op_susLevels[14], 0.001f);
	ADSR_setAll(&p->adsr[1], 0.05f, 0.0001f, FM4Op_susLevels[13], 0.0001f);
	ADSR_setAll(&p->adsr[2], 0.05f, 0.0020f, FM4Op_susLevels[11], 0.001f);
	ADSR_setAll(&p->adsr[3], 0.05f, 0.0010f, FM4Op_susLevels[13], 0.005f);
	p->twozero.gain = 0.0f;

	ADSR_keyOn(&p->adsr[0]);
	ADSR_keyOn(&p->adsr[1]);
	ADSR_keyOn(&p->adsr[2]);
	ADSR_keyOn(&p->adsr[3]);
}


void
percflute(FM4OP * p)
{
	Number *ar = p->ar;
	int32 nsmps = ksmps;
	Number amp = *p->amp * AMP_RSCALE;	/* Normalised */
	Number c1 = *p->control1;
	Number c2 = *p->control2;

	p->baseFreq = *p->frequency;
	p->gains[0] = amp * FM4Op_gains[99];
	p->gains[1] = amp * FM4Op_gains[71];
	p->gains[2] = amp * FM4Op_gains[93];
	p->gains[3] = amp * FM4Op_gains[85];
	do {
		Number lastOutput;
		lastOutput = FM4Alg4_tick(p, c1, c2);
		*ar++ = lastOutput * AMP_SCALE * 2.0f;
	} while (--nsmps);
}

Opcode opcodes[] = {
	FM4OP_OPCODE_LIST,
	{ NULL }
};
