/*
    Copyright (C) 1999 Paul Barton-Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: fileops.cc,v 1.4 1999/12/04 15:43:53 pbd Exp $
*/

#include <unistd.h>
#include <errno.h>
#include <string.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/opcode_enums.h>
#include <quasimodo/process.h>

#include "fileops.h"

void
fileopen (FILEOPEN *p)

{
	FILE *f;
	const char *path;
	const char *mode;

	if (!is_string_ptr (p->path)) {
		p->error ("fopen: first argument must be a string");
	}

	if (!is_string_ptr (p->mode)) {
		p->error ("fopen: second argument must be a string");
	}

	path = get_string (p->path);
	mode = get_string (p->mode);
	
	f = fopen (path, mode);
	if (f == 0) {
		p->error ("fopen: cannot open file \"%s\" with mode \"%s\"",
			  path, mode);
	}

	*p->file_index = p->PROCESS->add_open_file (f);
}

void
fileclose (FILECLOSE *p)

{
	FILE *f;

	if ((f = p->PROCESS->remove_open_file (*p->file_index)) == 0) {
		p->error ("fileclose: no such file (index=%d)", *p->file_index);
	}

	fclose (f);
}

void
fileread (FILEIO *p)

{
	FILE *f;
	int nread;
	int to_read;

	if ((f = p->PROCESS->get_open_file (*p->file_index)) == 0) {
		p->error ("fileread: no such file (index=%d)", *p->file_index);
	}

	if (p->nth_arg_is_audio (1)) {
		to_read = CycleFrames;
	} else {
		to_read = 1;
	}

	nread = fread (p->data, sizeof (float), to_read, f);

	if (nread != to_read) {
		if (ferror (f) || nread != 0) {
			p->error ("fileread: read failure");
		} 
	}
}

void
filewrite (FILEIO *p)

{
	FILE *f;
	int nwritten;
	int to_write;

	if ((f = p->PROCESS->get_open_file (*p->file_index)) == 0) {
		p->error ("filewrite: no such file (index=%d)", *p->file_index);
	}

	if (p->nth_arg_is_audio (1)) {
		to_write = CycleFrames;
	} else {
		to_write = 1;
	}

	nwritten = fwrite (p->data, sizeof (float), to_write, f);

	if (nwritten != to_write) {
		if (ferror (f) || nwritten != 0) {
			p->error ("fileread: write failure");
		} 
	}
}

void
fileseek (FILESEEK *p) 

{
	FILE *f;
	int whence = SEEK_SET;
	
	if ((f = p->PROCESS->get_open_file (*p->file_index)) == 0) {
		p->error ("filewrite: no such file (index=%d)", *p->file_index);
	}

	switch ((int) *p->whence) {
	case 0:
		whence = SEEK_SET;
		break;
	case 1:
		whence = SEEK_CUR;
		break;
	case 2:
		whence = SEEK_END;
		break;
	default:
		p->error ("fileseek: don't know how to seek using whence = %d",
			  *p->whence);
	}

	if (fseek (f, (long) *p->position, whence) != 0) {
		p->error ("fileseek: cannot seek to %d/%d",
			  *p->position, *p->whence);
	}
}

void
filetmp (FILETMP *p)

{
	FILE *f;

	if ((f = tmpfile ()) == 0) {
		p->error ("filetmp: cannot open temporary file");
	}

	*p->file_index = p->PROCESS->add_open_file (f);
}


void
filetell (FILEMISC *p) 

{
	FILE *f;
	
	if ((f = p->PROCESS->get_open_file (*p->file_index)) == 0) {
		p->error ("filewrite: no such file (index=%d)", *p->file_index);
	}

	*p->result = ftell (f);
}

void
fileeof (FILEMISC *p)

{
	FILE *f;
	
	if ((f = p->PROCESS->get_open_file (*p->file_index)) == 0) {
		p->error ("filewrite: no such file (index=%d)", *p->file_index);
	}

	*p->result = feof (f);
}

void
fileerror (FILEMISC *p)

{
	FILE *f;
	
	if ((f = p->PROCESS->get_open_file (*p->file_index)) == 0) {
		p->error ("filewrite: no such file (index=%d)", *p->file_index);
	}

	*p->result = ferror (f);
}

void
filetruncate (FILETRUNCATE *p) 

{
	FILE *f;
	
	if ((f = p->PROCESS->get_open_file (*p->file_index)) == 0) {
		p->error ("filewrite: no such file (index=%d)", *p->file_index);
	}

	if (ftruncate (fileno (f), (int) *p->new_size) != 0) {
		p->error ("filetruncate: truncation failed (%s)",
			  strerror (errno));
	}

	/* go back to the beginning as well */

	fseek (f, 0, SEEK_SET);
}

Opcode opcodes[] =
{
	FILEOPS_OPCODE_LIST,
	{ NULL }
};

