/*
    Copyright (C) 1998-99 Paul Barton-Davis 

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: envelope.cc,v 1.3 1999/12/15 19:41:40 pbd Exp $
*/

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/process.h>
#include <quasimodo/function_tables.h>
#include <quasimodo/module.h>

#include "envelope.h"

void
envelope_set(ENVELOPE * p)
{
	Number step_per_cycle;
	Number step_per_second;
	size_t i, j;
	size_t sections;
	bool use_whole_table_as_single_section = false;

	/* there are 4 required arguments, followed by a
	   number of optional arguments, possibly zero
	*/

	j = p->INOCOUNT - 4;

	for (i = 0; i < j; i++) {
		/* negative values imply missing or unset arguments */
		
		if (*p->args[i] < 0) {
			break;
		}
	}

	if (i == 0) {
		use_whole_table_as_single_section = true;
		sections = 1;
	} else {
		sections = i;
	}

	if ((p->ftp = ftfind(p->ifn)) == NULL) {
		p->error ("envelope: no such table (%s)", 
			  FunctionTable::name_as_string (p->ifn));
		return;
	}

	p->aux.alloc (sections * sizeof (ENVSECTION));
	p->sections = (ENVSECTION *) p->aux.auxp;

	/* compute length, start and end points for each section */

	step_per_second = p->ftp->flen / (*p->idur);
	step_per_cycle = step_per_second / ekr;

	if (use_whole_table_as_single_section) {

		/* just one section, covering the entire table.
		   Note that this implies no distinct release
		   section.
		*/

		p->sections[0].first = 0;
		p->sections[0].last = p->ftp->flen - 1;
		p->last_section = 0;

	} else if (sections == 1) {

		/* just one section, with the start specified
		   by the opcode arguments, and the end
		   being the end of the table. Again, this
		   implies no release section.
		*/

		p->sections[0].first = (size_t) 
			(step_per_second * *p->args[0]);
		p->sections[0].last = p->ftp->flen - 1;
		p->last_section = 0;

	} else {

		/* at least two sections, with start/end positions
		   specified by the opcode arguments.
		*/

		p->sections[0].first = (size_t) 
			(step_per_second * *p->args[0]);

		for (i = 1; i < sections - 2; i++) {
			p->sections[i-1].last = (size_t) 
				(step_per_second * *p->args[i+1]) - 1;
			p->sections[i].first = (size_t) 
				(step_per_second * *p->args[i]);

		}
		
		
		p->sections[i-1].last = (size_t) 
			(step_per_second * *p->args[i]) - 1;

		p->sections[i].first = (size_t ) 
			(step_per_second * *p->args[i]);
		p->sections[i].last = p->ftp->flen - 1;

		p->last_section = i;
	}


	p->section_offset = 0;
	p->current_section = 0;
	p->section_increment = step_per_cycle;
	p->PROCESS->set_enveloped ();
}

void
envelope (ENVELOPE * p)
{
	RCPointer<FunctionTable> ftp;
	size_t section;
	size_t section_offset;
	size_t index;
	ENVSECTION *envsection;

	ftp = p->ftp;

	if (p->PROCESS->releasing()) {
		section = p->last_section;
	} else {
		section = (size_t) *p->ksection;
	}

	if (section > p->last_section) {
		p->error ("%s:envelope: section index %d "
			       "used with only %d section(s) defined",
			       p->MODULE->name().c_str(),
			       p->ksection,
			       p->last_section + 1);
	}

	if (section != p->current_section) {
		section_offset = 0;
		p->section_offset = 0;
		p->current_section = section;
	} else {
		section_offset = (size_t) p->section_offset;
	}

	envsection = &p->sections[section];

	if ((index = envsection->first + section_offset) > envsection->last) {
		*p->out = ftp->ftable[envsection->last] * *p->kamp;
		if (section == p->last_section) {
			p->PROCESS->set_death_imminent ();
		}
	} else {
		*p->out = ftp->ftable[index] * *p->kamp;
		p->section_offset = p->section_offset + p->section_increment;
	}
}

Opcode opcodes[] = {
	ENVELOPE_OPCODE_LIST,
	{ NULL }
};



