/*
    This file is derived from source code distributed as part of
    Csound, a program licensed by MIT. It is Copyright (C) any of
    the named parties in the MIT license and/or original Csound
    source code. Every attempt has been made to leave copyright
    holders names and other identifying information in place.

    It is subject to the same licensing restrictions as Csound. A copy
    of the license is part of the distribution that this file was
    a part of, and is called Csound-Copyright. Because of the
    restrictions in that license, it is necessary to note that:

    This work was carried out to further education and research in the
    field of computer music. Although the program it is intended to be
    used with may be used for any purpose as a compiled binary, the
    source code can only be used for other purposes related to
    education and research.

    $Id: clarinet.cc,v 1.2 1999/12/16 03:53:34 pbd Exp $
*/

#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>
#include <quasimodo/dspcode.h>
#include <quasimodo/process.h>

#include "physutil.h"
#include "clarinet.h"

/* ************************************** */
/*  Waveguide Clarinet model ala Smith    */
/*  after McIntyre, Schumacher, Woodhouse */
/*  by Perry Cook, 1995-96                */
/*  Recoded for Csound by John ffitch     */
/*  November 1997                         */
/*                                        */
/*  This is a waveguide model, and thus   */
/*  relates to various Stanford Univ.     */
/*  and possibly Yamaha and other patents. */
/*                                        */
/* ************************************** */

/**********************************************/
/*  One break point linear reed table object  */
/*  by Perry R. Cook, 1995-96                 */
/*  Consult McIntyre, Schumacher, & Woodhouse */
/*        Smith, Hirschman, Cook, Scavone,    */
/*        more for information.               */
/**********************************************/

float
ReedTabl_LookUp(ReedTabl * r, float deltaP)
    /*   Perform "Table Lookup" by direct clipped  */
    /*   linear function calculation               */
{				/*   deltaP is differential reed pressure      
				 */
	r->lastOutput = r->offSet + (r->slope * deltaP);	/* basic
								   non-lin */
	if (r->lastOutput > 1.0f)
		r->lastOutput = 1.0f;	/* if other way, reed slams shut
					 */
	if (r->lastOutput < -1.0f)
		r->lastOutput = -1.0f;	/* if all the way open, acts like 
					   open end */
/*     info("ReedT: Lookup=%f\n", r->lastOutput); */
	return r->lastOutput;
}


/* *********************************************************************** */
void
clarinset(CLARIN * p)
{
	RCPointer<FunctionTable>	ftp;
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */

	if ((ftp = ftfind(p->ifn)) != 0)
		p->vibr = ftp;
	else
	    p->error ("No table for Clarinet");	/* Expect sine
							   wave */
	if (*p->lowestFreq >= 0.0f) {	/* Skip initialisation */
		if (*p->lowestFreq)
			p->length = (long) (esr / *p->lowestFreq + 1.0f);
		else if (*p->frequency)
			p->length = (long) (esr / *p->frequency + 1.0f);
		else
		    p->error ("No base frequency for clarinet");

		make_DLineL(&p->delayLine, p->length);
		p->reedTable.offSet = 0.7f;
		p->reedTable.slope = -0.3f;
		make_OneZero(&(p->filter));
		make_Envelope(&p->envelope);
		make_Noise(p->noise);
		/*    p->noiseGain = 0.2f; *//* Arguemnts; suggested values? */
		/*    p->vibrGain = 0.1f; */
		{
			unsigned short relestim = (unsigned short) (ekr *
								    0.1f);	/* 1/10th second decay extention */
			if (relestim > p->PROCESS->extra_time)
				p->PROCESS->extra_time = relestim;
		}
		p->kloop = (int) (p->PROCESS->time_off() * ekr) - (int) (ekr *
							     *p->attack);
/*     info("time_off()=%f  kloop=%d\n", */
/*            p->PROCESS->time_off(), p->kloop); */
		p->envelope.rate = amp / (*p->attack * esr);	/* Over
								   0.1sec */
		p->envelope.target = 0.55f + amp * 0.30f;	/* ?? values
								   here */
	}
}

void
clarin(CLARIN * p)
{
	float *ar = p->ar;
	long nsmps = ksmps;
	float amp = (*p->amp) * AMP_RSCALE;	/* Normalise */
	float nGain = *p->noiseGain;
	int v_len = (int) p->vibr->flen;
	float *v_data = p->vibr->ftable;
	float vibGain = *p->vibAmt;
	float vTime = p->v_time;

	p->outputGain = amp + 0.001f;
	DLineL_setDelay(&p->delayLine,	/* length - approx filter delay
					 */
			(esr / *p->frequency) * 0.5f - 1.5f);
	p->v_rate = *p->vibFreq * p->vibr->flen / esr;
	/* Check to see if into decay yet */
	if (p->kloop > 0 && p->PROCESS->releasing())
		p->kloop = 1;
	if ((--p->kloop) == 0) {
		p->envelope.state = 1;	/* Start change */
		p->envelope.rate = p->envelope.value / (*p->dettack * esr);
		p->envelope.target = 0.0f;
/*       info("Set off phase time = %f Breath v,r = %f, %f\n", */
/*              (float)kcounter/ekr, p->envelope.value, p->envelope.rate); */
	}
	do {
		float pressureDiff;
		float breathPressure;
		long temp;
		float temp_time, alpha;
		float nextsamp;
		float v_lastOutput;
		float lastOutput;

		breathPressure = Envelope_tick(&p->envelope);
		breathPressure += breathPressure * nGain *
		    Noise_tick(&p->noise);
		/* Tick on vibrato table */
		vTime += p->v_rate;	/*  Update current time    */
		while (vTime >= v_len)	/*  Check for end of sound */
			vTime -= v_len;		/*  loop back to
						   beginning */
		while (vTime < 0.0f)	/*  Check for end of sound */
			vTime += v_len;		/*  loop back to
						   beginning */

		temp_time = vTime;

#ifdef have_phase
		if (p->v_phaseOffset != 0.0f) {
			temp_time += p->v_phaseOffset;	/*  Add phase
							   offset       */
			while (temp_time >= v_len)	/*  Check for end of
							   sound */
				temp_time -= v_len;	/*  loop back to
							   beginning */
			while (temp_time < 0.0f)	/*  Check for end of
							   sound */
				temp_time += v_len;	/*  loop back to
							   beginning */
		}
#endif
		temp = (long) temp_time;	/*  Integer part of time
						   address    */
		/*  fractional part of time address */
		alpha = temp_time - (float) temp;
		v_lastOutput = v_data[temp];	/* Do linear interpolation
						 */
		/*  same as alpha*data[temp+1] + (1-alpha)data[temp] */
		v_lastOutput += (alpha * (v_data[temp + 1] - v_lastOutput));
		/* End of vibrato tick */
		breathPressure += breathPressure * vibGain * v_lastOutput;
		pressureDiff = OneZero_tick(&p->filter,		/*
								   differential pressure  */
					  DLineL_lastOut(&p->delayLine));
		pressureDiff = (-0.95f * pressureDiff) - breathPressure;
/* of reflected and mouth */
		nextsamp = pressureDiff * ReedTabl_LookUp(&p->reedTable,
							  pressureDiff);
		nextsamp = breathPressure + nextsamp;
		lastOutput =
		    DLineL_tick(&p->delayLine, nextsamp);	/* perform
								   scattering in economical way */
		lastOutput *= p->outputGain;
		*ar++ = lastOutput * AMP_SCALE;
	} while (--nsmps);
	p->v_time = vTime;

}

Opcode opcodes[] = {
	CLARINET_OPCODE_LIST,
	{ NULL }
};
