/*
    Permission granted for GPL release by Hans Mikelson, April 1999

    Copyright (C) 1998-99 Hans Mikelson et al.
 
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    aint32 with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    $Id: biquad.cc,v 1.1 1999/11/01 04:25:02 pbd Exp $
*/

/***************************************************************/
/* biquad, moogvcf, rezzy                                      */
/* Biquadratic digital filter with K-Rate coeff.               */
/*      Moog VCF Simulation by Comajuncosas/Stilson/Smith      */
/*      Rezzy Filter by Hans Mikelson                          */
/* Nested allpass and Lorenz added                             */
/* October 1998 by Hans Mikelson                               */
/***************************************************************/
#include <math.h>

#include <quasimodo/qm.h>
#include <quasimodo/opcode.h>

#include "biquad.h"

/***************************************************************************/
/* The biquadratic filter computes the digital filter two x components and */
/* two y values.                                                           */
/* Coded by Hans Mikelson October 1998                                     */
/***************************************************************************/

void biquadset(BIQUAD *p)
{
	/* The biquadratic filter is initialized to zero.    */
	if (*p->reinit==0.0) {      /* Only reset in in non-legato mode */
		p->xnm1 = p->xnm2 = p->ynm1 = p->ynm2 = 0.0f;
	}
} /* end biquadset(p) */

void biquad(BIQUAD *p)
{
	int32 n;
	Number *out, *in;
	Number xn, yn;
	Number a0 = *p->a0, a1 = *p->a1, a2 = *p->a2;
	Number b0 = *p->b0, b1 = *p->b1, b2 = *p->b2;

	n    = ksmps;
	in   = p->in;
	out  = p->out;

	do {
		xn = *in++;
		yn = (b0*xn + b1*(p->xnm1) + b2* (p->xnm2) - a1*(p->ynm1) -
		      a2*(p->ynm2))/a0;
		p->xnm2 = p->xnm1;
		p->xnm1 = xn;
		p->ynm2 = p->ynm1;
		p->ynm1 = yn;
		*out++ = yn;
	} while (--n);
}

/***************************************************************************/
/* begin MoogVCF by Stilson & Smith of CCRMA,  *****************************/
/* translated to Csound by Josep Comajuncosas, *****************************/
/* translated to C by Hans Mikelson            *****************************/
/***************************************************************************/

void moogvcfset(MOOGVCF *p)
{
	p->xnm1 = p->y1nm1 = p->y2nm1 = p->y3nm1 = 0.0f;
	p->y1n  = p->y2n   = p->y3n   = p->y4n   = 0.0f;
	p->fcocod = p->nth_arg_is_audio (1);
	p->rezcod = p->nth_arg_is_audio (2);
}

void moogvcf(MOOGVCF *p)
{
	int32 n;
	Number *out, *in;
	Number xn;
	Number *fcoptr, *resptr;
	Number fco, res, fcon, kp, pp1d2, scale, k;
	Number max = *p->max;
    
	n    = ksmps;
	in   = p->in;
	out  = p->out;
	fcoptr  = p->fco;
	resptr  = p->res;
	fco     = *fcoptr;
	res     = *resptr;
    
	fcon    = 2.0f*fco/esr;                /* normalized freq. 0 to Nyquist */
	kp      = 3.6f*fcon-1.6f*fcon*fcon-1;  /* Emperical tuning              */
	pp1d2   = (kp+1.0f)/2.0f;              /* Timesaver                     */
	scale   = (Number)exp((1.0-(double)pp1d2)*1.386249); /* Scaling factor  */
	k       = res*scale;
	do {
		xn = *in++ / max;
		xn = xn - k * p->y4n; /* Inverted feed back for corner peaking */

		/* Four cascaded onepole filters (bilinear transform) */
		p->y1n   = xn     * pp1d2 + p->xnm1  * pp1d2 - kp * p->y1n;
		p->y2n   = p->y1n * pp1d2 + p->y1nm1 * pp1d2 - kp * p->y2n;
		p->y3n   = p->y2n * pp1d2 + p->y2nm1 * pp1d2 - kp * p->y3n;
		p->y4n   = p->y3n * pp1d2 + p->y3nm1 * pp1d2 - kp * p->y4n;
				/* Clipper band limited sigmoid */
		p->y4n   = p->y4n - p->y4n * p->y4n * p->y4n / 6.0f;
		p->xnm1  = xn;		/* Update Xn-1	*/
		p->y1nm1 = p->y1n;	/* Update Y1n-1	*/
		p->y2nm1 = p->y2n;	/* Update Y2n-1	*/
		p->y3nm1 = p->y3n;	/* Update Y3n-1	*/
		*out++   = p->y4n * max;

		/* Handle a-rate modulation of fco & res. */
		if (p->fcocod) {
			fco = *(++fcoptr);
		}
		if (p->rezcod) {
			res = *(++resptr);
		}
		if ((p->rezcod==1) || (p->fcocod==1)) {
			fcon    = 2.0f*fco/esr;                /* normalized freq. 0 to Nyquist */
			kp      = 3.6f*fcon-1.6f*fcon*fcon-1;  /* Emperical tuning */
			pp1d2   = (kp+1.0f)/2.0f;              /* Timesaver */
			scale   = (Number)exp((1.0-(double)pp1d2)*1.386249); /* Scaling factor */
			k       = res*scale;
		}
	} while (--n);
}


/***************************************************************/
/* This filter is the Mikelson low pass resonant 2-pole filter */
/* Coded by Hans Mikelson October 1998                         */
/***************************************************************/

void rezzyset(REZZY *p)
{
	p->xnm1 = p->xnm2 = p->ynm1 = p->ynm2 = 0.0f; /* Initialize to zero */
	p->fcocod = p->nth_arg_is_audio (1);
	p->rezcod = p->nth_arg_is_audio (2);
    
} /* end rezzyset(p) */

void rezzy(REZZY *p)
{
	int32 n;
	Number *out, *fcoptr, *rezptr, *in;
	Number fco, rez, xn, yn;
	Number fqcadj, c, rez2, a, csq, b, tval; /* Temporary varibles for the filter */

	n      = ksmps;
	in     = p->in;
	out    = p->out;
	fcoptr = p->fco;
	rezptr = p->rez;
	fco    = *fcoptr;
	rez    = *rezptr;
    
	fqcadj = 0.149659863f*esr;	/* Freq. is adjusted based on sample rate */
	c      = fqcadj/fco;	/* Filter constant c=1/Fco * adjustment       */
	/* Try to keep the resonance under control     */
	if (*p->mode == 0.0f) { /* Low Pass */
		rez2   = rez/(1.0f + (Number)exp((double)fco/11000.0));
		a      = c/rez2 - 1.0f;	/* a depends on both Fco and Rez */
		csq    = c*c;		/* Precalculate c^2 */
		b      = 1.0f + a + csq;	/* Normalization constant */
    
		do {			/* do ksmp times   */
			xn = *in++;		/* Get the next sample */
			/* Mikelson Biquad Filter Guts*/
			yn = (1.0f/(Number)sqrt(1.0+(double)rez)*xn -
			      (-a-2.0f*csq)*(p->ynm1) - csq*(p->ynm2))/b;
        
			p->xnm2 = p->xnm1;	/* Update Xn-2 */
			p->xnm1 = xn;		/* Update Xn-1 */
			p->ynm2 = p->ynm1;	/* Update Yn-2 */
			p->ynm1 = yn;		/* Update Yn-1 */
			*out++ = yn;		/* Generate the output sample */
        
			/* Handle a-rate modulation of fco and rez */
			if (p->fcocod) {
				fco = *(++fcoptr);
			}
			if (p->rezcod) {
				rez = *(++rezptr);
			}
			if ((p->rezcod==1) || (p->fcocod==1)) {
				c      = fqcadj/fco;
				rez2   = rez/(1.0f + (Number)exp((double) fco/11000.0));
				a      = c/rez2 - 1.0f; /* a depends on both Fco and Rez */
				csq    = c*c;  /* Precalculate c^2 */
				b      = 1.0f + a + csq; /* Normalization constant */
			}
		} while (--n);
	}
	else { /* High Pass Rezzy */
		rez2   = rez/(1.0f + (Number)sqrt(sqrt(1.0/((double) c))));
		tval   = 0.75f/(Number)sqrt(1.0 + (double) rez);
		csq    = c*c;
		b      = (c/rez2 + csq);
		do {   /* do ksmp times   */
			xn = *in++;  /* Get the next sample */
			/* Mikelson Biquad Filter Guts*/
			yn = ((c/rez2 + 2.0f*csq - 1.0f)*(p->ynm1) - csq*(p->ynm2)
			      + ( c/rez2 + csq)*tval*xn + (-c/rez2 - 2.0f*csq)*tval*(p->xnm1)
			      + csq*tval*(p->xnm2))/b;
        
			p->xnm2 = p->xnm1;	/* Update Xn-2 */
			p->xnm1 = xn;		/* Update Xn-1 */
			p->ynm2 = p->ynm1;	/* Update Yn-2 */
			p->ynm1 = yn;		/* Update Yn-1 */
			*out++ = yn;		/* Generate the output sample */
          
			/* Handle a-rate modulation of fco and rez */
			if (p->fcocod) {
				fco = *(++fcoptr);
			}
			if (p->rezcod) {
				rez = *(++rezptr);
			}
			if (p->fcocod || p->rezcod) {
				c      = fqcadj/fco;
				rez2   = rez/(1.0f + (Number)sqrt(sqrt(1.0/((double) c))));
				tval   = 0.75f/(Number)sqrt(1.0 + (double) rez);
				csq    = c*c;
				b      = (c/rez2 + csq);
			}
		} while (--n);
	}
}

/***************************************************************************/
/* The distortion opcode uses modified hyperbolic tangent distortion.      */
/* Coded by Hans Mikelson November 1998                                    */
/***************************************************************************/

void distort(DISTORT *p)
{
	int32  n;
	Number *out, *in;
	Number pregain = *p->pregain, postgain  = *p->postgain;
	Number shape1 = *p->shape1, shape2 = *p->shape2;
	Number sig, x1, x2, x3;
    
	n    = ksmps;
	in   = p->in;
	out  = p->out;
	pregain  = pregain*.0002f;
	postgain = postgain*20000.0f;
	shape1 = shape1*.000125f;
	shape2 = shape2*.000125f;
	do {
		sig    = *in++;
		x1     =  sig*(pregain+shape1);  /* Precalculate a few values to make
						    things faster */
		x2     = -sig*(pregain+shape2);
		x3     =  sig*pregain;
      
		/* Generate tanh distortion and output the result */
		*out++ =
			(Number)((exp((double)x1)-exp((double)x2))/
				(exp((double)x3)+exp(-(double)x3))
				*postgain);
	} while (--n);
}

/***************************************************************************/
/* The vco is an analog modeling opcode                                    */
/* Generates bandlimited saw, square/PWM, triangle/Saw-Ramp-Mod            */
/* Coded by Hans Mikelson November 1998                                    */
/***************************************************************************/

void vcoset(VCO *p)
{
	uint32 ndel = (uint32)(*p->maxd * esr);  /* Number of bytes in the
						    delay*/
	Number *buf;    /* Delay buffer */
	RCPointer<FunctionTable> ftp;    /* Pointer to a sine function */
	Number ndsave;
    
	ndsave = (Number) ndel;
	p->iphs = 0;   /* Phasor */
  
	if ((ftp = ftfind(p->sine)) != 0) {
		p->ftp = ftp;
		if (p->iphs >= 0) {
		    p->lphs = (int32)(p->iphs * 0.5f * fmaxlen);
		}
		p->ampcod = p->nth_arg_is_audio (0);
		p->cpscod = p->nth_arg_is_audio (1);
	}
  
	p->ynm1 = (*p->wave == 1.0f) ? -0.5f : 0.0f;
	p->ynm2 = 0.0f;
  
	/* finished setting up buzz now set up internal vdelay */
  
	if (ndel == 0) ndel = 1;  /* fix due to Troxler */
	if (p->aux.auxp == NULL ||
	    (size_t)(ndel*sizeof(Number)) > p->aux.size) {
		p->aux.alloc (ndel * sizeof(Number));
	} else {
		buf = (Number *)p->aux.auxp;   /*    make sure buffer is empty       */
		do {
			*buf++ = 0.0f;
		} while (--ndel);
	}
	p->left = 0;
	p->leaky = (*p->wave == 3.0f) ? 0.995f : 0.999f;
    
} /* end vcoset(p) */

/* This code modified from Csound's buzz, integ, & vdelay opcodes */

void vco(VCO *p)
{
	RCPointer<FunctionTable> ftp;
	Number *ar, *ampp, *cpsp, *ftbl;
	/* Number   *del = 1.0f/(*p->xcps); */
	int32 phs, inc, lobits, dwnphs, tnp1, lenmask, nn, maxd, indx;
	Number   leaky, pw, pwn, rtfqc, amp, fqc;
	Number sicvt2, over2n, scal, num, denom, pulse=0.0f, saw=0.0f;
	Number sqr=0.0f, tri=0.0f;
	int16 n, knh;
    
	/* VDelay Inserted here */
	Number *buf = (Number *)p->aux.auxp;
	Number  fv1, fv2, out1;
	int32   v1, v2;
    
	leaky = p->leaky;
    
	if (buf==NULL) {            /* RWD fix */
		p->error ("vco: not initialized");
		return;
	}
	maxd = (uint32) (*p->maxd * esr);
	if (maxd == 0) maxd = 1; /* Degenerate case */
	nn = ksmps;
	indx = p->left;
	/* End of VDelay insert */
    
	ftp = p->ftp;
	if (ftp==NULL) {        /* RWD fix */
		p->error ("vco: not initialized");
		return;
	}
	ftbl = ftp->ftable;
	sicvt2 = sicvt * 0.5f;  /* for theta/2 */
	lobits = ftp->lobits;
	lenmask = ftp->lenmask;
	ampp = p->xamp;
	cpsp = p->xcps;
	fqc = *cpsp;
	rtfqc = (Number)sqrt(fqc);
	knh = (int)(esr/2/fqc);

	if ((n = (int)knh) <= 0) {
		n = 1;
	}

	tnp1 = (n <<1) + 1;   /* calc 2n + 1 */
	over2n = 0.5f / n;
	/*    scal = *ampp * over2n; */
	amp  = *ampp;
	scal = 1.0f * over2n;
	inc = (int32)(fqc * sicvt2);
	ar = p->ar;
	phs = p->lphs;
	nn = ksmps;

/*-----------------------------------------------------*/
/* PWM Wave                                            */
/*-----------------------------------------------------*/
	if (*p->wave==2.0f) {
		pw = *p->pw;
		pwn = pw - 0.5f;
		do {
			dwnphs = phs >> lobits;
			denom = *(ftbl + dwnphs);
			if (denom > .0002f || denom < -.0002f) {
				num = *(ftbl + (dwnphs * tnp1 & lenmask));
				pulse = (num / denom - 1.0f) * scal;
			}
			/*   else pulse = *ampp; */
			else pulse = 1.0f;
			phs += inc;
			phs &= PMASK;
			if (p->ampcod) {
				/*    scal = *(++ampp) * over2n; */
				amp  = *(++ampp);
				scal = 1.0f * over2n;
			}
			if (p->cpscod) {
				fqc = *(++cpsp);
				inc  = (int32)(fqc* sicvt2);
			}

			/* VDelay inserted here */
			buf[indx] = pulse;
			fv1 = indx - (1.0f/fqc/2.0f) * esr * pw;        /* Make sure Inside the
									   buffer      */
			while (fv1 > maxd)
			    fv1 -= maxd;
			while (fv1 < 0)
			    fv1 += maxd;
        
			if (fv1 < maxd - 1)  /* Find next sample for interpolation      */
			    fv2 = fv1 + 1;
			else
			    fv2 = 0.0f;
        
			v1 = (int32)fv1;
			v2 = (int32)fv2;
			out1 = buf[v1] + (fv1 - ( int32)fv1) * ( buf[v2] - buf[v1]);
        
			if (++indx == maxd) indx = 0;             /* Advance current pointer */
			/* End of VDelay */
        
			sqr  = pulse - out1 + leaky*p->ynm1;
			p->ynm1 = sqr;
			*ar++  = (sqr + pwn*0.5f-0.25f) * 1.5f * amp;
		}
		while (--nn);
	}
    
	/*-----------------------------------------------------*/
	/* Triangle Wave                                       */
	/*-----------------------------------------------------*/
	else if (*p->wave==3.0f) {
		pw = *p->pw;
		pwn = pw - 0.5f;
		do {
			dwnphs = phs >> lobits;
			denom = *(ftbl + dwnphs);
			if (denom > .0002f || denom < -.0002f) {
				num = *(ftbl + (dwnphs * tnp1 & lenmask));
				pulse = (num / denom - 1.0f) * scal;
			}
			/* else pulse = *ampp; */
			else pulse = 1.0f;
			phs += inc;
			phs &= PMASK;
			if (p->ampcod) {
				/* scal = *(++ampp) * over2n; */
				scal = 1.0f * over2n;
				amp = *(++ampp);
			}
			if (p->cpscod) {
				fqc = *(++cpsp);
				inc  = (int32)(fqc* sicvt2);
			}
        
			/* VDelay inserted here */
			buf[indx] = pulse;
			fv1 = indx - (1.0f/fqc/2.0f) * esr * pw;        /* Make sure Inside the
									   buffer      */
			while (fv1 > maxd)
			    fv1 -= maxd;
			while (fv1 < 0)
			    fv1 += maxd;
        
			if (fv1 < maxd - 1)  /* Find next sample for interpolation      */
			    fv2 = fv1 + 1;
			else
			    fv2 = 0.0f;
        
			v1 = (int32)fv1;
			v2 = (int32)fv2;
			out1 = buf[v1] + (fv1 - ( int32)fv1) * ( buf[v2] - buf[v1]);
        
			if (++indx == maxd) indx = 0;             /* Advance current pointer */
			/* End of VDelay */
        
			/* Integrate twice and ouput */
			sqr  = pulse - out1 + leaky*p->ynm1;
			tri  = sqr + leaky*p->ynm2;
			p->ynm1 = sqr;
			p->ynm2 = tri;
			*ar++ =  tri/600.0f/(0.1f+pw)*amp*((Number)sqrt(fqc));
		}
		while (--nn);
	}
	/*-----------------------------------------------------*/
	/* Sawtooth Wave                                       */
	/*-----------------------------------------------------*/
	else {
		do {
			dwnphs = phs >> lobits;
			denom = *(ftbl + dwnphs);
			if (denom > .0002f || denom < -.0002f) {
				num = *(ftbl + (dwnphs * tnp1 & lenmask));
				pulse = (num / denom - 1.0f) * scal;
			}
			/* else pulse = *ampp; */
			else pulse = 1.0f;
			phs += inc;
			phs &= PMASK;
			if (p->ampcod){
				/* scal = *(++ampp) * over2n; */
				scal = 1.0f * over2n;
				amp = *(++ampp);
			}
			if (p->cpscod) {
				fqc = *(++cpsp);
				inc  = (int32)(fqc*sicvt2);
			}
        
			/* Leaky Integration */
			saw  = pulse + leaky*p->ynm1;
			p->ynm1 = saw;
			*ar++ = saw*1.5f*amp;
		}
		while (--nn);
	}
    
	p->left = indx;             /*      and keep track of where you are */
	p->lphs = phs;
}

/***************************************************************************/
/* This is a simplified model of a planet orbiting in a binary star system */
/* Coded by Hans Mikelson December 1998                                    */
/***************************************************************************/

void planetset(PLANET *p)
{
	p->x  = *p->xval;  p->y  = *p->yval;  p->z  = *p->zval;
	p->vx = *p->vxval; p->vy = *p->vyval; p->vz = *p->vzval;
	p->ax = 0.0f; p->ay = 0.0f; p->az = 0.0f;
	p->hstep = *p->delta;
	p->friction = 1.0f - *p->fric/10000.0f;
    
} /* end planetset(p) */

/* Planet orbiting in a binary star system coded by Hans Mikelson */

void planet(PLANET *p)
{
	Number *outx, *outy, *outz;
	Number   sqradius1, sqradius2, radius1, radius2, fric;
	Number xxpyy, dz1, dz2, mass1, mass2, msqror1, msqror2;
	int32 nn;
    
	fric = p->friction;
    
	outx = p->outx;
	outy = p->outy;
	outz = p->outz;
    
	p->s1z = *p->sep/2.0f;
	p->s2z = -p->s1z;

	mass1 = *p->mass1;
	mass2 = *p->mass2;

	nn = ksmps;
	do {
		xxpyy = p->x * p->x + p->y * p->y;
		dz1 = p->s1z - p->z;

		/* Calculate Acceleration */
		sqradius1 = xxpyy + dz1 * dz1 + 1.0f;
		radius1 = (Number) sqrt(sqradius1);
		msqror1 = mass1/sqradius1/radius1;
      
		p->ax = msqror1 * -p->x;
		p->ay = msqror1 * -p->y;
		p->az = msqror1 * dz1;
      
		dz2 = p->s2z - p->z;
      
		/* Calculate Acceleration */
		sqradius2 = xxpyy + dz2 * dz2 + 1.0f;
		radius2 = (Number) sqrt(sqradius2);
		msqror2 = mass2/sqradius2/radius2;
      
		p->ax += msqror2 * -p->x;
		p->ay += msqror2 * -p->y;
		p->az += msqror2 * dz2;
      
		/* Update Velocity */
		p->vx = fric * p->vx + p->hstep * p->ax;
		p->vy = fric * p->vy + p->hstep * p->ay;
		p->vz = fric * p->vz + p->hstep * p->az;
      
		/* Update Position */
		p->x += p->hstep * p->vx;
		p->y += p->hstep * p->vy;
		p->z += p->hstep * p->vz;
      
		/* Output the results */
		*outx++ = p->x;
		*outy++ = p->y;
		*outz++ = p->z;
	}
	while (--nn);
}

/* ************************************************** */
/* ******** Parametric EQ *************************** */
/* ************************************************** */

/* Implementation of Zoelzer's Parmentric Equalizer Filters */
void pareqset(PAREQ *p)
{
	/* The equalizer filter is initialized to zero.    */
	p->xnm1 = p->xnm2 = p->ynm1 = p->ynm2 = 0.0f;
	p->imode = *p->mode;
} /* end pareqset(p) */


void pareq(PAREQ *p)
{
	int32 n;
	Number *out, *in;
	Number xn, yn, a0, a1, a2, b0, b1, b2;
	Number fc = *p->fc, v = *p->v, q = *p->q;
	Number omega, k, kk, vkk, vk, vkdq;
	n    = ksmps;
	in   = p->in;
	out  = p->out;

	/* Low Shelf */
	if (p->imode == 1) {
		omega = TWOPI_F*fc/esr;
		k = (Number) tan((double)omega/2.0);
		kk = k*k;
		vkk = v*kk;
		b0 =  1.0f + (Number) sqrt(2.0*(double)v)*k + vkk;
		b1 =  2.0f*(vkk - 1.0f);
		b2 =  1.0f - (Number) sqrt(2.0*(double)v)*k + vkk;
		a0 =  1.0f + k/q +kk;
		a1 =  2.0f*(kk - 1.0f);
		a2 =  1.0f - k/q + kk;
	}
	/* High Shelf */
	else if (p->imode == 2) {
		omega = TWOPI_F*fc/esr;
		k = (Number) tan((PI - (double)omega)/2.0);
		kk = k*k;
		vkk = v*kk;
		b0 =  1.0f + (Number) sqrt(2.0*(double)v)*k + vkk;
		b1 = -2.0f*(vkk - 1.0f);
		b2 =  1.0f - (Number) sqrt(2.0*(double)v)*k + vkk;
		a0 =  1.0f + k/q +kk;
		a1 = -2.0f*(kk - 1.0f);
		a2 =  1.0f - k/q + kk;
	}
	/* Peaking EQ */
	else {
		omega = TWOPI_F*fc/esr;
		k = (Number) tan((double)omega/2.0);
		kk = k*k;
		vk = v*k;
		vkdq = vk/q;
		b0 =  1.0f + vkdq + kk;
		b1 =  2.0f*(kk - 1.0f);
		b2 =  1.0f - vkdq + kk;
		a0 =  1.0f + k/q +kk;
		a1 =  2.0f*(kk - 1.0f);
		a2 =  1.0f - k/q + kk;
	}
    
	do {
		xn = *in++;
		yn = (b0*xn + b1*(p->xnm1) + b2* (p->xnm2) - a1*(p->ynm1) -
		      a2*(p->ynm2))/a0;
		p->xnm2 = p->xnm1;
		p->xnm1 = xn;
		p->ynm2 = p->ynm1;
		p->ynm1 = yn;
		*out++ = yn;
	} while (--n);
}

/* Nested all-pass filters useful for creating reverbs */
/* Coded by Hans Mikelson January 1999                 */
/* Derived from Csound's delay opcode                  */
/* Set up nested all-pass filter                       */

void nestedapset(NESTEDAP *p)
{
	int32    npts, npts1=0, npts2=0, npts3=0;
	char    *auxp;

	if (*p->istor && p->auxch.auxp != NULL)
	    return;

	npts2 = (int32)(*p->del2 * esr);
	npts3 = (int32)(*p->del3 * esr);
	npts1 = (int32)(*p->del1 * esr) - npts2 -npts3;
    
	if (((int32)(*p->del1 * esr)) <=
	    ((int32)(*p->del2 * esr) + (int32)(*p->del3 * esr))) {
		p->error ("illegal delay time");
		return;
	}
	npts = npts1 + npts2 + npts3;
	/* new space if reqd */
	if ((auxp = p->auxch.auxp) == NULL || npts != p->npts) { 
		p->auxch.alloc ((size_t) npts * sizeof(Number));
		auxp = p->auxch.auxp;
		p->npts = npts;
      
		if (*p->mode == 1.0f) {
			if (npts1 <= 0) {
				p->error ("illegal delay time");
				return;
			}
			p->beg1p = (Number *) p->auxch.auxp;
			p->end1p = (Number *) p->auxch.endp;
		}
		else if (*p->mode == 2.0f) {
			if (npts1 <= 0 || npts2 <= 0) {
				p->error ("illegal delay time");
				return;
			}
			p->beg1p = (Number *)  p->auxch.auxp;
			p->beg2p = (Number *) (p->auxch.auxp + (int32)npts1*sizeof(Number));
			p->end1p = (Number *) (p->beg2p - sizeof(Number));
			p->end2p = (Number *)  p->auxch.endp;
		}
		else if (*p->mode == 3.0f) {
			if (npts1 <= 0 || npts2 <= 0 || npts3 <= 0) {
				p->error ("illegal delay time");
				return;
			}
			p->beg1p = (Number *)  p->auxch.auxp;
			p->beg2p = (Number *) (p->auxch.auxp + (int32)npts1*sizeof(Number));
			p->beg3p = (Number *) (p->auxch.auxp +
					      (int32)npts1*sizeof(Number) +
					      (int32)npts2*sizeof(Number));
			p->end1p = (Number *) (p->beg2p - sizeof(Number));
			p->end2p = (Number *) (p->beg3p - sizeof(Number));
			p->end3p = (Number *)  p->auxch.endp;
		} 
	}
	/* else if requested */
	else if (!(*p->istor)) {
		int32 *lp = (int32 *)auxp;
		/*   clr old to zero */
		do  *lp++ = 0;
		while (--npts);
	}
	p->del1p = p->beg1p;
	p->del2p = p->beg2p;
	p->del3p = p->beg3p;
	p->out1 = 0.0f;
	p->out2 = 0.0f;
	p->out3 = 0.0f;
}

void nestedap(NESTEDAP *p)
{
	Number   *outp, *inp;
	Number   *beg1p, *beg2p, *beg3p, *end1p, *end2p, *end3p;
	Number   *del1p, *del2p, *del3p;
	Number   in1, g1, g2, g3;
	int16     nsmps = ksmps;

	if (p->auxch.auxp==NULL) { /* RWD fix */
		p->error ("delay: not initialized");
	}
    
	outp = p->out;
	inp  = p->in;
    
	/* Ordinary All-Pass Filter */
	if (*p->mode == 1.0f) {
      
		del1p = p->del1p;
		end1p = p->end1p;
		beg1p = p->beg1p;
		g1    = *p->gain1;
      
		do {
			in1 = *inp++;
			p->out1 = *del1p - g1*in1;

			/* dw1 delay dr1, dt1 */
			*del1p = in1 + g1 * p->out1;
			if (++del1p >= end1p)
			    del1p = beg1p;
        
			*outp++ = p->out1;
		} while (--nsmps);
		p->del1p = del1p;         /* save the new curp */
	}
    
	/* Single Nested All-Pass Filter */
	else if (*p->mode == 2.0f) {

		del1p = p->del1p;
		end1p = p->end1p;
		beg1p = p->beg1p;
		g1    = *p->gain1;

		del2p = p->del2p;
		end2p = p->end2p;
		beg2p = p->beg2p;
		g2    = *p->gain2;

		do {
			in1 = *inp++;

			p->out2 = *del2p  - g2 * *del1p;
			p->out1 = p->out2 - g1 * in1;

			*del1p = in1    + g1 * p->out1;
			*del2p = *del1p + g2 * p->out2;

			/* delay 2 */
			if (++del2p >= end2p)
			    del2p = beg2p;

			/* delay 1 */
			if (++del1p >= end1p)
			    del1p = beg1p;

			*outp++ = p->out1;
		} while (--nsmps);
		p->del1p = del1p;         /* save the new del1p */
		p->del2p = del2p;         /* save the new del2p */
	}

	/* Double Nested All-Pass Filter */
	else if (*p->mode == 3.0f) {

		del1p = p->del1p;
		end1p = p->end1p;
		beg1p = p->beg1p;
		g1    = *p->gain1;
      
		del2p = p->del2p;
		end2p = p->end2p;
		beg2p = p->beg2p;
		g2    = *p->gain2;
      
		del3p = p->del3p;
		end3p = p->end3p;
		beg3p = p->beg3p;
		g3    = *p->gain3;

		do {
			in1 = *inp++;

			p->out2 = *del2p  - g2 * *del1p;
			p->out3 = *del3p  - g3 * p->out2;
			p->out1 = p->out3 - g1 * in1;

			*del1p = in1      + g1 * p->out1;
			*del2p = *del1p   + g2 * p->out2;
			*del3p = p->out2  + g3 * p->out3;

			/* delay 1 */
			if (++del1p >= end1p)
			    del1p = beg1p;

			/* delay 2 */
			if (++del2p >= end2p)
			    del2p = beg2p;

			/* delay 3 */
			if (++del3p >= end3p)
			    del3p = beg3p;

			*outp++ = p->out1;
		} while (--nsmps);
		p->del1p = del1p;         /* save the new del1p */
		p->del2p = del2p;         /* save the new del2p */
		p->del3p = del3p;         /* save the new del3p */
	}
}

/***************************************************************************/
/* The Lorenz System                                                       */
/* Coded by Hans Mikelson Jauarary 1999                                    */
/***************************************************************************/

void lorenzset(LORENZ *p)
{
	p->valx = *p->inx; p->valy = *p->iny; p->valz = *p->inz;
}

/* Lorenz System coded by Hans Mikelson */

void lorenz(LORENZ *p)
{
	Number   *outx, *outy, *outz;
	Number   x, y, z, xx, yy, s, r, b, hstep;
	int32    nn, skip;

	outx  = p->outx;
	outy  = p->outy;
	outz  = p->outz;

	s     = *p->s;
	r     = *p->r;
	b     = *p->b;
	hstep = *p->hstep;
	skip  = (int32) *p->skip;
	x     = p->valx;
	y     = p->valy;
	z     = p->valz;

	nn = ksmps;
	do {
		do {
			xx   =      x+hstep*s*(y-x);
			yy   =      y+hstep*(-x*z+r*x-y);
			z    =      z+hstep*(x*y-b*z);
			x    =      xx;
			y    =      yy;
		} while (--skip>0);

		/* Output the results */
		*outx++ = x;
		*outy++ = y;
		*outz++ = z;
	}
	while (--nn);

	p->valx = x;
	p->valy = y;
	p->valz = z;
}

Opcode opcodes[] =
{
	BIQUAD_OPCODE_LIST,
	{NULL}
};

