// ------------------------------------------------------------------------
// audiofx_amplitude.cpp: Amplitude effects and dynamic processors.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>

#include <kvutils.h>

#include "samplebuffer_iterators.h"
#include "audiofx_amplitude.h"

#include "eca-debug.h"
#include "eca-error.h"

EFFECT_AMPLIFY::EFFECT_AMPLIFY (parameter_type multiplier_percent = 100.0) {
  set_parameter(1, multiplier_percent);
}

void EFFECT_AMPLIFY::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    kerroin = value / 100.0;
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_AMPLIFY::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(kerroin * 100.0);
  }
  return(0.0);
}

void EFFECT_AMPLIFY::init(SAMPLE_BUFFER* sbuf) { i.init(sbuf); }

void EFFECT_AMPLIFY::process(void) {
  i.begin();
  while(!i.end()) {
    *i.current() = *i.current() *  kerroin;
    i.next();
  }
}

EFFECT_AMPLIFY_CLIPCOUNT::EFFECT_AMPLIFY_CLIPCOUNT (parameter_type multiplier_percent, int max_clipped) {
  set_parameter(1, multiplier_percent);
  set_parameter(2, max_clipped);
  num_of_clipped = 0;
}

void EFFECT_AMPLIFY_CLIPCOUNT::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    kerroin = value / 100.0;
    break;
  case 2:
    maxnum_of_clipped = (int)value;
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_AMPLIFY_CLIPCOUNT::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(kerroin * 100.0);
  case 2:
    return(maxnum_of_clipped);
  }
  return(0.0);
}

void EFFECT_AMPLIFY_CLIPCOUNT::init(SAMPLE_BUFFER* sbuf) { i.init(sbuf); }

void EFFECT_AMPLIFY_CLIPCOUNT::process(void) {
  i.begin();
  while(!i.end()) {
    *i.current() = *i.current() *  kerroin;
    if (*i.current() > SAMPLE_BUFFER::impl_max_value ||
	*i.current() < SAMPLE_BUFFER::impl_min_value) {
      num_of_clipped++;
    }
    else {
      num_of_clipped = 0;
    }
    i.next();
  }

  if (num_of_clipped > maxnum_of_clipped && maxnum_of_clipped != 0) {
    MESSAGE_ITEM otemp;
    otemp.setprecision(0);
    otemp << "(audiofx_amplitude) WARNING! Signal is clipping! ";
    otemp << num_of_clipped;
    otemp << " consecutive clipped samples.";
    ecadebug->msg(otemp.to_string());
  }
}

EFFECT_AMPLIFY_CHANNEL::EFFECT_AMPLIFY_CHANNEL (parameter_type multiplier_percent, int channel) {
  set_parameter(1, multiplier_percent);
  set_parameter(2, channel);
}

void EFFECT_AMPLIFY_CHANNEL::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    kerroin = value / 100.0;
    break;

 case 2: 
    channel_rep = static_cast<int>(value);
    assert(channel_rep > 0);
    channel_rep--;
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_AMPLIFY_CHANNEL::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(kerroin * 100.0);

  case 2: 
    return(static_cast<parameter_type>(channel_rep + 1));
  }
  return(0.0);
}

void EFFECT_AMPLIFY_CHANNEL::init(SAMPLE_BUFFER *insample) { i.init(insample); }

void EFFECT_AMPLIFY_CHANNEL::process(void) {
  i.begin(channel_rep);
  while(!i.end()) {
    *i.current() = *i.current() * kerroin;
    i.next();
  }
}

EFFECT_COMPRESS::EFFECT_COMPRESS (parameter_type compress_rate, parameter_type thold) {
  // map_parameters();

  set_parameter(1, compress_rate);
  set_parameter(2, thold);

  first_time = true;
}

EFFECT_COMPRESS::EFFECT_COMPRESS (const EFFECT_COMPRESS& x) {
  crate = x.crate;
  threshold = x.threshold;
  delta = x.delta;
  first_time = x.first_time;

  s = x.s;
  temp = x.temp;
  lastin = x.lastin;
  lastout = x.lastout;
}

void EFFECT_COMPRESS::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    crate = pow(2.0, value / 3.0);
    break;
  case 2: 
    threshold = value;
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_COMPRESS::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return((log (crate)) / (log (2)) * 3.0);
  case 2: 
    return(threshold);
  }
  return(0.0);
}

void EFFECT_COMPRESS::init(SAMPLE_BUFFER *insample) { 
  i.init(insample);
  
  s.resize(insample->number_of_channels());
  temp.resize(insample->number_of_channels());
  lastin.resize(insample->number_of_channels());
  lastout.resize(insample->number_of_channels());
} 

void EFFECT_COMPRESS::process(void) {
  i.begin();
  while(!i.end()) {
    if (first_time) {
      first_time = false;
      lastin[i.channel()] = lastout[i.channel()] = *i.current();
    }
    else {
      delta = *i.current() / lastin[i.channel()];
      
      if (fabs(*i.current()) > threshold) {
	if (delta > 1.0) {
	  delta = (delta - 1.0) / crate;
	  delta += 1.0;
	}
	new_value = lastout[i.channel()] * delta;
      }
      else
	new_value = lastin[i.channel()] * delta;
      
      if (new_value > SAMPLE_BUFFER::impl_max_value) new_value = SAMPLE_BUFFER::impl_max_value;
      else if (new_value < SAMPLE_BUFFER::impl_min_value) new_value = SAMPLE_BUFFER::impl_min_value;
      temp[i.channel()] = new_value;
    }
    lastin[i.channel()] = *i.current();
    lastout[i.channel()] = temp[i.channel()];
    *i.current() = temp[i.channel()];
    i.next();
  }
}

EFFECT_NOISEGATE_MONO::EFFECT_NOISEGATE_MONO (parameter_type thlevel_percent, parameter_type thtime, parameter_type a, parameter_type h, parameter_type r) {
  // map_parameters();

  set_parameter(1, thlevel_percent);
  set_parameter(2, thtime);
  set_parameter(3, a);
  set_parameter(4, h);
  set_parameter(5, r);
}

void EFFECT_NOISEGATE_MONO::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    th_level = SAMPLE_BUFFER::max_amplitude * (value / 100.0);
    break;
  case 2: 
    th_time = (value * (parameter_type)SAMPLE_BUFFER::sample_rate / 1000.0);
    break;
  case 3: 
    atime = (value * (parameter_type)SAMPLE_BUFFER::sample_rate / 1000.0);
    break;
  case 4: 
    htime = (value * (parameter_type)SAMPLE_BUFFER::sample_rate / 1000.0);
    break;
  case 5: 
    rtime = (value * (parameter_type)SAMPLE_BUFFER::sample_rate / 1000.0);
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_NOISEGATE_MONO::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(th_level * 100.0 / (parameter_type)SAMPLE_BUFFER::max_amplitude);
  case 2: 
    return(th_time * 1000.0 / (parameter_type)SAMPLE_BUFFER::sample_rate);
  case 3: 
    return(atime * 1000.0 / (parameter_type)SAMPLE_BUFFER::sample_rate);
  case 4: 
    return(htime * 1000.0 / (parameter_type)SAMPLE_BUFFER::sample_rate);
  case 5: 
    return(rtime * 1000.0 / (parameter_type)SAMPLE_BUFFER::sample_rate);
  }
  return(0.0);
}

void EFFECT_NOISEGATE_MONO::init(SAMPLE_BUFFER *insample) {
  i.init(insample);

  th_time_lask.resize(insample->number_of_channels());
  attack_lask.resize(insample->number_of_channels());
  hold_lask.resize(insample->number_of_channels());
  release_lask.resize(insample->number_of_channels());
  kerroin.resize(insample->number_of_channels());

  ng_status.resize(insample->number_of_channels(), int(ng_waiting));
}

void EFFECT_NOISEGATE_MONO::process(void) {
  i.begin();
  while(!i.end()) {
    if (fabs(*i.current()) <= th_level) {
      if (ng_status[i.channel()] == ng_holding || ng_status[i.channel()] == ng_releasing) {
	ng_status[i.channel()] = ng_waiting; 
	ecadebug->msg(10,"(audiofx) ng_mono waiting (again)");
      }

      if (ng_status[i.channel()] == ng_active) {
	*i.current() = 0.0;
	return;
      }
      
      if (ng_status[i.channel()] == ng_attacking) {
	attack_lask[i.channel()]++;
	kerroin[i.channel()] = (1 - (attack_lask[i.channel()] / atime));
	if (attack_lask[i.channel()] >= atime) {
	  ng_status[i.channel()] = ng_active;
	  ecadebug->msg(10,"(audiofx) ng_mono active");
	  attack_lask[i.channel()] = 0.0;
	}
      }
      
      if (ng_status[i.channel()] == ng_waiting) {
	th_time_lask[i.channel()]++;
	if (th_time_lask[i.channel()] >= th_time) {
	  ng_status[i.channel()] = ng_attacking;
	  ecadebug->msg(10,"(audiofx) ng_mono attacking");
	  th_time_lask[i.channel()] = 0.0;
	}
	return;
      }
    }
    else {
      if (ng_status[i.channel()] == ng_releasing) {
	release_lask[i.channel()]++;
	kerroin[i.channel()] = release_lask[i.channel()] / rtime;
	if (release_lask[i.channel()] >= rtime) {
	  ecadebug->msg(10,"(audiofx) ng_mono waiting");
	  ng_status[i.channel()] = ng_waiting;
	  release_lask[i.channel()] = 0.0;
	}
      }
      
      if (ng_status[i.channel()] == ng_holding) {
	hold_lask[i.channel()]++;
	if (hold_lask[i.channel()] >= htime) {
	  ng_status[i.channel()] = ng_releasing;
	  ecadebug->msg(10,"(audiofx) ng_mono releasing");
	  hold_lask[i.channel()] = 0.0;
	}
	*i.current() = 0.0;
	return;
      }

      if (ng_status[i.channel()] == ng_active) {
	ng_status[i.channel()] = ng_holding;
	ecadebug->msg(10,"(audiofx) ng_mono holding");
	return;
      }
      
      if (ng_status[i.channel()] == ng_waiting
	  || ng_status[i.channel()] == ng_attacking) {
	ng_status[i.channel()] = ng_waiting; 
	th_time_lask[i.channel()] = 0.0;
	//      ecadebug->msg(10,"(audiofx) ng_mono waiting (reseted)");
	return;
      }
    }

    *i.current() = *i.current() * kerroin[i.channel()];
    i.next();
  }
}

EFFECT_NORMAL_PAN::EFFECT_NORMAL_PAN (parameter_type right_percent) {
  set_parameter(1, right_percent);
}

void EFFECT_NORMAL_PAN::set_parameter(int param, parameter_type value) {
  switch (param) {
  case 1: 
    right_percent_rep = value;
    if (value == 50.0) {
      l_kerroin = r_kerroin = 1.0;
    }
    else if (value < 50.0) {
      l_kerroin = 1.0;
      r_kerroin = value / 50.0;
    }
    else if (value > 50.0) {
      r_kerroin = 1.0;
      l_kerroin = (100.0 - value) / 50.0;
    }
    break;
  }
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_NORMAL_PAN::get_parameter(int param) const { 
  switch (param) {
  case 1: 
    return(right_percent_rep);
  }
  return(0.0);
}

void EFFECT_NORMAL_PAN::init(SAMPLE_BUFFER *insample) { i.init(insample); }

void EFFECT_NORMAL_PAN::process(void) {
  i.begin(0);
  while(!i.end()) {
    *i.current() = *i.current() * l_kerroin;
    i.next();
  }

  i.begin(1);
  while(!i.end()) {
    *i.current() = *i.current() * r_kerroin;
    i.next();
  }
}
