// ------------------------------------------------------------------------
// eca-fileio-mmap.cpp: mmap based file-I/O and buffering routines.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cstdio>
#include <cstring>
#include <unistd.h>
#include <fcntl.h>
#include <errno.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <sys/types.h>

#include "eca-error.h"
#include "eca-fileio.h"
#include "eca-fileio-mmap.h"
#include "eca-fileio-mmap-fthread.h"

ECA_FILE_IO_MMAP::~ECA_FILE_IO_MMAP(void) {
//    if (mmap_low != -1) {
//      munlock(internal_buffer, internal_bsize);
//      munmap(internal_buffer, internal_bsize);
//    }
}

void ECA_FILE_IO_MMAP::open_file(const string& fname, 
			    const string& fmode,
			    bool handle_errors)
{ 
  if (fmode == "rb") {
    f1 = open(fname.c_str(), O_RDWR);
    if (!f1) {
      if (handle_errors) {
	throw(new ECA_ERROR("ECA-FILEIO", "unable to open file " + fname +
			    " in mode \"" + fmode + "\".", ECA_ERROR::retry));
      }
      mode_rep = "";
    }
    else {
      file_ready = true;
      file_ended = false;
      mode_rep = fmode;
      fposition = 0;
      flength = get_file_length();
      mmap_high = 0;
      mmap_low = -1;
      internal_bsize = ecasound_fiommap_register_fd(f1);
    }
  }
  else {
    if (handle_errors) {
      throw(new ECA_ERROR("ECA-FILEIO", "unable to open file " + fname +
			    " in mode \"" + fmode + "\".", ECA_ERROR::retry));
    }
  }
}

void ECA_FILE_IO_MMAP::close_file(void) { 
  ecasound_fiommap_close_fd(f1);
  close(f1);
}

void ECA_FILE_IO_MMAP::read_to_buffer_old(void* obuf, long int bytes) { 
//    if (mmap_high <= fposition) {
//      if (mmap_low != -1) {
//              munlock(internal_buffer, internal_bsize);
//              munmap(internal_buffer, internal_bsize);
//      }
//      while (fposition >= mmap_high) mmap_high += internal_bsize;
//      mmap_low = mmap_high - internal_bsize;
//      cerr << "(eca-fileio-mmap) mapping from " << mmap_low 
//           << " to " << mmap_high << ".\n";
//      internal_buffer = (unsigned char*)mmap(0, internal_bsize, 
//  					   PROT_READ,
//  					   MAP_PRIVATE,
//  					   f1, mmap_low);
//      mlock(internal_buffer, internal_bsize);
//    }
//    else if (fposition < mmap_low) {
//      munlock(internal_buffer, internal_bsize);
//      munmap(internal_buffer, internal_bsize);
//      while (fposition < mmap_low) mmap_low -= internal_bsize;
//      mmap_high = mmap_low + internal_bsize;
//      cerr << "(eca-fileio-mmap) mapping[2] from " << mmap_low 
//           << " to " << mmap_high << ".\n";
//      internal_buffer = (unsigned char*)mmap(0, internal_bsize, 
//  					   PROT_READ,
//  					   MAP_PRIVATE,
//  					   f1, mmap_low);
//      mlock(internal_buffer, internal_bsize);
//    }
  
//    if (internal_buffer == MAP_FAILED) {
//      file_ready = false;
//      cerr << "(eca-fileio-mmap) mmap() failed! - errno:" << errno <<
//        ".\n";
//      perror(0);
//    }
//    else {
//      internal_bindex = fposition % internal_bsize;

//      if (internal_bindex + bytes > internal_bsize) 
//        bytes = internal_bsize - internal_bindex;
//      if (fposition + bytes > flength) 
//        bytes =  flength - fposition;

//      memcpy(obuf, internal_buffer + internal_bindex, bytes);

//      bytes_rep = bytes;
//      set_file_position(fposition + bytes);
//    }
}

void ECA_FILE_IO_MMAP::read_to_buffer(void* obuf, long int bytes) {
  internal_buffer = ecasound_fiommap_active_buffer(f1);

  if (internal_buffer == MAP_FAILED) {
    file_ready = false;
    cerr << "(eca-fileio-mmap) mmap() failed! - errno:" << errno << ".\n";
    perror(0);
  }
  else {
    internal_bindex = fposition % internal_bsize;

    if (internal_bindex + bytes >= internal_bsize) {
      bytes = internal_bsize - internal_bindex;
      switch_buffer = true;
    }
    if (fposition + bytes > flength) {
      bytes =  flength - fposition;
      switch_buffer = true;
    }

    memcpy(obuf, internal_buffer + internal_bindex, bytes);

    bytes_rep = bytes;
    set_file_position(fposition + bytes, false);

    if (switch_buffer == true) {
      ecasound_fiommap_next_buffer(f1);
      switch_buffer = false;
    }
  }
}

void ECA_FILE_IO_MMAP::write_from_buffer(void* obuf, long int
				   bytes) { 
  //  bytes_rep = fwrite(obuf, 1, bytes, f1);
}

long int ECA_FILE_IO_MMAP::file_bytes_processed(void) { return(bytes_rep); }

bool ECA_FILE_IO_MMAP::is_file_ready(void) { return(file_ready); }
bool ECA_FILE_IO_MMAP::is_file_ended(void) { return(file_ended); }
bool ECA_FILE_IO_MMAP::is_file_error(void) { return(!file_ready && !file_ended); }

void ECA_FILE_IO_MMAP::set_file_position(long int newpos, bool seek) { 
  fposition = newpos;
  if (fposition >= flength) {
    file_ready = false;
    file_ended = true;
  }
  else {
    file_ready = true;
    file_ended = false;
    if (seek) ecasound_fiommap_reset(f1, fposition);
  }
}

void ECA_FILE_IO_MMAP::set_file_position_advance(long int fw) { 
  set_file_position(fposition + fw, false);
}

void ECA_FILE_IO_MMAP::set_file_position_end(void) { 
  fposition = get_file_length();
}

long int ECA_FILE_IO_MMAP::get_file_position(void) { return(fposition); }

long int ECA_FILE_IO_MMAP::get_file_length(void) {
  struct stat stattemp;
  fstat(f1, &stattemp);
  return((long int)stattemp.st_size);
}

