#ifndef _ECA_CONTROLLER_H
#define _ECA_CONTROLLER_H

#include <pthread.h>

#include "eca-session.h"
#include "eca-chainop.h"

extern string ecasound_lockfile;

enum {
  ECA_QUIT = 1
};

class ECA_CONTROLLER {

  ECA_SESSION* session; 
  int retcode;
  pthread_t th_cqueue;
  bool show_prompt;
  bool engine_started;

 public:

  enum DIRECTION { input, output };

  // ---
  // Status
  // ---
  bool prompt(void);

  // ---
  // CLI
  // ---
  void command(const string& cmd);

  // ---
  // Runtime control
  // ---
  void start(void);
  void stop(void);
  void close_engine(void);
  void start_engine(bool ignore_lock = false);
  void quit(void);

  // ---
  // Session info / global resources (~/.ecasoundrc)
  // ---

  const string& resource_value(const string& key) { return session->ecaresources.resource(key); }

  // ---
  // Session info / output to ecadebug
  // ---
  void print_general_status(void);
  void print_chain_status(void);
  void print_effect_status(void);
  void print_file_status(void);

  // ---
  // Session info / functions
  // ---
  string connected_chains_input(CHAIN::aio_id_type aiod);
  string connected_chains_output(CHAIN::aio_id_type aiod);
  vector<string> connected_chains(const string& filename);

  // ---
  // Session info / master input
  // ---
  long length_in_samples(void) const { return session->length_in_samples(); }
  double length_in_seconds_exact(void) const { return session->length_in_seconds_exact(); }
  long position_in_samples(void) const { return session->position_in_samples(); }
  double position_in_seconds_exact(void) const { return session->position_in_seconds_exact(); }

  // ---
  // Modify session
  // ---
  void enable_mthreaded_use(void) { session->mthreaded_use_of_session = true; } 
  void toggle_mthreaded_use(bool v) { session->mthreaded_use_of_session = v; } 
  void toggle_interactive_mode(bool v) { session->iactive = v; } 
  void toggle_sfx(bool v) { session->sfx = v; } 
  void toggle_looping(bool v) { session->loop_active = v; } 
  void toggle_multitrack_mode(bool v) { session->multitrack_mode = v; } 

  void set_buffersize(int bsize);
  void toggle_raise_priority(bool v);

  // ---
  // Session status
  // ---
  bool is_running(void);
  string engine_status(void);
  bool is_engine_ready(void) const;

  // ---
  // Chainsetups  (if not specified, active chainsetup is used)
  // ---
  void activate_chainsetup(const string& name, bool handle_errors = true);
  string active_chainsetup(void);
  void deactivate_chainsetup(const string& name, bool handle_errors = true);
  void connect_active_chainsetup(void);
  string connected_chainsetup(void);
  void disconnect_active_chainsetup(void);

  void new_chainsetup(const string& name);
  void remove_chainsetup(const string& name, bool handle_errors = true);
  const ECA_CHAINSETUP* get_chainsetup(const string& name);
  void load_chainsetup(const string& name, bool handle_errors = true);
  void save_chainsetup(const string& csetup, const string filename, bool handle_errors = true);
  void save_active_chainsetup(const string& filename, bool handle_errors = true);

  // ---
  // Chains (if not specified, active chainsetup is used)
  // ---
  void add_chain(const string& name);
  void remove_chain(const string& name);
  void set_active_chains(const vector<string>& chains);

  // ---
  // Audio-devices  (if not specified, active chainsetup is used)
  // ---
  void set_audio_format(int bits, int channels, long int srate);
  void set_audio_format(const AIO_PARAMS* format);
  void get_audio_format(const string& name, AIO_PARAMS* format);
  void set_explicit_format(bool enabled);
  void set_io_mode(SIMODE iomode);
  void add_audio_device(const string& filename, DIRECTION dir);
  AUDIO_IO_DEVICE* get_audio_device(const string& csname, const string& name);
  void add_default_output(void);
  void remove_audio_device(const string& filename);
  void attach_iodev_to_active_chains(const string& filename);

  // ---
  // Chain operators  (if not specified, active chainsetup is used)
  // ---
  void add_chain_operator(const string& chainop_params);
  void add_chain_operator(CHAIN_OPERATOR* cotmp);
  const CHAIN_OPERATOR* get_chain_operator(const string& chain, int chainop_id);
  void remove_chain_operator(const string& chain, int chainop_id);
  void add_general_controller(const string& gcontrol_params);

  ECA_CONTROLLER (ECA_SESSION* psession);
};

void show_controller_help(void);
void show_controller_help_more(void);

void start_normal_thread(ECA_SESSION* param, int retcode, pthread_t* th_cqueue);
void* start_normal(void* param);
void start_normal(ECA_SESSION* param);

#endif

