#ifndef _SAMPLEBUFFER_H
#define _SAMPLEBUFFER_H

#include <config.h>

// #include <sys/soundcard.h>
#include <vector>

#define SBTYPE_DOUBLE        0
#define SBTYPE_SSINT         1
#define SBTYPE_UCHAR         2
#define SBTYPE_SWAPPED_SSINT 3

typedef struct {
   public:
   signed short int sample[2];
} SAMPLE;

class SAMPLE_SPECS {
 public:

  typedef float single_type;   // should be a floating-point value!

  static const char ch_count = 2;
  static const char ch_left = 0;
  static const char ch_right = 1;
  
  static const single_type silent_value = 0.0;     // do not change!
  static const single_type max_amplitude = 1.0;
  static const single_type max_value = silent_value + max_amplitude;
  static const single_type min_value = silent_value - max_amplitude;
};


class SINGLE_SAMPLE {
  // --
  // Represents one sample. Sample type and related paramters are set in 
  // SAMPLE_SPECS class declaration. Channel count is currently hard
  // coded. 
  // --

public:

    SAMPLE_SPECS::single_type sample[2];

public:

    inline void put_left(const SAMPLE_SPECS::single_type in) { sample[SAMPLE_SPECS::ch_left] = in; }
    inline void put_right(const SAMPLE_SPECS::single_type in) { sample[SAMPLE_SPECS::ch_right] = in; }
    inline void put_channel(unsigned short int channel , const SAMPLE_SPECS::single_type in) { sample[channel] = in; }
    
    inline SAMPLE_SPECS::single_type get_left(void) const { return(sample[SAMPLE_SPECS::ch_left]); }
    inline SAMPLE_SPECS::single_type get_right(void) const { return(sample[SAMPLE_SPECS::ch_right]); }
    inline SAMPLE_SPECS::single_type get_channel(unsigned short int channel ) const { return(sample[channel]); }

    inline void mix (SINGLE_SAMPLE& x, float kerroin) {
        // --
        // Mixes ('kerroin' + 100)% of 'x' into current object.
        // --
        sample[SAMPLE_SPECS::ch_left] = (sample[SAMPLE_SPECS::ch_left] * (1.0 - kerroin) + x.sample[SAMPLE_SPECS::ch_left] * kerroin);
        sample[SAMPLE_SPECS::ch_right] = (sample[SAMPLE_SPECS::ch_right] * (1.0 - kerroin) + x.sample[SAMPLE_SPECS::ch_right] * kerroin);
    }

    inline void swap_channels(void) { SAMPLE_SPECS::single_type temp = sample[SAMPLE_SPECS::ch_left]; sample[SAMPLE_SPECS::ch_left] = sample[SAMPLE_SPECS::ch_right]; sample[SAMPLE_SPECS::ch_right] = temp; }
    inline void sum_to_mono(void) { sample[SAMPLE_SPECS::ch_right] = sample[SAMPLE_SPECS::ch_left] = ((sample[SAMPLE_SPECS::ch_left] + sample[SAMPLE_SPECS::ch_right]) / 2.0); }
    
    inline bool is_clipped(void) {
      if (sample[SAMPLE_SPECS::ch_left] > SAMPLE_SPECS::max_value ||
	  sample[SAMPLE_SPECS::ch_right] > SAMPLE_SPECS::max_value) return true;
      if (sample[SAMPLE_SPECS::ch_left] < SAMPLE_SPECS::min_value ||
	  sample[SAMPLE_SPECS::ch_right] < SAMPLE_SPECS::min_value) return true;
      return false;
    }

    inline void limit_values(void) {
      if (sample[SAMPLE_SPECS::ch_left] > SAMPLE_SPECS::max_value) sample[SAMPLE_SPECS::ch_left] = SAMPLE_SPECS::max_value;
      else if (sample[SAMPLE_SPECS::ch_left] < SAMPLE_SPECS::min_value) sample[SAMPLE_SPECS::ch_left] = SAMPLE_SPECS::min_value;
      if (sample[SAMPLE_SPECS::ch_right] > SAMPLE_SPECS::max_value) sample[SAMPLE_SPECS::ch_right] = SAMPLE_SPECS::max_value;
      else if (sample[SAMPLE_SPECS::ch_right] < SAMPLE_SPECS::min_value) sample[SAMPLE_SPECS::ch_right] = SAMPLE_SPECS::min_value;
    }

    inline void make_silent(void) { sample[SAMPLE_SPECS::ch_left] = sample[SAMPLE_SPECS::ch_right] = SAMPLE_SPECS::silent_value; }
    inline void multiply(double x) { 
      sample[SAMPLE_SPECS::ch_left] *= x;
      sample[SAMPLE_SPECS::ch_right] *= x;
    }

    SINGLE_SAMPLE& operator+(const SINGLE_SAMPLE& x) { 
      sample[SAMPLE_SPECS::ch_left] += x.sample[SAMPLE_SPECS::ch_left];
      sample[SAMPLE_SPECS::ch_right] += x.sample[SAMPLE_SPECS::ch_right];
      return *this;
    }
    SINGLE_SAMPLE& operator=(const SINGLE_SAMPLE& x) {
        if (this != &x) {
	  sample[SAMPLE_SPECS::ch_left] = x.sample[SAMPLE_SPECS::ch_left];
	  sample[SAMPLE_SPECS::ch_right] = x.sample[SAMPLE_SPECS::ch_right];
        }
	return *this;
    }

    SINGLE_SAMPLE (void) { make_silent(); }
};

class SAMPLE_BUFFER {
  //
  // Represents a buffer of samples. The primary goal of this class is to 
  // hide the actual type information, but still provide a reasonably
  // efficient implementation.
  //

 public:

  static const unsigned int long sample_rate = 44100;
 
  typedef SINGLE_SAMPLE sample_type;
  typedef SAMPLE_SPECS::single_type single_type;

  static const char ch_count = SAMPLE_SPECS::ch_count;
  static const char ch_left = SAMPLE_SPECS::ch_left;
  static const char ch_right = SAMPLE_SPECS::ch_right;
  static const single_type silent_value = SAMPLE_SPECS::silent_value;
  static const single_type max_amplitude = SAMPLE_SPECS::max_amplitude;

#ifdef WORDS_BIGENDIAN
  static const bool is_system_littleendian = false;
#else
  static const bool is_system_littleendian = true;
#endif
  static const single_type s16_to_st_constant = (max_amplitude / 32767.0);
  static const single_type u8_to_st_delta = 128; 
  static const single_type u8_to_st_constant = (max_amplitude / 128.0);
  
    
 private:
    
    vector<sample_type> buffer;
    
    int index;             // index of the current sample
    int iobuf_size;         // size of io-buffer

    bool validindex;       // is index value usable

    unsigned short int channel_sizet; // type used to subscript channels
    vector<sample_type>::const_iterator buf_citer; // buffer const iterator
    vector<sample_type>::iterator buf_iter; // buffer iterator
    size_t ubuf_sizet;   // type used to subscript channels
    vector<sample_type>::size_type buf_sizet;   // type used to subscript channels
    
    SAMPLE_SPECS::single_type temp_avg;     // temporary value of type 'SAMPLE_SPECS::single_type'
    signed short int s16temp;  // temporary pointer value of type s16
    unsigned char ps16temp[2];  // temporary value of type s16

    void resize(int buffersize);

    vector<sample_type> old_buffer;
    double step;
    double counter;
    size_t last_sizet;
    size_t newbuf_sizet;

    void resample_extfilter(unsigned int long from_srate,
			    unsigned int long to_srate);
    void resample_simplefilter(unsigned int long from_srate,
			       unsigned int long to_srate);
    void resample_nofilter(unsigned int long from_srate,
			   unsigned int long to_srate);


public:
    
// ---
// Operations that modify the current object:
// ---

    // --- for debugging signal flow
    //    size_t nro;
    // -----------------------------

    unsigned short swapword(unsigned short us) {
        return ((us >> 8) | (us << 8)) & 0xffff;
    }

    void sb_swap_bytes(void) { }
    void limit_values(void);
    void make_silent(void);
    void divide_by(double dvalue);
    void add(const SAMPLE_BUFFER& x);
    void add_with_weight(const SAMPLE_BUFFER& x, int weight);

    void resample_from(unsigned int long from_srate);
    void resample_to(unsigned int long to_srate);

// ---
// Direct access pointers
// ---

    void reserve_buffer(const int len);
    unsigned char* iobuf_uchar;  // buffer for IO-operations
    void iobuf_to_buffer(int samples, bool iobuf_little_endian, int
			 bits, unsigned short int channels, unsigned
			 int long srate = SAMPLE_BUFFER::sample_rate);
    void buffer_to_iobuf(bool iobuf_little_endian, int
			 bits, unsigned short int channels, unsigned 
			 int long srate = SAMPLE_BUFFER::sample_rate);

    inline sample_type* current_sample(void) { return(&buffer[index]); }
    
// ---
// Iterator functions.
// ---
    
    inline bool is_readable(void) { return(validindex); }  // can you use iterators?
    inline void first(void) { index = 0; if (buffer.size() > 0) validindex = true; }
    void previous(void);         // if object is copied or cloned.
    inline void next(void) {
        ++index;
        if (index >= (int)buffer.size()) validindex = false;
	//        validindex = ( index >= bsize_in_samples ) ? false : true;
    }
    void last(void);    
            
// ---
// Info about the audio content 
// ---

    single_type average_volume(void);
    single_type average_RMS_volume(void);
    single_type average_volume(char channel, int count_samples = 0);
    single_type average_RMS_volume(char channel, int count_samples = 0);
    single_type max_value(char channel);
    single_type min_value(char channel);
    single_type amplitude(void);

// ---
// Buffer setup
// ---

 public:

    void length_in_samples(const int len);
    inline int length_in_samples(void) const { return(buffer.size()); }
    inline double length_in_seconds(void) const {
      return((double)buffer.size() / sample_rate);
    }

    inline int element_size(void) const { return(sizeof(SAMPLE_SPECS::SAMPLE_SPECS::single_type) * 2); }

// ---
// Buffer i/o
// ---
    
    inline double get_double(unsigned short int channel ) { return(buffer[index].sample[channel]); }
    inline SAMPLE_SPECS::single_type get(unsigned short int channel ) { return(buffer[index].sample[channel]); }
    inline void put(unsigned short int channel , SAMPLE_SPECS::single_type value) { buffer[index].sample[channel] = value; }

// ---
// Constructors/destructors
// ---

    SAMPLE_BUFFER& operator= (const SAMPLE_BUFFER& t);
    SAMPLE_BUFFER (int buffersize);
    ~SAMPLE_BUFFER (void);
    SAMPLE_BUFFER (const SAMPLE_BUFFER& x);
};

#endif



