#include <cstdio>
#include <cmath>
#include <fcntl.h>
#include <unistd.h>
#include <sys/time.h>
#include <pthread.h>
#include <vector>

#include "eca-resources.h"
#include "eca-midi.h"
#include "eca-debug.h"
#include "eca-error.h"

MIDI_IN_QUEUE midi_in_queue;
// MIDI_OUT_QUEUE midi_out_queue;
pthread_mutex_t midi_in_lock;     // mutex ensuring exclusive access to MIDI-buffer

MIDI_IN_QUEUE::MIDI_IN_QUEUE(void) {
    right = false;

    current_put = 0;
    current_get = 0;

    bufsize = MIDI_IN_QUEUE_SIZE;

    controller_value = 0.0;

    buffer = vector<char> (bufsize, char(0));
}

bool MIDI_IN_QUEUE::is_status_byte(char byte) {
    if ((byte & 128) == 128) return(true);
    else return(false);
}

void MIDI_IN_QUEUE::put(char byte) {
    buffer[current_put] = byte;
    current_put++;
    if (current_put == bufsize) current_put = 0;
}

double MIDI_IN_QUEUE::get_controller_value(double controller, double channel) {
    for(current_get = current_put;;) {
        if (is_status_byte(buffer[current_get]) == true) {
            if ((buffer[current_get] & (15 << 4)) ^ (11 << 4)) right = false; // non cc-byte
            else if ((buffer[current_get] & 15) != channel) right = false; // wrong channel
            else {
                right = true;
            }

            if (right == false) {
                if (!forth_get()) break;
                continue;
            }

            // first data-byte
            if (!forth_get()) break;
            if (buffer[current_get] != controller) {
                if (!forth_get()) break;
                if (!forth_get()) break;
                continue;                
            }

            // second data-byte
            if (!forth_get()) break;

            if (is_status_byte(buffer[current_get])) continue;
            controller_value = (double)buffer[current_get];
        }
        else if (right == true) {
            if (buffer[current_get] != controller) {
                if (!forth_get()) break;
                if (!forth_get()) break;
                continue;                
            }                        
            if (!forth_get()) break;

            if (is_status_byte(buffer[current_get])) continue;
            
            controller_value = (double)buffer[current_get];
        }
        if (!forth_get()) break;
    }
    return(controller_value);
}

bool MIDI_IN_QUEUE::forth_get(void) {
    current_get++;
    if (current_get == current_put - 1) return(false);
    else if (current_put == 0 && current_get == bufsize) return(false);
    else if (current_put == 1 && current_get == 1) return(false);
    if (current_get == bufsize) current_get = 0;
    return(true);
};

void init_midi_queues(void) {
  static bool ready = false;

  if (ready == true) return; 
  else ready = true;

  pthread_mutex_init(&midi_in_lock, NULL);
  pthread_t th_midi;
  int retcode = pthread_create(&th_midi, NULL, update_midi_queues, NULL);
  if (retcode != 0)
    throw(new ECA_ERROR("ECA-MIDI", "unable to create MIDI-thread"));
}

void *update_midi_queues(void *) {
  fd_set rfds;
  struct timeval tv;
  int retval, fd;
  char buf[MIDI_IN_QUEUE_SIZE];
  size_t temp;
  
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);  // other threads can cancel this one
    
  tv.tv_sec = 0;
  tv.tv_usec = 0;

  ECA_RESOURCES erc;
  erc.load();
    
  //  if ((fd = open("/dev/midi00", O_RDONLY)) == -1) {
  //  if ((fd = open("/dev/midi", O_RDONLY)) == -1) {
  if ((fd = open(erc.resource("midi-device").c_str(), O_RDONLY)) == -1) {
    throw(new ECA_ERROR("ECA-MIDI", "unable to open midi device " +
			erc.resource("midi-device") + "."));
  }

  ecadebug->control_flow("MIDI-thread ready " + erc.resource("midi-device"));

  while(true) {
    FD_ZERO(&rfds);
    FD_SET(fd, &rfds);

    tv.tv_sec = 0;
    tv.tv_usec = 100;
    
    retval = select(fd + 1, &rfds, NULL, NULL, &tv);

    if (retval > 0) {
      temp = read(fd, buf, sizeof(buf));
    }
    else temp = 0;
    pthread_mutex_lock(&midi_in_lock);
    for(size_t n = 0; n < temp; n++) {
      midi_in_queue.put(buf[n]);
    }
    pthread_mutex_unlock(&midi_in_lock);
  }
   
  close(fd);
}
