// ------------------------------------------------------------------------
// audioio-alsa.cpp: ALSA (/dev/snd/pcm*) input/output.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>
#ifdef COMPILE_ALSA

#include <string>
#include <cstring>
#include <cstdio>
#include <dlfcn.h>  

#include <kvutils.h>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-alsa.h"

#include "eca-error.h"
#include "eca-debug.h"

static int (*dl_snd_pcm_open)(snd_pcm_t **,int,int,int);
static int (*dl_snd_pcm_close)(snd_pcm_t *handle);
static const char* (*dl_snd_strerror)(int);
static int (*dl_snd_pcm_block_mode)(snd_pcm_t *handle, int enable);
static int (*dl_snd_pcm_info)(snd_pcm_t *handle, snd_pcm_info_t * info);
static int (*dl_snd_pcm_playback_info)(snd_pcm_t *handle, snd_pcm_playback_info_t * info);
static int (*dl_snd_pcm_playback_format)(snd_pcm_t *handle, snd_pcm_format_t * format);
static int (*dl_snd_pcm_playback_params)(snd_pcm_t *handle, snd_pcm_playback_params_t * params);
static int (*dl_snd_pcm_playback_status)(snd_pcm_t *handle, snd_pcm_playback_status_t * status);
static int (*dl_snd_pcm_flush_capture)(snd_pcm_t *handle);
static int (*dl_snd_pcm_drain_playback)(snd_pcm_t *handle);
static int (*dl_snd_pcm_flush_playback)(snd_pcm_t *handle);
static int (*dl_snd_pcm_playback_pause)(snd_pcm_t *handle, int enable);
static int (*dl_snd_pcm_playback_time)(snd_pcm_t *handle, int enable);
static ssize_t (*dl_snd_pcm_write)(snd_pcm_t *handle, const void *buffer, size_t size);
static ssize_t (*dl_snd_pcm_read)(snd_pcm_t *handle, void *buffer, size_t size);
static int (*dl_snd_pcm_capture_info)(snd_pcm_t *handle, snd_pcm_capture_info_t * info);
static int (*dl_snd_pcm_capture_format)(snd_pcm_t *handle, snd_pcm_format_t * format);
static int (*dl_snd_pcm_capture_params)(snd_pcm_t *handle, snd_pcm_capture_params_t * params);
static int (*dl_snd_pcm_capture_status)(snd_pcm_t *handle, snd_pcm_capture_status_t * status);
static int (*dl_snd_pcm_capture_time)(snd_pcm_t *handle, int enable);

static int eca_alsadevice_dynlib_count = 0;
static void *eca_alsadevice_dynlib_handle;

ALSADEVICE::ALSADEVICE (int card, int device, const SIMODE
			mode, const AIO_PARAMS& form, int bsize)
  :  AUDIO_IO_DEVICE(string("alsa,") + kvu_numtostr(card) +
		     string(",") + kvu_numtostr(device) , mode, form, bsize)
{
  card_number = card;
  device_number = device;
  //    format(fmt);
  finished(false);

  if (eca_alsadevice_dynlib_count == 0) load_dynlibs();
  eca_alsadevice_dynlib_count++;
  //  if (eca_alsadevice_dynlib_count == 0) load_dynlibs();
  //  eca_alsadevice_dynlib_count++;

  open_device();
  
  is_triggered = false;
}

void ALSADEVICE::load_dynlibs(void) {
  ecadebug->msg("(audioio-alsa) Loading libasound shared library.");
  
  eca_alsadevice_dynlib_handle = dlopen ("libasound.so", RTLD_LAZY);
  if (!eca_alsadevice_dynlib_handle) {
    throw(new ECA_ERROR("AUDIOIO-ALSA", "Unable to load asound.so dynamic library."));
  }

  dl_snd_pcm_open = 
    (int (*)(snd_pcm_t **, int, int, int))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_open");
  dl_snd_pcm_close = 
    (int (*)(snd_pcm_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_close");
  dl_snd_strerror = (const char* (*)(int))dlsym(eca_alsadevice_dynlib_handle, "snd_sterror");
  dl_snd_pcm_playback_pause =
    (int (*)(snd_pcm_t *, int))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_playback_pause");
  dl_snd_pcm_block_mode = 
    (int (*)(snd_pcm_t *, int))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_block_mode");
  dl_snd_pcm_info = 
    (int (*)(snd_pcm_t *, snd_pcm_info_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_info");
  dl_snd_pcm_playback_info = 
    (int (*)(snd_pcm_t *, snd_pcm_playback_info_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_playback_info");
  dl_snd_pcm_playback_format = 
    (int (*)(snd_pcm_t *, snd_pcm_format_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_playback_format");
  dl_snd_pcm_playback_params = 
    (int (*)(snd_pcm_t *handle, snd_pcm_playback_params_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_playback_params");
  dl_snd_pcm_playback_status = 
    (int (*)(snd_pcm_t *handle, snd_pcm_playback_status_t * status))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_playback_status");
  dl_snd_pcm_drain_playback = 
    (int (*)(snd_pcm_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_drain_playback");
  dl_snd_pcm_flush_playback = 
    (int (*)(snd_pcm_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_flush_playback");
  dl_snd_pcm_playback_pause = 
    (int (*)(snd_pcm_t *, int))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_playback_pause");
  dl_snd_pcm_playback_time = 
    (int (*)(snd_pcm_t *, int))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_playback_time");
  dl_snd_pcm_write = 
    (int (*)(snd_pcm_t *, const void *, size_t))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_write");
  dl_snd_pcm_read = 
    (int (*)(snd_pcm_t *, const void *, size_t))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_read");
#ifdef OLD_ALSALIB    
  dl_snd_pcm_capture_info = 
    (int (*)(snd_pcm_t *, snd_pcm_capture_info_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_record_info");
  dl_snd_pcm_capture_format = 
    (int (*)(snd_pcm_t *handle, snd_pcm_format_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_record_format");
  dl_snd_pcm_capture_params = 
    (int (*)(snd_pcm_t *handle, snd_pcm_capture_params_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_record_params");
  dl_snd_pcm_capture_status = 
    (int (*)(snd_pcm_t *handle, snd_pcm_capture_status_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_record_status");
  dl_snd_pcm_flush_capture = 
    (int (*)(snd_pcm_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_flush_record");
  dl_snd_pcm_capture_time = 
    (int (*)(snd_pcm_t *, int))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_record_time");
#else // ** ALSA >0.3.2
  dl_snd_pcm_capture_info = 
    (int (*)(snd_pcm_t *, snd_pcm_capture_info_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_capture_info");
  dl_snd_pcm_capture_format = 
    (int (*)(snd_pcm_t *handle, snd_pcm_format_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_capture_format");
  dl_snd_pcm_capture_params = 
    (int (*)(snd_pcm_t *handle, snd_pcm_capture_params_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_capture_params");
  dl_snd_pcm_capture_status = 
    (int (*)(snd_pcm_t *handle, snd_pcm_capture_status_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_capture_status");
  dl_snd_pcm_flush_capture = 
    (int (*)(snd_pcm_t *))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_flush_capture");
  dl_snd_pcm_capture_time = 
    (int (*)(snd_pcm_t *, int))dlsym(eca_alsadevice_dynlib_handle, "snd_pcm_capture_time");
#endif

  if (dlerror() != NULL) {
    throw(new ECA_ERROR("AUDIOIO-ALSA", "Error while loading asound.so dynamic library."));
  }
}

void ALSADEVICE::open_device(void) {
  if (is_open == true) return;
  int err;
  if (io_mode() == si_read) {
#ifdef OLD_ALSALIB
    if ((err = dl_snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_RECORD)) < 0) {
#else
    if ((err = dl_snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_CAPTURE)) < 0) {
#endif
      throw(new ECA_ERROR("AUDIOIO-ALSA", "unable to open ALSA-device for recording; error: " + string(dl_snd_strerror(err))));
    }
  }    
  else if (io_mode() == si_write) {
    if ((err = dl_snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_PLAYBACK)) < 0) {
      // Opening device failed
      throw(new ECA_ERROR("AUDIOIO-ALSA", "unable to open ALSA-device for playback; error: " +  string(dl_snd_strerror(err))));
    }
    // output triggering
    dl_snd_pcm_playback_pause(audio_fd, 1);
  }
  else if (io_mode() == si_readwrite) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Simultanious intput/ouput not supported."));
  }
  
  // ---
  // Set blocking mode.
  // ---
  dl_snd_pcm_block_mode(audio_fd, 1);    // enable block mode

  // ---
  // Set fragment size.
  // ---
  if (buffersize() == 0) 
    throw(new ECA_ERROR("AUDIOIO-ALSA", "Buffersize() is 0!", ECA_ERROR::stop));
    
  if (io_mode() == si_read) {
#ifdef OLD_ALSALIB
    snd_pcm_record_info_t pcm_info;
    snd_pcm_record_params_t pp;
#else
    snd_pcm_capture_info_t pcm_info;
    snd_pcm_capture_params_t pp;
#endif
    dl_snd_pcm_capture_info(audio_fd, &pcm_info);
    memset(&pp, 0, sizeof(pp));

    if (buffersize() * format().align > (int)pcm_info.buffer_size) 
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Buffer size too big, can't setup fragments."));

    pp.fragment_size = buffersize() * format().align;
    pp.fragments_min = 1;

    err = dl_snd_pcm_capture_params(audio_fd, &pp);

    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up buffer fragments: " + string(dl_snd_strerror(err))));
    }
  }
  else {
    snd_pcm_playback_info_t pcm_info;
    dl_snd_pcm_playback_info(audio_fd, &pcm_info);

    snd_pcm_playback_params_t pp;
    memset(&pp, 0, sizeof(pp));

    pp.fragment_size = buffersize() * format().align;
    pp.fragments_max = pcm_info.buffer_size / pp.fragment_size;
    pp.fragments_room = 1;
    
    err = dl_snd_pcm_playback_params(audio_fd, &pp);
    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up buffer fragments: " + string(dl_snd_strerror(err))));
    }
  }

  // ---
  // Select audio format
  // ---

  snd_pcm_format_t pf;

  memset(&pf, 0, sizeof(pf));
  if (format().bits == 16)
    pf.format = SND_PCM_SFMT_S16_LE; // DL_SND_PCM_FMT_U16_LE
  else 
    pf.format = SND_PCM_SFMT_U8;

  pf.rate = format().srate;
  pf.channels = format().channels; // Stereo

  if (io_mode() == si_read) {

    err = dl_snd_pcm_capture_format(audio_fd, &pf);

    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up record parameters: " + string(dl_snd_strerror(err))));
    }
  }
  else {
    err = dl_snd_pcm_playback_format(audio_fd, &pf);
    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up playback parameters: " + string(dl_snd_strerror(err))));
    }
  }
  is_open = true;
}

void ALSADEVICE::rt_stop(void) {
  ecadebug->msg(1, "(audioio-alsa) Audio device \"" + label() + "\" disabled.");
  if (io_mode() == si_write) {
    dl_snd_pcm_playback_pause(audio_fd, 1);
  }
  else {
    if (is_open) close_device();
  }
  is_triggered = false;
}

void ALSADEVICE::close_device(void) {
  if (is_open) {
    if (io_mode() == si_write) dl_snd_pcm_drain_playback(audio_fd);
    else if (io_mode() == si_read) dl_snd_pcm_flush_capture(audio_fd);
    dl_snd_pcm_close(audio_fd);
  }    
  is_open = false;
}

void ALSADEVICE::rt_ready(void) {
  ecadebug->msg(1, "(audioio-alsa) Audio device \"" + label() + "\" ready.");
  if (is_open == false) {
    open_device();
  }    
}

void ALSADEVICE::rt_activate(void) {
  if (is_triggered == false) {
    if (io_mode() == si_write) {
      snd_pcm_playback_status_t pb_status;
      dl_snd_pcm_playback_status(audio_fd, &pb_status);
      ecadebug->msg(2, "(audioio-alsa) Bytes in output-queue: " + 
		    kvu_numtostr(pb_status.queue) + ".");
      dl_snd_pcm_playback_pause(audio_fd, 0);
    }
    is_triggered = true;
  }
}

void ALSADEVICE::get_sample(SAMPLE_BUFFER* t) {
  t->reserve_buffer(buffersize());
  if (!is_open) throw(new ECA_ERROR("AUDIOIO-ALSA","get_sample(): trying to read from a closed device!"));

  bytes_read = dl_snd_pcm_read(audio_fd, t->iobuf_uchar, format().align * buffersize());

  if (bytes_read < 0)
    throw(new ECA_ERROR("AUDIOIO-ALSA","get_sample(): read error!", ECA_ERROR::stop));

  //    t->length_in_samples(samples_read / format().align);
  //  else 
  t->iobuf_to_buffer(bytes_read / format().align, true, format().bits, format().channels, format().srate);
}

void ALSADEVICE::put_sample(SAMPLE_BUFFER* t) {
  if (!is_open) throw(new ECA_ERROR("AUDIOIO-ALSA","put_sample(): trying to write to a closed device!"));
  
  //  t->buffer_to_iobuf(format().bits, SAMPLE_BUFFER::is_system_littleendian, format().channels);

  t->buffer_to_iobuf(format().bits, true, format().channels, format().srate);
  dl_snd_pcm_write(audio_fd, t->iobuf_uchar, format().align * t->length_in_samples());
}

ALSADEVICE::~ALSADEVICE(void) { 
  close_device(); 
  eca_alsadevice_dynlib_count--;
  if (eca_alsadevice_dynlib_count == 0) {
    ecadebug->msg("(audioio-alsa) Closing libasound shared library.");
    dlclose(eca_alsadevice_dynlib_handle);
  }
}

#endif // COMPILE_ALSA













