#ifndef _AUDIOFX_FILTER_H
#define _AUDIOFX_FILTER_H

#include <vector>
// #include <deque>

#include <kvutils.h>

#include "audiofx.h"
#include "samplebuffer.h"

class EFFECT_FILTER : public AUDIOFX_BASE {
  //
  // Base class for filter effects.
  //
private:

  int nm;
  SAMPLE_BUFFER::sample_type outputSample;

  vector<SAMPLE_BUFFER::sample_type> sin;
  vector<SAMPLE_BUFFER::sample_type> sout;

  vector<SAMPLE_BUFFER::single_type> a;
  vector<SAMPLE_BUFFER::single_type> b;

  void init_values(void);

 protected:

  inline void constant_a(int index, double newval) { a[index] = newval; }
  inline double constant_a(int index) { return(a[index]); }
  inline void constant_b(int index, double newval) { b[index] = newval; }
  inline double constant_b(int index) { return(b[index]); }
    
public:

  void process(SAMPLE_BUFFER::sample_type *insample);

  virtual EFFECT_FILTER* clone(void) = 0;

  EFFECT_FILTER(void) : sin(2), sout(2), a(3), b(2) { 
    support_for_dynamic_parameters(true); 
    init_values();
  }
};

class EFFECT_BANDPASS: public EFFECT_FILTER {
  //
  // A bandpass filter.
  //

private:
    
  double center;
  double width;

  double C;
  double D;

public:

  string label(void) { return("Bandpass filter"); }
  string id_string(void) { return("efb"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  EFFECT_BANDPASS* clone(void)  { return new EFFECT_BANDPASS(*this); }
  EFFECT_BANDPASS::EFFECT_BANDPASS (double centerf, double width);
};

class EFFECT_BANDREJECT: public EFFECT_FILTER {
  //
  // A band reject filter.
  //

private:
    
  double center;
  double width;

  double C;
  double D;

public:

  string label(void) { return("Bandreject filter"); }
  string id_string(void) { return("efr"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  EFFECT_BANDREJECT* clone(void)  { return new EFFECT_BANDREJECT(*this); }
  EFFECT_BANDREJECT::EFFECT_BANDREJECT (double centerf, double width);

};


class EFFECT_HIGHPASS : public EFFECT_FILTER {
  //
  // A highpass filter.
  //
  
 private:

  double cutOffFreq;

  double C;
    
public:
    
  string label(void) { return("Highpass filter"); }
  string id_string(void) { return("efh"); }
  string params(void) { return("cutoff-freq"); }


  void set_parameter(int param, double value);
  double get_parameter(int param);

  EFFECT_HIGHPASS* clone(void)  { return new EFFECT_HIGHPASS(*this); }
  EFFECT_HIGHPASS::EFFECT_HIGHPASS (double cutoff);

};

class EFFECT_INVERSE_COMB_FILTER: public EFFECT_FILTER {
  //
  // Inverse comb filter. 
  //
  // ---
  // The basic theory behind this can be found from Ken Steiglitz's book 
  // "A digital signal processing primer", page 77.
  // ---

  //  typedef RING_BUFFER<SAMPLE_BUFFER::sample_type> FSINGLE_BUFFER;
  typedef deque<SAMPLE_BUFFER::sample_type> FSINGLE_BUFFER;
  //  typedef vector<SAMPLE_BUFFER::sample_type> SSINGLE_BUFFER;
  vector<FSINGLE_BUFFER> buffer;
  SAMPLE_BUFFER::sample_type outputSample;
  double laskuri;
  SAMPLE_BUFFER::sample_type temp;

  double C;
  double D;

public:

  string label(void) { return("Inverse comb filter"); }
  string id_string(void) { return("efi"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_INVERSE_COMB_FILTER* clone(void)  { return new EFFECT_INVERSE_COMB_FILTER(*this); }  
  EFFECT_INVERSE_COMB_FILTER::EFFECT_INVERSE_COMB_FILTER (int delay_in_samples, double constant);
};

class EFFECT_LOWPASS: public EFFECT_FILTER {
  //
  // A lowpass filter.
  //
  // ---
  //   Algorithm:  1nd order filter.
  //   From Fugue source code:
  //  
  //    output[N] = input[N] * A + input[N-1] * B
  //  
  //    A = 2.0 * pi * center
  //    B = exp(-A / frequency)
  //  
  // ---

private:

  double cutOffFreq;

  double C;

public:

  string label(void) { return("Lowpass filter"); }
  string id_string(void) { return("efl"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  EFFECT_LOWPASS* clone(void)  { return new EFFECT_LOWPASS(*this); }  
  EFFECT_LOWPASS::EFFECT_LOWPASS (double cutoff);
};

class EFFECT_RESONANT_BANDPASS: public AUDIOFX_BASE {
//
// A resonant bandpass filter. 
//

private:

  vector<SAMPLE_BUFFER::sample_type> outhist;
  
  double center;
  double width;
  
  double a, b, c, R;
  double pole_angle;

public:

  string label(void) { return("Resonant bandpass filter"); }
  string id_string(void) { return("ef1"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONANT_BANDPASS* clone(void)  { return new EFFECT_RESONANT_BANDPASS(*this); }  
  EFFECT_RESONANT_BANDPASS::EFFECT_RESONANT_BANDPASS (double centerf, double width);
};

class EFFECT_RESONANT_LOWPASS: public EFFECT_FILTER {
  //
  // A resonant lowpass filter.
  //
  // ---
  // Algorithm is based on a sample filter-routine (iir_filter) posted to comp.dsp.
  // ---
    
  vector<SAMPLE_BUFFER::sample_type> outhist;
  vector<SAMPLE_BUFFER::sample_type> newhist;
    
  typedef struct {
    double a0, a1, a2;       // numerator coefficients
    double b0, b1, b2;       // denominator coefficients
  } BIQUAD;

  typedef struct {
    double A, B, C, D;       // filter coefficients
  } FILTER_COEF;
    
  vector<BIQUAD> ProtoCoef;         // Filter prototype coefficients,
                               // for each filter section
  vector<FILTER_COEF> Coef;
    
  double cutoff, Q, gain, gain_orig;
  double pi;
  double laskuri;

  double ad, bd, wp;      // for szxform()

  void szxform(int section);
  void refresh_values(void);

public:

  string label(void) { return("Lowpass filter"); }
  string id_string(void) { return("ef3"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONANT_LOWPASS* clone(void)  { return new EFFECT_RESONANT_LOWPASS(*this); }  
  EFFECT_RESONANT_LOWPASS (double cutoff, double resonance, double gain);
  EFFECT_RESONANT_LOWPASS (const EFFECT_RESONANT_LOWPASS& x);
};

class EFFECT_RESONATOR: public AUDIOFX_BASE {
  //
  // A resonating bandpass filter.
  //
  // ---
  // Based on a second order all-pole (IIR) band-pass filter from SPKit 
  // (for more info, see: http://www.music.helsinki.fi/research/spkit)
  // --

private:

  int nm; // temporary value 
  SAMPLE_BUFFER::sample_type temp;        // temporary values
    
  double center;
  double width;

  vector<SAMPLE_BUFFER::single_type> cona;
  vector<SAMPLE_BUFFER::single_type> conb;

  vector<SAMPLE_BUFFER::sample_type> saout;

public:

  string label(void) { return("Resonator filter"); }
  string id_string(void) { return("efr"); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONATOR* clone(void)  { return new EFFECT_RESONATOR(*this); }
  EFFECT_RESONATOR (double center, double width);
};

#endif



