// ------------------------------------------------------------------------
// ecatools-normalize.cpp: A simple command-line tools for normalizing
//                         sample volume.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <cstdio>
#include <signal.h>

#include <kvutils.h>

#include <eca-debug.h>
#include <eca-error.h>
#include <eca-controller.h>
#include <eca-main.h>
#include <eca-session.h>
#include <audiofx.h>
#include <eca-version.h>

#include "ecatools_normalize.h"

int main(int argc, char *argv[])
{
  ecadebug->set_debug_level(0);
  ecadebug->disable();

  signal(SIGTERM, signal_handler);
  signal(SIGINT, signal_handler);
  signal(SIGQUIT, signal_handler);

  COMMAND_LINE cline = COMMAND_LINE (argc, argv);

  if (cline.size() < 2) {
    print_usage();
    return(1);
  }

  try {
    string filename;
    string tempfile;
    double multiplier;
    EFFECT_ANALYZE* volume;
    EFFECT_AMPLIFY_SIMPLE* amp;

    for(int n = 1; n < cline.size(); n++) {
      filename = cline[n];
      ecatools_normalize_tempfile = string(tmpnam(NULL));
      ecatools_normalize_tempfile += ".wav";
      for(int m = 0;m < 2; m++) {
	ECA_SESSION esession;
	ECA_CONTROLLER ectrl (&esession);
  
	ectrl.new_chainsetup("default");
	ectrl.activate_chainsetup("default");
	if (m == 0) {
	  cerr << "Analyzing file \"" << filename << "\".\n";
	  ectrl.add_audio_device(filename, ECA_CONTROLLER::input);
	  ectrl.add_audio_device(ecatools_normalize_tempfile, ECA_CONTROLLER::output);

	  volume = new EFFECT_ANALYZE();
	  ectrl.add_chain_operator((CHAIN_OPERATOR*)volume);
	}
	else {

	  ectrl.add_audio_device(ecatools_normalize_tempfile, ECA_CONTROLLER::input);
	  ectrl.add_audio_device(filename, ECA_CONTROLLER::output);
	  
	  amp = new EFFECT_AMPLIFY_SIMPLE(multiplier * 100.0);
	  ectrl.add_chain_operator((CHAIN_OPERATOR*)amp);
	}
	ectrl.connect_active_chainsetup();
  
	ECA_PROCESSOR emain (&esession);
	emain.exec();
	
	if (m == 0) {
	  EFFECT_ANALYZE* pvolume = (EFFECT_ANALYZE*)ectrl.get_chain_operator("default", 0);
	  multiplier = pvolume->max_multiplier();
	  if (multiplier <= 1.0) {
	    cerr << "File \"" << filename << "\" is already normalized.\n";
	    break;
	  }
	  else {
	    cerr << "Normalizing file \"" << filename << "\" (amp-%: ";
	    cerr << multiplier * 100.0 << ").\n";
	  }
	}
      }
      remove(ecatools_normalize_tempfile.c_str());
    }
  }
  catch(ECA_ERROR* e) {
    cerr << "---\nERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "\nCaught an unknown exception.\n";
  }
  return(0);
}

void print_usage(void) {
  cerr << "****************************************************************************\n";
  cerr << "* [1mecatools_normalize, " << ecatools_normalize_version;
  cerr << " (linked to ecasound " << ecasound_version 
       << ")\n";
  cerr << "* (C) 1997-1999 Kai Vehmanen, released under GPL licence[0m \n";
  cerr << "****************************************************************************\n";

  cerr << "\nUSAGE: ecatools_normalize file1 [ file2, ... fileN ]\n\n";
}

void signal_handler(int signum) {
  cerr << "Unexpected interrupt... cleaning up.\n";
  remove(ecatools_normalize_tempfile.c_str());
  exit(1);
}

