// ------------------------------------------------------------------------
// eca-fileio-mmap-fthread.cpp: Thread sub-routines for mmap based file-io.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <vector>
#include <map>
#include <pthread.h>

#include <cstdio>
#include <cstring>
#include <fcntl.h>
#include <errno.h>

#include <sys/resource.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>

#include <kvutils.h>

#include "eca-debug.h"
#include "eca-error.h"

#include "eca-fileio-mmap-fthread.h"

pthread_mutex_t ecasound_fiommap_lock;
vector<ECASOUND_FIOMMAP_BUFFER*> ecasound_fiommap_buffer;
map<int,ECASOUND_FIOMMAP_BUFFER*> ecasound_fiommap_buffermap;
pthread_t ecasound_fiommap_thread;
const long int ecasound_fiommap_buffersize = 524288;

long int ecasound_fiommap_register_fd(int fd) {
  //  cerr << "(eca-fileio-mmap-fthread) register_fd()\n";

  static bool running = false;
  if (!running) {
    running = true;
    pthread_mutex_init(&ecasound_fiommap_lock, NULL);
    ecasound_fiommap_exec_thread();
  }

  ECASOUND_FIOMMAP_BUFFER* temp = new ECASOUND_FIOMMAP_BUFFER;
  temp->buffers[0] = new unsigned char [ecasound_fiommap_buffersize];
  temp->buffers[1] = new unsigned char [ecasound_fiommap_buffersize];
  temp->buffersize = ecasound_fiommap_buffersize; 
  temp->ready_for_fill[0] = true;
  temp->ready_for_fill[1] = true;
  temp->fd = fd;

  pthread_mutex_lock(&ecasound_fiommap_lock);
  ecasound_fiommap_buffer.push_back(temp);
  ecasound_fiommap_buffermap[fd] = temp;
  pthread_mutex_unlock(&ecasound_fiommap_lock);

  unsigned long usleep_count = 50;
  bool is_ready = false;
  while(!is_ready) {
    pthread_mutex_lock(&ecasound_fiommap_lock);
    if (ecasound_fiommap_buffer.back()->ready) is_ready = true;
    pthread_mutex_unlock(&ecasound_fiommap_lock);
    usleep(++usleep_count);
  }

  return(ecasound_fiommap_buffersize);
}

void ecasound_fiommap_close_fd(int fd) {
  //  cerr << "(eca-fileio-mmap-fthread) close_fd()\n";

  pthread_cancel(ecasound_fiommap_thread);
  pthread_join(ecasound_fiommap_thread,NULL);

  vector<ECASOUND_FIOMMAP_BUFFER*>::iterator p = ecasound_fiommap_buffer.begin();
  while(p != ecasound_fiommap_buffer.end()) {
    if ((*p)->fd == fd) {
     ecasound_fiommap_buffermap.erase(fd);
     ecasound_fiommap_buffer.erase(p);
     break;
    }
    ++p;
  }
  ecasound_fiommap_exec_thread();
}

void ecasound_fiommap_reset(int fd, long fposition) {
  //  cerr << "(eca-fileio-mmap-fthread) reset()\n";
  //  static unsigned char* buffer = new unsigned char [ecasound_fiommap_buffersize];

  pthread_mutex_lock(&ecasound_fiommap_lock);

  if (ecasound_fiommap_buffermap[fd]->ready_for_fill[0] == false) {
    munmap(ecasound_fiommap_buffermap[fd]->buffers[0], ecasound_fiommap_buffersize);
  }
  if (ecasound_fiommap_buffermap[fd]->ready_for_fill[1] == false) {
    munmap(ecasound_fiommap_buffermap[fd]->buffers[1], ecasound_fiommap_buffersize);
  }

  ecasound_fiommap_buffermap[fd]->mmap_high = 0;

  while (fposition >= ecasound_fiommap_buffermap[fd]->mmap_high) {
    ecasound_fiommap_buffermap[fd]->mmap_high += ecasound_fiommap_buffersize;
  }

  ecasound_fiommap_buffermap[fd]->mmap_low =
    ecasound_fiommap_buffermap[fd]->mmap_high - 
    ecasound_fiommap_buffersize;

  ecasound_fiommap_buffermap[fd]->locked_buffer = 0;

//    cerr << "(eca-fileio-mmap-fthread) mapping[s] (reset) from " << ecasound_fiommap_buffermap[fd]->mmap_low
//         << " to " << ecasound_fiommap_buffermap[fd]->mmap_high << " (fd:"
//         << ecasound_fiommap_buffermap[fd]->fd << ", buf: 0)\n";

  //  buffer = (unsigned char*)mmap(0,
  ecasound_fiommap_buffermap[fd]->buffers[0] = (unsigned char*)mmap(0,
								    ecasound_fiommap_buffersize,
								    PROT_READ,
								    MAP_SHARED,
								    ecasound_fiommap_buffermap[fd]->fd,
								    ecasound_fiommap_buffermap[fd]->mmap_low);

  //  memcpy(ecasound_fiommap_buffermap[fd]->buffers[0], buffer, ecasound_fiommap_buffersize);
  //  munmap(0, ecasound_fiommap_buffersize);

  //  mlock(ecasound_fiommap_buffermap[fd]->buffers[0],
  //	ecasound_fiommap_buffermap[fd]->buffersize);

//    cerr << "(eca-fileio-mmap-fthread) mapping[f] (reset) from " << ecasound_fiommap_buffermap[fd]->mmap_low
//         << " to " << ecasound_fiommap_buffermap[fd]->mmap_high << " (fd:"
//         << ecasound_fiommap_buffermap[fd]->fd << ", buf: 0).\n";


  ecasound_fiommap_buffermap[fd]->ready_for_fill[0] = false;
  ecasound_fiommap_buffermap[fd]->ready_for_fill[1] = true;

  ecasound_fiommap_buffermap[fd]->ready = true;

  pthread_mutex_unlock(&ecasound_fiommap_lock);
}

void ecasound_fiommap_next_buffer(int fd) {
  //  cerr << "(eca-fileio-mmap-fthread) next_buffer()\n";
  pthread_mutex_lock(&ecasound_fiommap_lock);

  if (ecasound_fiommap_buffermap[fd]->locked_buffer == 0) {
    ecasound_fiommap_buffermap[fd]->ready_for_fill[0] = true;
    ecasound_fiommap_buffermap[fd]->locked_buffer = 1;
  }
  else if (ecasound_fiommap_buffermap[fd]->locked_buffer == 1) {
    ecasound_fiommap_buffermap[fd]->ready_for_fill[1] = true;
    ecasound_fiommap_buffermap[fd]->locked_buffer = 0;
  }

  pthread_mutex_unlock(&ecasound_fiommap_lock);
}

unsigned char* ecasound_fiommap_active_buffer(int fd) {
  return(ecasound_fiommap_buffermap[fd]->buffers[ecasound_fiommap_buffermap[fd]->locked_buffer]);
}

void ecasound_fiommap_exec_thread(void) {
  int retcode = pthread_create(&ecasound_fiommap_thread, NULL, ecasound_fiommap_process, NULL);
  if (retcode != 0)
    throw(new ECA_ERROR("ECA-FILEIO-MMAP-FTHREAD", "unable to create thread for mmap()'ed file-I/O"));
}

void *ecasound_fiommap_process(void *) { 
  //  cerr << "(eca-fileio-mmap-fthread) process()\n";
  vector<ECASOUND_FIOMMAP_BUFFER*>::const_iterator p;
  vector<unsigned char*>::size_type q;
  int count;
  unsigned long usleep_count = 0;

  //  unsigned char* buffer = new unsigned char [ecasound_fiommap_buffersize];

  //  if (setpriority(PRIO_PROCESS, 0, -10) == -1)
  //    ecadebug->msg("(eca-fileio-mmap-fthread) Unable to change priority.");
  //  MESSAGE_ITEM mtemp;
  //  mtemp << "(eca-fileio-mmap-fthread) Raised process priority (for PID): " << (int)getpid();
  //  ecadebug->msg(1, mtemp.to_string());

  mlockall(MCL_CURRENT | MCL_FUTURE);

  while(true) {
    count = 0;
    pthread_mutex_lock(&ecasound_fiommap_lock);
    p = ecasound_fiommap_buffer.begin();
    while(p != ecasound_fiommap_buffer.end()) {
      q = 0;
      while(q != (*p)->buffers.size()) {
	if ((*p)->mmap_low != -1) {
	  if ((*p)->locked_buffer != q) {
	    if ((*p)->ready_for_fill[q] == false) {
	      ++q;
	      continue;
	    }
	    
	    ++count;
	    //	    cerr << "20000x" << usleep_count << ",";
	    usleep_count = 0;
	    
	    munmap((*p)->buffers[q], ecasound_fiommap_buffersize);
	    
	    (*p)->mmap_low += ecasound_fiommap_buffersize;
	    (*p)->mmap_high += ecasound_fiommap_buffersize;
	  
	    //	    cerr << "(eca-fileio-mmap-fthread) mapping[s] (processs) from " << (*p)->mmap_low;
	    //	    cerr << " to " << (*p)->mmap_high << " (fd:" << (*p)->fd;
	    //	    cerr << ", buf:" << q << ").\n";
	  

	    //  buffer = (unsigned char*)mmap(0, 
	    (*p)->buffers[q] = (unsigned char*)mmap(0, 
						    ecasound_fiommap_buffersize,
						    PROT_READ,
						    MAP_SHARED,
						    (*p)->fd,
						    (*p)->mmap_low);
	    //	    memcpy((*p)->buffers[q], buffer, ecasound_fiommap_buffersize);
	    //	    munmap(0, ecasound_fiommap_buffersize);
	    //	    mlock((*p)->buffers[q], (*p)->buffersize);

	    //	    cerr << "(eca-fileio-mmap-fthread) mapping[f] (processs) from " << (*p)->mmap_low;
	    //	    cerr	 << " to " << (*p)->mmap_high << " (fd:" << (*p)->fd;
	    //	    cerr     << ", buf:" << q << ").\n";

	    (*p)->ready_for_fill[q] = false;
	  }
	}
	else {
	  pthread_mutex_unlock(&ecasound_fiommap_lock);
	  ecasound_fiommap_reset((*p)->fd, 0);
	  pthread_mutex_lock(&ecasound_fiommap_lock);
	}
	++q;
      }
      ++p;
    }
    pthread_mutex_unlock(&ecasound_fiommap_lock);
    pthread_testcancel();
    if (count == 0) {
      usleep(20000);
      ++usleep_count;
    }
 }
}




