#ifndef _AUDIOFX_FILTER_H
#define _AUDIOFX_FILTER_H

#include <vector>

#include "audiofx.h"
#include "samplebuffer.h"

/**
 * Virtual base for filter effects.
 * @author Kai Vehmanen
 */
class EFFECT_FILTER : public EFFECT_BASE {

 public:
  virtual ~EFFECT_FILTER(void) { }
};

/**
 * Base class for butterworth filter effects.
 * 
 * Based on SPKit Butterworth algorithms. 
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_BW_FILTER : public EFFECT_FILTER {

private:
  
  int nm;

  SAMPLE_BUFFER::sample_type outputSample;

  vector<SAMPLE_BUFFER::sample_type> sin;
  vector<SAMPLE_BUFFER::sample_type> sout;

  void init_values(void);

 protected:

  vector<SAMPLE_BUFFER::single_type> a;
  vector<SAMPLE_BUFFER::single_type> b;
   
public:

  void process(SAMPLE_BUFFER::sample_type *insample);
  void process_notused(SAMPLE_BUFFER* sbuf);

  virtual EFFECT_BW_FILTER* clone(void) = 0;

  EFFECT_BW_FILTER(void) : sin(2), sout(2), a(3), b(2) {
    init_values();
  }
};

/**
 * Bandpass filter
 *  
 * Based on SPKit Butterworth algorithms
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_BANDPASS: public EFFECT_BW_FILTER {

private:
    
  parameter_type center;
  parameter_type width;

  parameter_type C;
  parameter_type D;

public:

  string name(void) { return("Bandpass filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("center-freq");
    t.push_back("width");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  EFFECT_BANDPASS* clone(void)  { return new EFFECT_BANDPASS(*this); }
  EFFECT_BANDPASS (parameter_type centerf = 0.0, parameter_type width = 1.0);
};

/**
 * Bandreject filter
 *  
 * Based on SPKit Butterworth algorithms
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_BANDREJECT: public EFFECT_BW_FILTER {

private:
    
  parameter_type center;
  parameter_type width;

  parameter_type C;
  parameter_type D;

public:

  string name(void) { return("Bandreject filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("center-freq");
    t.push_back("width");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  EFFECT_BANDREJECT* clone(void)  { return new EFFECT_BANDREJECT(*this); }
  EFFECT_BANDREJECT (parameter_type centerf = 0.0, parameter_type width = 1.0);
};

/**
 * Highpass filter
 *
 * Based on SPKit Butterworth algorithms
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_HIGHPASS : public EFFECT_BW_FILTER {
  
 private:

  parameter_type cutOffFreq;

  parameter_type C;
    
public:
    
  string name(void) { return("Highpass filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("cutoff-freq");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  EFFECT_HIGHPASS* clone(void)  { return new EFFECT_HIGHPASS(*this); }
  EFFECT_HIGHPASS (parameter_type cutoff = 0.0);
};

/**
 * Inverse comb filter
 *
 * The basic theory behind this can be found from Ken Steiglitz's book 
 * "A digital signal processing primer", page 77.
 */
class EFFECT_INVERSE_COMB_FILTER : public EFFECT_FILTER {

  deque<SAMPLE_BUFFER::sample_type> buffer;
  parameter_type laskuri;
  SAMPLE_BUFFER::sample_type temp;

  parameter_type C;
  parameter_type D;

public:

  string name(void) { return("Inverse comb filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("delay");
    t.push_back("radius");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_INVERSE_COMB_FILTER* clone(void)  { return new EFFECT_INVERSE_COMB_FILTER(*this); }  
  EFFECT_INVERSE_COMB_FILTER (int delay_in_samples = 0, parameter_type constant = 1.0);
};

/**
 * Lowpass filter
 *  
 * Based on SPKit Butterworth algorithms
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_LOWPASS: public EFFECT_BW_FILTER {

private:

  parameter_type cutOffFreq;

  parameter_type C;

public:

  string name(void) { return("Lowpass filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("cutoff-freq");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void set_cutoff(parameter_type value, long int srate);

  EFFECT_LOWPASS* clone(void)  { return new EFFECT_LOWPASS(*this); }  
  EFFECT_LOWPASS (parameter_type cutoff = 0.0);
};

/**
 * A simple lowpass filter
 *                                              
 *   Algorithm:  1nd order filter.             
 *   From Fugue source code:                   
 *                                             
 *    output[N] = input[N] * A + input[N-1] * B
 *                                             
 *    A = 2.0 * pi * center                    
 *    B = exp(-A / frequency)
 */                                            
class EFFECT_LOWPASS_SIMPLE : public EFFECT_FILTER {

private:

  parameter_type cutOffFreq;
  parameter_type A, B;
  SAMPLE_BUFFER::sample_type outhist;
  SAMPLE_BUFFER::sample_type tempin, temphist;

public:

  string name(void) { return("Simple lowpass filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("cutoff-freq");
    return(t);
  }


  void process(SAMPLE_BUFFER::sample_type *insample);

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  EFFECT_LOWPASS_SIMPLE* clone(void)  { return new EFFECT_LOWPASS_SIMPLE(*this); }
  EFFECT_LOWPASS_SIMPLE (parameter_type cutoff = 0.0);
};

/**
 * Resonant bandpass filter
 */
class EFFECT_RESONANT_BANDPASS : public EFFECT_FILTER {

private:

  vector<SAMPLE_BUFFER::sample_type> outhist;
  
  parameter_type center;
  parameter_type width;
  
  parameter_type a, b, c, R;
  parameter_type pole_angle;

public:

  string name(void) { return("Resonant bandpass filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("center-freq");
    t.push_back("width");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONANT_BANDPASS* clone(void)  { return new EFFECT_RESONANT_BANDPASS(*this);
  }  
  EFFECT_RESONANT_BANDPASS (parameter_type centerf, parameter_type width);
  EFFECT_RESONANT_BANDPASS (void) : outhist(2) { }
};

/**
 * Resonant lowpass filter
 *
 * Algorithm is based on a sample filter-routine (iir_filter) posted to comp.dsp.
 */
class EFFECT_RESONANT_LOWPASS : public EFFECT_FILTER {
    
  vector<SAMPLE_BUFFER::sample_type> outhist;
  vector<SAMPLE_BUFFER::sample_type> newhist;
    
  class TRIPLE_COEFS {
  public:
    parameter_type a0, a1, a2;       // numerator coefficients
    parameter_type b0, b1, b2;       // denominator coefficients
  };

  class FILTER_COEFS {
  public:
    parameter_type A, B, C, D;       // filter coefficients
  };
    
  vector<TRIPLE_COEFS> ProtoCoef;         // Filter prototype coefficients,
                               // for each filter section
  vector<FILTER_COEFS> Coef;
    
  parameter_type cutoff, Q, gain, gain_orig;
  parameter_type pi;
  parameter_type laskuri;

  parameter_type ad, bd, wp;      // for szxform()

  void szxform(int section);
  void refresh_values(void);

public:

  string name(void) { return("Resonant Lowpass filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("cutoff-freq");
    t.push_back("resonance");
    t.push_back("gain");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONANT_LOWPASS* clone(void)  { return new EFFECT_RESONANT_LOWPASS(*this); }  
  EFFECT_RESONANT_LOWPASS (parameter_type cutoff = 0.0,
			   parameter_type resonance = 1.0,
			   parameter_type gain = 1.0);
};

/**
 * Resonating bandpass filter
 *
 * Based on a second order all-pole (IIR) band-pass filter from SPKit 
 * (for more info, see: http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_RESONATOR : public EFFECT_FILTER {

private:

  int nm; // temporary value 
  SAMPLE_BUFFER::sample_type temp;        // temporary values
    
  parameter_type center;
  parameter_type width;

  vector<SAMPLE_BUFFER::single_type> cona;
  vector<SAMPLE_BUFFER::single_type> conb;

  vector<SAMPLE_BUFFER::sample_type> saout;

public:

  string name(void) { return("Resonator filter"); }

  vector<string> parameter_names(void) const {
    vector<string> t;
    t.push_back("center-freq");
    t.push_back("width");
    return(t);
  }

  void set_parameter(int param, parameter_type value);
  parameter_type get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONATOR* clone(void)  { return new EFFECT_RESONATOR(*this); }
  EFFECT_RESONATOR (parameter_type center, parameter_type width);
  EFFECT_RESONATOR (void) : cona(1), conb(2), saout(2) { 
    map_parameters();
  }
};

#endif
