// ------------------------------------------------------------------------
// audioio-mp3.cpp: Interface to mpg123 (input) and lame (output).
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>
#include <string>
#include <cstring>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include "unistd.h"
#include <pthread.h>

#include <kvutils.h>

#include "audioio-mp3.h"
#include "audioio-mp3_impl.h"
#include "samplebuffer.h"
#include "audioio.h"

#include "eca-error.h"
#include "eca-debug.h"

// int audioio_mp3_pid_of_mpg123;
FILE* audioio_mp3_pipe;

MP3FILE::MP3FILE(const string& name, const SIMODE mode, const AIO_PARAMS& fmt, int bsize) 
  :  AUDIO_IO_FILE(name, mode, fmt, bsize) 
{

  AIO_PARAMS sffmt;
  sffmt.channels = 2;
  sffmt.bits = 16;
  sffmt.srate = 44100;
  format(sffmt);

  finished(false);

  is_open = false;
  if (io_mode() == si_read) {
    get_mp3_params(label());
  }
}

MP3FILE::~MP3FILE(void) {
  close_device();
}

void MP3FILE::open_device(void) { 
  if (io_mode() == si_read) {
    fork_mpg123();
  }
  set_open_state(true);
}

void MP3FILE::close_device(void) {
  kill_mpg123();
  set_open_state(false);
}

void MP3FILE::read_buffer(SAMPLE_BUFFER* t) {
  if (is_open == false) fork_mpg123();
  t->reserve_buffer(buffersize());
  samples_read = fread(t->iobuf_uchar, format().align, buffersize(), fobject);

  if (!ferror(fobject)) {
    t->iobuf_to_buffer(samples_read, SAMPLE_BUFFER::is_system_littleendian, format().bits, format().channels, format().srate);
    position_in_samples(position_in_samples() + t->length_in_samples());
    if (feof(fobject)) finished(true);
  }
  else
    throw(new ECA_ERROR("AUDIOIO-MP3","Read error!"));
}


void MP3FILE::write_buffer(SAMPLE_BUFFER* t) {
  if (is_open == false) fork_lame();
  t->buffer_to_iobuf(SAMPLE_BUFFER::is_system_littleendian,
		     format().bits,
		     format().channels, 
		     format().srate);

  position_in_samples(position_in_samples() + t->length_in_samples());
  length_in_samples(position_in_samples());

  fwrite(t->iobuf_uchar, format().align, t->length_in_samples(), fobject);
  if (ferror(fobject)) finished(true);
}

void MP3FILE::position_in_samples_hook(void) {
  kill_mpg123();

  if (position_in_samples() > length_in_samples()) 
    position_in_samples(length_in_samples());
  else if (position_in_samples() < 0) 
    position_in_samples(0);

  finished(false);
}

void MP3FILE::get_mp3_params(const string& fname) {
  Layer newlayer;
  FILE *temp;
  temp = fopen(fname.c_str(),"r");
  if (!temp)
    throw(new ECA_ERROR("ECA-MP3","Unable to open temp file " + fname, ECA_ERROR::retry));
  newlayer.get(temp);
  fseek(temp, 0, SEEK_END);
  double fsize = (double)ftell(temp);
  double bitrate = ((double)newlayer.bitrate() * 1000.0);
  double bsecond = (double)(long)format().byte_second;
  MESSAGE_ITEM m;
  m << "MP3 file size: " << fsize << "\n.";
  m << "MP3 length_value: " << newlayer.length() << "\n.";
  m << "bsecond: " << bsecond << "\n.";
  m << "bitrate: " << bitrate << "\n.";
  length_in_samples((long)ceil(8.0 * fsize / bitrate * bsecond / format().align));

  m << "Setting MP3 length_value: " << length_in_seconds() << "\n.";
  pcm = newlayer.pcmPerFrame();

  m << "MP3 pcm value: " << pcm << ".";
  ecadebug->msg(4,m.to_string());
  fclose(temp);
}

void MP3FILE::kill_mpg123(void) {
  if (is_open) {
    ecadebug->msg(1, "(audioio-mp3) Killing mpg123 child thread");
    //		  + kvu_numtostr(audioio_mp3_pid_of_mpg123) + ").");
    //    pthread_cancel(mp3_thread);
    if (io_mode() == si_read) {
      pthread_kill(mp3_thread, SIGKILL);
      pthread_join(mp3_thread,NULL);
    }
    pclose(fobject);
    is_open = false;
  }
}

void MP3FILE::fork_mpg123(void) {
 if (!is_open) {

    MESSAGE_ITEM komen;
    //    komen << "sh -c mpg123 -s -k ";
    komen << "mpg123 -s -k ";
    komen << (long)(position_in_samples() / pcm);
    //    komen << " " << label() << " >/tmp/ecasound_mp3_fifo
    //    2>/dev/null";
    komen << " '" << label() << "' 2>/dev/null";
    ecadebug->msg(2,komen.to_string());
    string* mpg123cmd = new string("");
    *mpg123cmd = komen.to_string();

    int retcode = pthread_create(&mp3_thread, NULL, start_mpg123, (void*)mpg123cmd);
    if (retcode != 0) {
      throw(new ECA_ERROR("ECA-MP3","Can't start mpg123-thread!"));
    }

    if (io_mode() == si_read) {
      fobject = popen((char*)mpg123cmd->c_str(),"r");
      //      usleep(1000);
      //      fobject = audioio_mp3_pipe;
      if (fobject == NULL) {
	throw(new ECA_ERROR("ECA-MP3","Can't start mpg123-thread! Check that 'mpg123' is installed properly."));
      }
    }
    is_open = true;
 }
}

void* start_mpg123(void* mcmd) {
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);  // other
  // threads can stop this one
  pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);

  ecadebug->control_flow("Mpg123 child-thread ready.");
  ecadebug->msg(1,"mp3_thread-pid: " + kvu_numtostr(getpid()));
  
  string* mpg123cmd = (string*)mcmd;

  audioio_mp3_pipe = popen((char*)mpg123cmd->c_str(),"r");
  pclose(audioio_mp3_pipe);
  return(0);
}

void MP3FILE::fork_lame(void) {
 if (!is_open) {

    MESSAGE_ITEM komen;
    //    komen << "sh -c mpg123 -s -k ";
    //    komen << "lame -S - ";
    komen << "lame -x -S - '";
    komen << label() << "' 2>/dev/null";
    ecadebug->msg("(audioio-mp3) Starting to encode " + label() + " with lame.");
    ecadebug->msg(2,komen.to_string());
    string* lamecmd = new string("");
    *lamecmd = komen.to_string();

    //    int retcode = pthread_create(&mp3_thread, NULL, start_lame, (void*)lamecmd);
    //    if (retcode != 0) {
    //      throw(new ECA_ERROR("ECA-MP3","Can't start lame-thread!"));
    //    }

    //    fobject = audioio_mp3_pipe;
    fobject = popen((char*)lamecmd->c_str(),"w");
    //    audioio_mp3_pipe = popen((char*)mpg123cmd->c_str(),"r");
    if (fobject == NULL) {
      throw(new ECA_ERROR("ECA-MP3","Can't start lame-thread! Check that 'lame' is installed properly."));
    }

    is_open = true;
 }
}

void* start_lame(void* mcmd) {
  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);  // other
  // threads can stop this one
  pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);

  ecadebug->control_flow("Lame child-thread ready.");
  ecadebug->msg(1,"mp3_thread-pid: " + kvu_numtostr(getpid()));
  
  string* lamecmd = (string*)mcmd;

  audioio_mp3_pipe = popen((char*)lamecmd->c_str(),"w");
  pclose(audioio_mp3_pipe);
  return(0);
}
