// ------------------------------------------------------------------------
// audiofx_analysis.cpp: Signal analyzing.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>

#include <kvutils.h>

#include "samplebuffer.h"
#include "audiofx_analysis.h"

#include "eca-debug.h"
#include "eca-error.h"

EFFECT_ANALYZE::EFFECT_ANALYZE (void) {
  max = 0;

  for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
    num_of_samples[ch] = 0;
  for(nm = 0; nm < 16; nm++)
    for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
      ranges[nm][ch] = 0;
}

EFFECT_ANALYZE::~EFFECT_ANALYZE (void) { }

EFFECT_ANALYZE::EFFECT_ANALYZE (const EFFECT_ANALYZE& x) {
  for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
    num_of_samples[ch] = x.num_of_samples[ch];
  for(nm = 0; nm < 16; nm++)
    for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
      ranges[nm][ch] = x.ranges[nm][ch];
  nm = 0;
}

string EFFECT_ANALYZE::status(void) {
  MESSAGE_ITEM otemp;
  otemp << "(audiofx) -- Printing volume statistics --\n";
  for(nm = 0; nm < 16; nm++) {
    otemp.setprecision(2);
    otemp << "(audiofx) Vol-range: ";
    otemp << nm << "\t L: ";
    otemp << ranges[nm][SAMPLE_BUFFER::ch_left] << " (";
    otemp << ((DYNAMIC_PARAMETERS::parameter_type)ranges[nm][SAMPLE_BUFFER::ch_left] / (DYNAMIC_PARAMETERS::parameter_type)num_of_samples[SAMPLE_BUFFER::ch_left] * 100.0);
    otemp << ") \t\t";
    otemp << "R: ";
    otemp << ranges[nm][SAMPLE_BUFFER::ch_right] << " (";
    otemp << ((DYNAMIC_PARAMETERS::parameter_type)ranges[nm][SAMPLE_BUFFER::ch_right] / (DYNAMIC_PARAMETERS::parameter_type)num_of_samples[SAMPLE_BUFFER::ch_right] * 100.0);
    otemp << ").\n";
  
  }
  otemp << "(audiofx) -- End of statistics (counters reseted)--\n";
  otemp.setprecision(5);
  otemp << "(audiofx) Max amplitude " << max;
  otemp << "; signal can be amplified by " << max_multiplier() * 100.0;
  otemp << "%.";

  for(nm = 0; nm < 16; nm++)
    for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
      ranges[nm][ch] = 0;
  for(int ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++)
    num_of_samples[ch] = 0;
  return(otemp.to_string());
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_ANALYZE::max_multiplier(void) const { 
  DYNAMIC_PARAMETERS::parameter_type kerroin;
  
  if (max != 0.0) kerroin = SAMPLE_BUFFER::max_amplitude / max;
  if (kerroin < 1.0) kerroin = 1.0;

  return(kerroin);
}

void EFFECT_ANALYZE::process(SAMPLE_BUFFER::sample_type *insample) {
  if (fabs(insample->get_left()) > max) max = fabs(insample->get_left());
  if (fabs(insample->get_right()) > max) max = fabs(insample->get_right());    

  for(nm = 0;nm < SAMPLE_BUFFER::ch_count; nm++) {
    num_of_samples[nm]++;
    if (insample->get_channel(nm)      <  SAMPLE_BUFFER::max_amplitude * -7.0/8.0) ranges[0][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -6.0/8.0) ranges[1][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -5.0/8.0) ranges[2][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -4.0/8.0) ranges[3][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -3.0/8.0) ranges[4][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -2.0/8.0) ranges[5][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * -1.0/8.0) ranges[6][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::silent_value) ranges[7][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 1.0/8.0) ranges[8][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 2.0/8.0) ranges[9][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 3.0/8.0) ranges[10][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 4.0/8.0) ranges[11][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 5.0/8.0) ranges[12][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 6.0/8.0) ranges[13][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude * 7.0/8.0) ranges[14][nm]++;
    else if (insample->get_channel(nm) <  SAMPLE_BUFFER::max_amplitude) ranges[15][nm]++;
  }
}

EFFECT_DCFIND::EFFECT_DCFIND (void) {
    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
        pos_sum[nm] = neg_sum[nm] = num_of_samples[nm] = 0.0;
    }
}

EFFECT_DCFIND::EFFECT_DCFIND (const EFFECT_DCFIND& x) {
  for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
    pos_sum[nm] = x.pos_sum[nm];
    neg_sum[nm] = x.neg_sum[nm];
    num_of_samples[nm] = x.num_of_samples[nm];
  }
  tempval = x.tempval;
  nm = x.nm;
}

string EFFECT_DCFIND::status(void) {
    MESSAGE_ITEM mitem;
    mitem.setprecision(5);
    mitem << "(audiofx) Optimal value for DC-adjust: ";
    mitem << get_deltafix(SAMPLE_BUFFER::ch_left) << " (left), ";
    mitem << get_deltafix(SAMPLE_BUFFER::ch_right) << " (right).";
    return(mitem.to_string());
}

DYNAMIC_PARAMETERS::parameter_type EFFECT_DCFIND::get_deltafix(int channel) { 
  SAMPLE_BUFFER::single_type deltafix;

  if (pos_sum[channel] > neg_sum[channel]) deltafix = -(pos_sum[channel] - neg_sum[channel]) / num_of_samples[channel];
  else deltafix = (neg_sum[channel] - pos_sum[channel]) / num_of_samples[channel];

  return((DYNAMIC_PARAMETERS::parameter_type)deltafix); 
}

void EFFECT_DCFIND::process(SAMPLE_BUFFER::sample_type *insample) {
    for(nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
        tempval = insample->get_channel(nm);
        if (tempval > SAMPLE_BUFFER::silent_value)
            pos_sum[nm] += tempval;
        else
            neg_sum[nm] += fabs(tempval);
        num_of_samples[nm]++;
    }
}
