// ------------------------------------------------------------------------
// audioio-alsa.cpp: ALSA (/dev/snd/pcm*) input/output.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>
#ifdef COMPILE_ALSA

#include <string>
#include <cstring>
#include <cstdio>
#include <dlfcn.h>  

#include <kvutils.h>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-alsa.h"
#include "eca-alsa-dyn.h"

#include "eca-error.h"
#include "eca-debug.h"

ALSADEVICE::ALSADEVICE (int card, int device, const SIMODE
			mode, const AIO_PARAMS& form, int bsize)
  :  AUDIO_IO_DEVICE(string("alsa,") + kvu_numtostr(card) +
		   string(",") + kvu_numtostr(device) , mode, form, bsize)
{
  card_number = card;
  device_number = device;
  //    format(fmt);
  finished(false);
  set_open_state(false);
  is_triggered = false;

  overruns = underruns = 0;

  eca_alsa_load_dynamic_support();
}

void ALSADEVICE::open_device(void) {
  if (is_open() == true) return;
  int err;
  if (io_mode() == si_read) {
#ifdef OLD_ALSALIB
    err = dl_snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_RECORD);
#else
    err = dl_snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_CAPTURE);
#endif
    if (err != 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "unable to open ALSA-device for recording; error: " + string(dl_snd_strerror(err))));
    }
  }    
  else if (io_mode() == si_write) {
    err = dl_snd_pcm_open(&audio_fd, card_number, device_number,
			    SND_PCM_OPEN_PLAYBACK);
    if (err != 0) {
      // Opening device failed
      throw(new ECA_ERROR("AUDIOIO-ALSA", "unable to open ALSA-device for playback; error: " +  string(dl_snd_strerror(err))));
    }
    // output triggering
    dl_snd_pcm_playback_pause(audio_fd, 1);
  }
  else if (io_mode() == si_readwrite) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Simultanious intput/ouput not supported."));
  }
  
  // ---
  // Set blocking mode.
  // ---
  dl_snd_pcm_block_mode(audio_fd, 1);    // enable block mode

  // ---
  // Set fragment size.
  // ---
  if (buffersize() == 0) 
    throw(new ECA_ERROR("AUDIOIO-ALSA", "Buffersize() is 0!", ECA_ERROR::stop));
    
  if (io_mode() == si_read) {
#ifdef OLD_ALSALIB
    snd_pcm_record_info_t pcm_info;
    snd_pcm_record_params_t pp;
#else
    snd_pcm_capture_info_t pcm_info;
    snd_pcm_capture_params_t pp;
#endif
    dl_snd_pcm_capture_info(audio_fd, &pcm_info);
    memset(&pp, 0, sizeof(pp));

    if (buffersize() * format().align > (int)pcm_info.buffer_size) 
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Buffer size too big, can't setup fragments."));

    pp.fragment_size = buffersize() * format().align;
    pp.fragments_min = 1;

    err = dl_snd_pcm_capture_params(audio_fd, &pp);

    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up buffer fragments: " + string(dl_snd_strerror(err))));
    }
  }
  else {
    snd_pcm_playback_info_t pcm_info;
    dl_snd_pcm_playback_info(audio_fd, &pcm_info);

    snd_pcm_playback_params_t pp;
    memset(&pp, 0, sizeof(pp));

    pp.fragment_size = buffersize() * format().align;
    pp.fragments_max = pcm_info.buffer_size / pp.fragment_size;
    pp.fragments_room = 2;
    
    err = dl_snd_pcm_playback_params(audio_fd, &pp);
    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up buffer fragments: " + string(dl_snd_strerror(err))));
    }
  }

  // ---
  // Select audio format
  // ---

  snd_pcm_format_t pf;

  memset(&pf, 0, sizeof(pf));
  if (format().bits == 16)
    pf.format = SND_PCM_SFMT_S16_LE; // DL_SND_PCM_FMT_U16_LE
  else 
    pf.format = SND_PCM_SFMT_U8;

  pf.rate = format().srate;
  pf.channels = format().channels; // Stereo

  if (io_mode() == si_read) {

    err = dl_snd_pcm_capture_format(audio_fd, &pf);

    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up record parameters: " + string(dl_snd_strerror(err))));
    }
  }
  else {
    err = dl_snd_pcm_playback_format(audio_fd, &pf);
    if (err < 0) {
      throw(new ECA_ERROR("AUDIOIO-ALSA", "Error when setting up playback parameters: " + string(dl_snd_strerror(err))));
    }
  }
  set_open_state(true);
}

void ALSADEVICE::rt_stop(void) {
  ecadebug->msg(1, "(audioio-alsa) Audio device \"" + label() + "\" disabled.");
  if (io_mode() == si_write) {
    dl_snd_pcm_playback_pause(audio_fd, 1);
  }
  else {
    if (is_open()) close_device();
  }
  is_triggered = false;
}

void ALSADEVICE::close_device(void) {
  if (is_open()) {
    if (io_mode() != si_read) {
      snd_pcm_playback_status_t pb_status;
      dl_snd_pcm_playback_status(audio_fd, &pb_status);
      underruns += pb_status.underrun;
      dl_snd_pcm_drain_playback(audio_fd);
    }
    else if (io_mode() == si_read) {
#ifdef OLD_ALSALIB
      snd_pcm_record_status_t ca_status;
#else
      snd_pcm_capture_status_t ca_status;
#endif
      dl_snd_pcm_capture_status(audio_fd, &ca_status);
      overruns += ca_status.overrun;
      dl_snd_pcm_flush_capture(audio_fd);
    }
    dl_snd_pcm_close(audio_fd);
  }    
  set_open_state(false);
}

void ALSADEVICE::rt_ready(void) {
  ecadebug->msg(1, "(audioio-alsa) Audio device \"" + label() + "\" ready.");
  if (is_open() == false) {
    open_device();
  }    
}

void ALSADEVICE::rt_activate(void) {
  if (is_triggered == false) {
    if (io_mode() == si_write) {
      snd_pcm_playback_status_t pb_status;
      dl_snd_pcm_playback_status(audio_fd, &pb_status);
      ecadebug->msg(2, "(audioio-alsa) Bytes in output-queue: " + 
		    kvu_numtostr(pb_status.queue) + ".");
      dl_snd_pcm_playback_pause(audio_fd, 0);
    }
    is_triggered = true;
  }
}

void ALSADEVICE::read_buffer(SAMPLE_BUFFER* t) {
  t->reserve_buffer(buffersize());
  if (!is_open()) throw(new ECA_ERROR("AUDIOIO-ALSA","get_sample(): trying to read from a closed device!"));

  bytes_read = dl_snd_pcm_read(audio_fd, t->iobuf_uchar, format().align * buffersize());

  if (bytes_read < 0)
    throw(new ECA_ERROR("AUDIOIO-ALSA","get_sample(): read error!", ECA_ERROR::stop));

  //    t->length_in_samples(samples_read / format().align);
  //  else 
  t->iobuf_to_buffer(bytes_read / format().align, true, format().bits, format().channels, format().srate);
}

void ALSADEVICE::write_buffer(SAMPLE_BUFFER* t) {
  if (!is_open()) throw(new ECA_ERROR("AUDIOIO-ALSA","put_sample(): trying to write to a closed device!"));
  
  //  t->buffer_to_iobuf(format().bits, SAMPLE_BUFFER::is_system_littleendian, format().channels);

  t->buffer_to_iobuf(true, format().bits, format().channels, format().srate);
  dl_snd_pcm_write(audio_fd, t->iobuf_uchar, format().align * t->length_in_samples());
}

ALSADEVICE::~ALSADEVICE(void) { 
  close_device(); 

  if (io_mode() != si_read) {
    if (underruns != 0) {
      cerr << "(audioio-alsa) WARNING! While writing to ALSA-pcm device ";
      cerr << "C" << card_number << "D" << device_number;
      cerr << ", there were " << underruns << " underruns.\n";
    }
  }
  else {
    if (overruns != 0) {
      cerr << "(audioio-alsa) WARNING! While reading from ALSA-pcm device ";
      cerr << "C" << card_number << "D" << device_number;
      cerr << ", there were " << overruns << " overruns.\n";
    }
  }

  eca_alsa_unload_dynamic_support();
}

#endif // COMPILE_ALSA
