#ifndef _AUDIOFX_FILTER_H
#define _AUDIOFX_FILTER_H

#include <vector>

#include "audiofx.h"
#include "samplebuffer.h"

/**
 * Virtual base for filter effects.
 * @author Kai Vehmanen
 */
class EFFECT_FILTER : public EFFECT_BASE {

 public:
  virtual ~EFFECT_FILTER(void) { }
};

/**
 * Base class for butterworth filter effects.
 * 
 * Based on SPKit Butterworth algorithms. 
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_BW_FILTER : public EFFECT_FILTER {

private:

  int nm;
  SAMPLE_BUFFER::sample_type outputSample;

  vector<SAMPLE_BUFFER::sample_type> sin;
  vector<SAMPLE_BUFFER::sample_type> sout;

  vector<SAMPLE_BUFFER::single_type> a;
  vector<SAMPLE_BUFFER::single_type> b;

  void init_values(void);

 protected:

  inline void constant_a(int index, double newval) { a[index] = newval; }
  inline double constant_a(int index) { return(a[index]); }
  inline void constant_b(int index, double newval) { b[index] = newval; }
  inline double constant_b(int index) { return(b[index]); }
    
public:

  void process(SAMPLE_BUFFER::sample_type *insample);

  virtual EFFECT_BW_FILTER* clone(void) = 0;

  EFFECT_BW_FILTER(void) : sin(2), sout(2), a(3), b(2) { 
    init_values();
  }
};

/**
 * Bandpass filter
 *  
 * Based on SPKit Butterworth algorithms
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_BANDPASS: public EFFECT_BW_FILTER {

private:
    
  double center;
  double width;

  double C;
  double D;

public:

  string name(void) { return("Bandpass filter"); }

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  EFFECT_BANDPASS* clone(void)  { return new EFFECT_BANDPASS(*this); }
  EFFECT_BANDPASS (double centerf, double width);
  EFFECT_BANDPASS (void) {
    add_parameter("center-freq");
    add_parameter("width");
  }
};

/**
 * Bandreject filter
 *  
 * Based on SPKit Butterworth algorithms
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_BANDREJECT: public EFFECT_BW_FILTER {

private:
    
  double center;
  double width;

  double C;
  double D;

public:

  string name(void) { return("Bandreject filter"); }

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  EFFECT_BANDREJECT* clone(void)  { return new EFFECT_BANDREJECT(*this); }
  EFFECT_BANDREJECT (double centerf, double width);
  EFFECT_BANDREJECT (void) {
    add_parameter("center-freq");
    add_parameter("width");
  }
};

/**
 * Highpass filter
 *
 * Based on SPKit Butterworth algorithms
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_HIGHPASS : public EFFECT_BW_FILTER {
  
 private:

  double cutOffFreq;

  double C;
    
public:
    
  string name(void) { return("Highpass filter"); }

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  EFFECT_HIGHPASS* clone(void)  { return new EFFECT_HIGHPASS(*this); }
  EFFECT_HIGHPASS (double cutoff);
  EFFECT_HIGHPASS (void) {
    add_parameter("cutoff-freq");
  }
};

/**
 * Inverse comb filter
 *
 * The basic theory behind this can be found from Ken Steiglitz's book 
 * "A digital signal processing primer", page 77.
 */
class EFFECT_INVERSE_COMB_FILTER : public EFFECT_FILTER {

  deque<SAMPLE_BUFFER::sample_type> buffer;
  double laskuri;
  SAMPLE_BUFFER::sample_type temp;

  double C;
  double D;

public:

  string name(void) { return("Inverse comb filter"); }

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_INVERSE_COMB_FILTER* clone(void)  { return new EFFECT_INVERSE_COMB_FILTER(*this); }  
  EFFECT_INVERSE_COMB_FILTER (int delay_in_samples, double constant);
  EFFECT_INVERSE_COMB_FILTER (void) {
    add_parameter("delay");
    add_parameter("radius");
  }
};

/**
 * Lowpass filter
 *  
 * Based on SPKit Butterworth algorithms
 * (for more info, see http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_LOWPASS: public EFFECT_BW_FILTER {

private:

  double cutOffFreq;

  double C;

public:

  string name(void) { return("Lowpass filter"); }

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  void set_cutoff(double value, long int srate);

  EFFECT_LOWPASS* clone(void)  { return new EFFECT_LOWPASS(*this); }  
  EFFECT_LOWPASS (double cutoff);
  EFFECT_LOWPASS (void) {
    add_parameter("cutoff-freq");
  }
};

/**
 * A simple lowpass filter
 *                                              
 *   Algorithm:  1nd order filter.             
 *   From Fugue source code:                   
 *                                             
 *    output[N] = input[N] * A + input[N-1] * B
 *                                             
 *    A = 2.0 * pi * center                    
 *    B = exp(-A / frequency)
 */                                            
class EFFECT_LOWPASS_SIMPLE : public EFFECT_FILTER {

private:

  double cutOffFreq;
  double A, B;
  SAMPLE_BUFFER::sample_type outhist;
  SAMPLE_BUFFER::sample_type tempin, temphist;

public:

  string name(void) { return("Simple lowpass filter"); }

  void process(SAMPLE_BUFFER::sample_type *insample);

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  EFFECT_LOWPASS_SIMPLE* clone(void)  { return new EFFECT_LOWPASS_SIMPLE(*this); }
  EFFECT_LOWPASS_SIMPLE (double cutoff);
  EFFECT_LOWPASS_SIMPLE (void) {
    add_parameter("cutoff-freq");
  }
};

/**
 * Resonant bandpass filter
 */
class EFFECT_RESONANT_BANDPASS : public EFFECT_FILTER {

private:

  vector<SAMPLE_BUFFER::sample_type> outhist;
  
  double center;
  double width;
  
  double a, b, c, R;
  double pole_angle;

public:

  string name(void) { return("Resonant bandpass filter"); }

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONANT_BANDPASS* clone(void)  { return new EFFECT_RESONANT_BANDPASS(*this);
  }  
  EFFECT_RESONANT_BANDPASS (double centerf, double width);
  EFFECT_RESONANT_BANDPASS (void) : outhist(2) {
    add_parameter("center-freq");
    add_parameter("width");
  }
};

/**
 * Resonant lowpass filter
 *
 * Algorithm is based on a sample filter-routine (iir_filter) posted to comp.dsp.
 */
class EFFECT_RESONANT_LOWPASS : public EFFECT_FILTER {
    
  vector<SAMPLE_BUFFER::sample_type> outhist;
  vector<SAMPLE_BUFFER::sample_type> newhist;
    
  class TRIPLE_COEFS {
  public:
    double a0, a1, a2;       // numerator coefficients
    double b0, b1, b2;       // denominator coefficients
  };

  class FILTER_COEFS {
  public:
    double A, B, C, D;       // filter coefficients
  };
    
  vector<TRIPLE_COEFS> ProtoCoef;         // Filter prototype coefficients,
                               // for each filter section
  vector<FILTER_COEFS> Coef;
    
  double cutoff, Q, gain, gain_orig;
  double pi;
  double laskuri;

  double ad, bd, wp;      // for szxform()

  void szxform(int section);
  void refresh_values(void);

public:

  string name(void) { return("Resonant Lowpass filter"); }

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONANT_LOWPASS* clone(void)  { return new EFFECT_RESONANT_LOWPASS(*this); }  
  EFFECT_RESONANT_LOWPASS (double cutoff = 0.0,
			   double resonance = 1.0,
			   double gain = 1.0);
};

/**
 * Resonating bandpass filter
 *
 * Based on a second order all-pole (IIR) band-pass filter from SPKit 
 * (for more info, see: http://www.music.helsinki.fi/research/spkit)
 */
class EFFECT_RESONATOR : public EFFECT_FILTER {

private:

  int nm; // temporary value 
  SAMPLE_BUFFER::sample_type temp;        // temporary values
    
  double center;
  double width;

  vector<SAMPLE_BUFFER::single_type> cona;
  vector<SAMPLE_BUFFER::single_type> conb;

  vector<SAMPLE_BUFFER::sample_type> saout;

public:

  string name(void) { return("Resonator filter"); }

  void set_parameter(int param, double value);
  double get_parameter(int param) const;

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_RESONATOR* clone(void)  { return new EFFECT_RESONATOR(*this); }
  EFFECT_RESONATOR (double center, double width);
  EFFECT_RESONATOR (void) : cona(1), conb(2), saout(2) {
    add_parameter("center-freq");
    add_parameter("width");
  }
};

#endif
