#ifndef _AUDIO_GATE_H
#define _AUDIO_GATE_H

#include "eca-chainop.h"

class SAMPLE_BUFFER;

/**
 * Interface to gate effects. When gate is open, the signal is let 
 * through untouched. When closed, signal is turned off completely.
 */
class GATE_BASE : public CHAIN_OPERATOR {

private:

  bool gate_open;
  
protected:

  void gate_process(SAMPLE_BUFFER* sbuf);
  void close_gate(void) { gate_open = false; }
  void open_gate(void) { gate_open = true; }
  bool is_open(void) { return(gate_open); }
 
public:

  virtual GATE_BASE* clone(void) = 0;   
  virtual ~GATE_BASE(void) { }

  GATE_BASE(void) { close_gate(); }
};

/**
 *
 * A time crop gate. Initially the gate is closed, but is opened after 
 * 'open_at' seconds has elapsed. Gate remains open for 
 * 'duration' seconds. If 'duration' is 0, gate will stay open
 * forever.
 */
class TIME_CROP_GATE : public GATE_BASE {

public:

  // Functions returning info about effect and its parameters.
  // ---
  double get_parameter(int param) const;
  void set_parameter(int param, double value);

  string name(void) { return("Time crop gate"); }
 
  void process(SAMPLE_BUFFER* sbuf);

  TIME_CROP_GATE* clone(void)  { return new TIME_CROP_GATE(*this); }
  TIME_CROP_GATE (double open_at, double duration);
  TIME_CROP_GATE (void) : curtime(0.0) {
    add_parameter("open_at_sec");
    add_parameter("duration_sec");
    close_gate();
  }

private:

  double curtime, btime, etime; 
};

/**
 * A threshold open gate. When the average volume goes 
 * over 'threshold_openlevel', gate is opened. In the 
 * same way, when the average volume drops below 
 * 'threshold_closelevel', gate is closed. Either 
 * peak or RMS level is used for calculating average 
 * volume. The thresholds are given in percents. Unlike
 * noise gates, threshold gate is opened and closed 
 * only once. 
 */
class THRESHOLD_GATE : public GATE_BASE {

public:

  // Functions returning info about effect and its parameters.
  // ---
  double get_parameter(int param) const;
  void set_parameter(int param, double value);

  string name(void) { return("Threshold gate"); }
  virtual string id_string(void) { return("ge"); }

  void process(SAMPLE_BUFFER* sbuf);

  THRESHOLD_GATE* clone(void)  { return new THRESHOLD_GATE(*this); }
  THRESHOLD_GATE (double threshold_openlevel, double
		  threshold_closelevel,  bool use_rms = false);
  THRESHOLD_GATE (void) 
    : rms(false), is_opened(false), is_closed(false) {
    add_parameter("threshold_openlevel_%");
    add_parameter("threshold_closelevel_%");
    add_parameter("rms_enabled", 0.0, 1.0);
  }
  
private:

    double openlevel, closelevel, avolume;
    bool rms;
    bool is_opened, is_closed;
};

#endif









