// ------------------------------------------------------------------------
// eca-chainsetup.cpp: A class representing a group of chains and their
//                     configuration.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>

#include <string>
#include <cstring>
#include <algorithm>
#include <vector>

#include <kvutils.h>

#include "eca-resources.h"
#include "eca-session.h"

#include "eca-chain.h"

#include "eca-chainop.h"
#include "audiofx.h"
#include "audiogate.h"
#include "audiofx_compressor.h"
#include "audiofx_filter.h"
#include "audiofx_timebased.h"

#include "audioio.h"
#include "audioio-cdr.h"
#include "audioio-wave.h"
#include "audioio-oss.h"
// #include "audioio-oss_dma.h"
#include "audioio-ewf.h"
#include "audioio-mp3.h"
#include "audioio-alsa.h"
#include "audioio-alsalb.h"
#include "audioio-af.h"
#include "audioio-raw.h"

#include "osc-gen.h"
#include "osc-sine.h"
#include "midi-cc.h"

#include "eca-error.h"
#include "eca-debug.h"
#include "eca-chainsetup.h"

map<string, CHAIN_OPERATOR*> ECA_CHAINSETUP::chainop_map;
map<string, string> ECA_CHAINSETUP::chainop_prefix_map;

ECA_CHAINSETUP::ECA_CHAINSETUP(ECA_RESOURCES* ecarc, COMMAND_LINE& cline) {
  // require:
  assert(ecarc != 0);
  // ---

  setup_name = "command-line-setup";
  setup_filename = "";
  ecaresources = ecarc;

  set_defaults();

  cline.back_to_start();
  cline.next(); // skip program name
  string temp;
  while(cline.ready()) {
    temp = cline.next();
    if (temp == "") continue;
    ecadebug->msg(5, "(eca-chainseup) Adding \"" + temp + "\" to options.");
    options.push_back(temp);
  }

  interpret_options();

  // ensure:
  assert(buffersize != 0);
  // ---
}

ECA_CHAINSETUP::ECA_CHAINSETUP(ECA_RESOURCES* ecarc, const string&
			       setup_file, bool fromfile) {
  // require:
  assert(ecarc != 0);
  // ---

  setup_name = "";
  ecaresources = ecarc;

  set_defaults();

  if (fromfile) load_from_file(setup_file);
  if (setup_name == "") setup_name = setup_file;

  interpret_options();

  // ensure:
  assert(buffersize != 0);
  // ---
}

void ECA_CHAINSETUP::set_defaults(void) {
  register_default_chainops();

  active = false;
  mixmode = ep_mm_auto;
  output_openmode = si_readwrite;

  buffersize = atoi(ecaresources->resource("default-buffersize").c_str());

  if (ecaresources->resource("default-to-raisepriority")
      == "true") {
    raisepriority = true;
  }
  else {
    raisepriority = false;
  }

  if (ecaresources->resource("default-to-double-buffering")
      == "true") {
    double_buffering = true;
  }
  else {
    double_buffering = false;
  }
}

ECA_CHAINSETUP::~ECA_CHAINSETUP(void) {
  ecadebug->msg(1,"ECA_CHAINSETUP destructor!");
  //  disable();

  for(vector<AUDIO_IO*>::iterator q = inputs.begin(); q != inputs.end(); q++) {
    ecadebug->msg("(eca-chainsetup) Closing audio device/file \"" + (*q)->label() + "\".");
    delete *q;
  }
  inputs.resize(0);
  
  for(vector<AUDIO_IO*>::iterator q = outputs.begin(); q != outputs.end(); q++) {
    ecadebug->msg("(eca-chainsetup) Closing audio device/file \"" + (*q)->label() + "\".");
    delete *q;
  }
  outputs.resize(0);

  for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
    ecadebug->msg("(eca-chainsetup) Closing chain \"" + (*q)->name() + "\".");
    delete *q;
  }
  chains.resize(0);
}

void ECA_CHAINSETUP::register_chain_operator(const string& id_string,
					     CHAIN_OPERATOR* chainop) { 
  ECA_CHAINSETUP::chainop_map[id_string] = chainop;
  ECA_CHAINSETUP::chainop_prefix_map[chainop->name()] = id_string;
}

void ECA_CHAINSETUP::register_default_chainops(void) { 
  static bool defaults_registered = false;
  if (defaults_registered) return;
  defaults_registered = true;
  
  ECA_CHAINSETUP::register_chain_operator("ea", new EFFECT_AMPLIFY());
  ECA_CHAINSETUP::register_chain_operator("eas", new EFFECT_AMPLIFY_SIMPLE());
  ECA_CHAINSETUP::register_chain_operator("eca", new ADVANCED_COMPRESSOR());
  ECA_CHAINSETUP::register_chain_operator("ec", new EFFECT_COMPRESS());
  ECA_CHAINSETUP::register_chain_operator("ef1", new EFFECT_RESONANT_BANDPASS());
  ECA_CHAINSETUP::register_chain_operator("ef3", new EFFECT_RESONANT_LOWPASS());
  ECA_CHAINSETUP::register_chain_operator("efb", new EFFECT_BANDPASS());
  ECA_CHAINSETUP::register_chain_operator("efh", new EFFECT_HIGHPASS());
  ECA_CHAINSETUP::register_chain_operator("efi", new EFFECT_INVERSE_COMB_FILTER());
  ECA_CHAINSETUP::register_chain_operator("efl", new EFFECT_LOWPASS());
  ECA_CHAINSETUP::register_chain_operator("efr", new EFFECT_BANDREJECT());
  ECA_CHAINSETUP::register_chain_operator("efs", new EFFECT_RESONATOR());
  ECA_CHAINSETUP::register_chain_operator("enm", new EFFECT_NOISEGATE_MONO());
  ECA_CHAINSETUP::register_chain_operator("epp", new EFFECT_NORMAL_PAN());
  ECA_CHAINSETUP::register_chain_operator("etd", new EFFECT_DELAY());
  ECA_CHAINSETUP::register_chain_operator("etf", new EFFECT_FAKE_STEREO());
  ECA_CHAINSETUP::register_chain_operator("etr", new EFFECT_REVERB());
  ECA_CHAINSETUP::register_chain_operator("ev", new EFFECT_ANALYZE());
  ECA_CHAINSETUP::register_chain_operator("ezf", new EFFECT_DCFIND());
  ECA_CHAINSETUP::register_chain_operator("ezx", new EFFECT_DCFIX());

  ECA_CHAINSETUP::register_chain_operator("gc", new TIME_CROP_GATE());
  ECA_CHAINSETUP::register_chain_operator("ge", new THRESHOLD_GATE());
}

void ECA_CHAINSETUP::enable(void) {
  if (active == false) {
    
    for(vector<AUDIO_IO*>::iterator q = inputs.begin(); q != inputs.end(); q++) {
      (*q)->open_device();
      ecadebug->msg(open_info(*q));
    }
    
    for(vector<AUDIO_IO*>::iterator q = outputs.begin(); q != outputs.end(); q++) {
      (*q)->open_device();
      ecadebug->msg(open_info(*q));
    }
    update_chain_mappings();
  }
  active = true;
}

void ECA_CHAINSETUP::disable(void) {
  update_option_strings();
  if (active) {
    ecadebug->control_flow("Closing chainsetup \"" + name() + "\"");
    for(vector<AUDIO_IO*>::iterator q = inputs.begin(); q != inputs.end(); q++) {
      ecadebug->msg("(eca-chainsetup) Closing audio device/file \"" + (*q)->label() + "\".");
      (*q)->close_device();
      //      delete *q;
    }
    
    for(vector<AUDIO_IO*>::iterator q = outputs.begin(); q != outputs.end(); q++) {
      ecadebug->msg("(eca-chainsetup) Closing audio device/file \"" + (*q)->label() + "\".");
      (*q)->close_device();
      //      delete *q;
    }

    chains_assigned_to_idev.clear();
    chains_assigned_to_odev.clear();
    number_of_chains_assigned_to_idev.clear();
    number_of_chains_assigned_to_odev.clear();
    
    active = false;
  }
}


bool ECA_CHAINSETUP::is_valid(void) const {
  if (inputs.size() == 0) return(false);
  if (outputs.size() == 0) return(false);
  if (chains.size() == 0) return(false);

  return(true);
}

void ECA_CHAINSETUP::interpret_options(void) {
  vector<string>::const_iterator p = options.begin();
  while(p != options.end()) {
    interpret_general_option(*p);
    interpret_mix_mode(*p);
    ecadebug->msg(5, "(eca-chainsetup) Interpreting general option \""
		  + *p + "\".");
    ++p;
  }

  string temp, another;

  active_sinfo_explicit = false;
  p = options.begin();
  while(p != options.end()) {
    temp = *p;
    ++p;
    if (p == options.end()) {
      another = "";
      --p;
    }
    else {
      another = *p;
      if (another != "" && another[0] == '-') {
	--p;
	another = "";
      }
    }
    ecadebug->msg(5, "(eca-chainsetup) Interpreting setup, with args \""
		  + temp + "\", \"" + another + "\".");
    interpret_setup_state(temp);
    interpret_audioio_device(temp, another);
    interpret_effect_preset(temp);
    interpret_chain_operator(temp);
    interpret_controller(temp);
    ++p;
  }

  if (inputs.size() == 0) {
    // No -i[:] options specified; let's try to be artificially intelligent 
    p = options.begin();
    while(p != options.end()) {
      if ((*p) != "" && (*p)[0] != '-') {
	interpret_audioio_device("-i", *p);
	break;
      }
      ++p;
    }
  }
  if (inputs.size() > 0 && outputs.size() == 0) {
    // No -o[:] options specified; let's use the default output 
    vector<string> temp;
    temp.push_back("all");
    set_active_chains(temp);
    interpret_audioio_device("-o", ecaresources->resource("default-output"));
  }
}

void ECA_CHAINSETUP::interpret_general_option (const string& argu) {
  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'b':
    {
      buffersize =  atoi(get_argument_number(1, argu).c_str());
      MESSAGE_ITEM mitemb;
      mitemb << "(eca-chainsetup) Setting buffersize to (samples) " << buffersize << ".";
      ecadebug->msg(0, mitemb.to_string()); 
      break;
    }

  case 'n':
    {
      setup_name = get_argument_number(1, argu);
      ecadebug->msg("(eca-chainsetup) Setting chainsetup name to \""
		    + setup_name + "\"");
      break;
    }

  case 'r':
    {
      ecadebug->msg("(eca-chainsetup) Raised-priority mode enabled.");
      raisepriority = true;
      break;
    }

  case 'x':
    {
      ecadebug->msg("(eca-chainsetup) Truncating outputs.");
      output_openmode = si_write;
      break;
    }

  case 'z':
    {
      if (get_argument_number(1, argu) == "db") {
	ecadebug->msg("(eca-chainsetup) Using double-buffering with all audio inputs that support it.");
	double_buffering = true;
      }
      break;
    }

  default: { }
  }
}

void ECA_CHAINSETUP::interpret_region (const string& argu) {
  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'y': 
    { 
      if (argu.size() < 3) return;
      switch(argu[2]) {
      case ':': 
	{
	  region_start_pos_in_seconds(atof(get_argument_number(1, argu).c_str()));
	  region_end_pos_in_seconds(atof(get_argument_number(2, argu).c_str()));
	  break;
	}
	
      case 'l': 
	{
	  enable_region_looping();
	  break;
	}

      case 'p': 
	{
	  enable_region_sfx_control();
	  break;
	}
      }
      break;
    }
    break;
  }
}


void ECA_CHAINSETUP::interpret_mix_mode (const string& argu) {
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECA-TEXT", "empty argument", retry));
  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'm':      // mixmode
    {
      string temp = get_argument_number(1, argu);
      if (temp == "auto") {
	mixmode = ep_mm_auto;
	ecadebug->msg("(eca-chainsetup) Mix-mode is selected automatically.");
      }
      else if (temp == "mthreaded") {
	ecadebug->msg("(eca-chainsetup) Multithreaded mixmode selected.");
	mixmode = ep_mm_mthreaded;
      }
      else if (temp == "simple") {
	ecadebug->msg("(eca-chainsetup) Simple mixmode selected.");
	mixmode = ep_mm_simple;
      }
      else if (temp == "normal") {
	ecadebug->msg("(eca-chainsetup) Simple mixmode selected.");
	mixmode = ep_mm_normal;
      }
    }
    break;

  default: { }
  }
}

void ECA_CHAINSETUP::interpret_setup_state (const string& argu) {
  if (argu.size() < 2) return;  
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'a':
    {
      active_chainids = get_arguments(argu);
      add_new_chains(active_chainids);
      MESSAGE_ITEM mtempa;
      mtempa << "(eca-chainsetup) Active chain ids: ";
      for (vector<string>::const_iterator p = active_chainids.begin(); p != active_chainids.end(); p++) {
	mtempa << *p << " ";
      }
      ecadebug->msg(1, mtempa.to_string());
      break;
    }
  case 'f':
    {
      active_sinfo.bits = (unsigned short int) atoi(get_argument_number(1, argu).c_str());
      active_sinfo.channels = (unsigned short int) atoi(get_argument_number(2, argu).c_str());
      active_sinfo.srate = (unsigned short int) atoi(get_argument_number(3, argu).c_str());
      active_sinfo_explicit = true;
      
      MESSAGE_ITEM ftemp;
      ftemp << "(eca-chainsetup) Set active format to (bits/channels/srate): ";
      ftemp << (int)active_sinfo.bits << "/" << (int)active_sinfo.channels << "/" << active_sinfo.srate;
      ecadebug->msg(ftemp.to_string());
      break;
    }
  default: { }
  }
  return; // to avoid a internal compiler error
}

void ECA_CHAINSETUP::interpret_effect_preset (const string& argu) {
  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'p':
    {
      if (argu.size() < 3) return;  
      switch(argu[2]) {
      case 'm':
	{
	  break;
	}

      case 's': 
	{
	  add_singlechain_preset(get_argument_number(1,argu));
	  break;
	}
	
      default: { }
      }
      break;
    }
  default: { }
  }
}

void ECA_CHAINSETUP::add_singlechain_preset(const string&  preset_name) {
  ecadebug->msg(1,"(eca-chainsetup) Opening sc-preset file.");
  string filename =
    ecaresources->resource("resource-directory") + "/" + ecaresources->resource("resource-file-single-effect-presets");
  ifstream fin (filename.c_str());

  if (!fin) {
    throw(new ECA_ERROR("ECA_CHAINSETUP", "Unable to open single-chain preset file: " + filename + "."));
    return;
  }
  
  ecadebug->msg(1,"(eca-chainsetup) Starting to process sc-preset file. Trying to find \"" + preset_name + "\".");
  string sana;
  while(fin >> sana) {
    if (sana.size() > 0 && sana[0] == '#') {
      while(fin.get() != '\n' && fin.eof() == false);
      continue;
    }
    else {
      ecadebug->msg(5, "(eca-chainsetup) Next sc-preset is " + sana + ".");
      if (preset_name == sana) {
	ecadebug->msg(5, "(eca-chainsetup) Found the right preset!");

	getline(fin,sana);
	vector<string> chainops = string_to_words(sana);
	vector<string>::const_iterator p = chainops.begin();
	while (p != chainops.end()) {
	  ecadebug->msg(5, "(eca-chainsetup) Adding chainop " + *p + ".");
	  interpret_chain_operator(*p);
	  interpret_controller(*p);
	  ++p;
	}
	break;
      }
      else 
	while(fin.get() != '\n' && fin.eof() == false);
    }
  }

  fin.close();
}

void ECA_CHAINSETUP::interpret_audioio_device (const string& argu, const string& argu_param) {
  if (argu.size() == 0) return;
  if (argu[0] != '-') return;
  
  string tname = get_argument_number(1, argu);
  if (tname == "") tname = argu_param;                    // -i and -o
  else if (tname[0] == '-') tname = argu_param;           // -i and -o 
  else {                                                  // -i:
	string temp = tname;
	to_lowercase(temp);
	if (temp == "alsa" ||
	    temp == "alsalb") {                             // -i:alsa,c,d
	  string::const_iterator p = argu.begin();        // -o:alsa,c,d
	  ++p; ++p; ++p;
	  tname = string(p, argu.end());
	}
  }
  ecadebug->msg(5,"(eca-chainsetup) adding file \"" + tname + "\".");
  if (argu.size() < 2) return;  
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'i':
    {
      ecadebug->control_flow("Chainsetup/Adding a new input");
      AUDIO_IO* aiod;
      if (active_sinfo_explicit == true) {
	aiod = new_aio_device(tname, si_read, active_sinfo, buffersize);
      }
      else {
	AIO_PARAMS default_sinfo;
	default_sinfo.bits = 16;
	default_sinfo.srate = 44100;
	default_sinfo.channels = 2;
	default_sinfo.byte_second = 176400;

	aiod = new_aio_device(tname, si_read, default_sinfo, buffersize);
      }
	
      add_input(aiod);
	
      //      MESSAGE_ITEM itemp;
      //      itemp << "Infile (" << inputs.size() - 1 << ") \"" << inputs.back()->label() << "\".";
      //      ecadebug->msg(itemp.to_string());
      break;
    }

  case 'o':
    {
      ecadebug->control_flow("Chainsetup/Adding a new output");
      AUDIO_IO* aiod;
      if (active_sinfo_explicit == true)
	aiod = new_aio_device(tname, output_openmode, active_sinfo, buffersize);
      else {
	AIO_PARAMS default_sinfo;
	default_sinfo.bits = 16;
	default_sinfo.srate = 44100;
	default_sinfo.channels = 2;
	default_sinfo.byte_second = 176400;
	aiod = new_aio_device(tname, output_openmode, default_sinfo, buffersize);
      }
	
      add_output(aiod);
	
      //      MESSAGE_ITEM otemp;
      //      otemp << "Outfile (" << outputs.size() - 1 << ") \"" << outputs.back()->label() << "\".";
      //      ecadebug->msg(otemp.to_string());
      break;
    }

  default: { }
  }
}

int ECA_CHAINSETUP::get_type_from_extension(const string& filename) {
  int typevar;
  string teksti = filename;
  to_lowercase(teksti);

  if (strstr(teksti.c_str(),".wav") != 0) typevar = TYPE_WAVE;
  else if (strstr(teksti.c_str(),".cdr") != 0) typevar = TYPE_CDR;
  else if (strstr(teksti.c_str(),".ewf") != 0) typevar = TYPE_EWF;
  else if (strstr(teksti.c_str(),"/dev/dsp") != 0) typevar = TYPE_OSS;
  else if (strstr(teksti.c_str(),".ess") != 0) typevar = TYPE_ESS;
  else if (strstr(teksti.c_str(),".mp3") != 0) typevar = TYPE_MP3;
  else if (strstr(teksti.c_str(),".mp2") != 0) typevar = TYPE_MP3;
  else if (strstr(teksti.c_str(),"/dev/snd/pcm") != 0) typevar = TYPE_ALSAFILE;
  else if (strstr(teksti.c_str(),".aif") != 0) typevar = TYPE_AUDIOFILE;
  else if (strstr(teksti.c_str(),".au") != 0) typevar = TYPE_AUDIOFILE;
  else if (strstr(teksti.c_str(),".snd") != 0) typevar = TYPE_AUDIOFILE;
  else if (strstr(teksti.c_str(),".raw") != 0) typevar = TYPE_RAWFILE;
  else if (strstr(teksti.c_str(),"alsalb") != 0) typevar = TYPE_ALSALOOPBACK;
  else if (strstr(teksti.c_str(),"alsa") != 0) typevar = TYPE_ALSA;
  else if (teksti == "stdin") typevar = TYPE_STDIN;
  else if (teksti == "stdout") typevar = TYPE_STDOUT;
  else if (strstr(teksti.c_str(),"null") != 0) typevar = TYPE_NULL;
  else typevar = TYPE_UNKNOWN;

  return(typevar);
}

AUDIO_IO* ECA_CHAINSETUP::new_aio_device(const string& argu, const SIMODE mode, const AIO_PARAMS& format, int buffersize) {

  if (argu.size() == 0) {
    throw(new ECA_ERROR("ECA-CHAINSETUP","Tried to create a audio device with null filename."));
  }
  string tname = get_argument_number(1, argu);

  AUDIO_IO* main_file;
  int type = get_type_from_extension(tname);
  switch(type) {
  case TYPE_WAVE:
    main_file = new WAVEFILE (tname, mode, format, buffersize, double_buffering);
    break;
    
  case TYPE_CDR:
    main_file = new CDRFILE (tname, mode, format, buffersize);
    break;
    
  case TYPE_OSS:
#ifdef COMPILE_OSS
    if (mode != si_read) 
      main_file = new OSSDEVICE (tname, si_write, format, buffersize);
    else
      main_file = new OSSDEVICE (tname, mode, format, buffersize);
#endif
    break;
    
  case TYPE_EWF:
    main_file = new EWFFILE (tname, mode, format, buffersize);
    break;
    
    //  case TYPE_OSSDMA:
    // #ifdef COMPILE_OSS
    //    main_file = new OSSDMA (tname, mode, format, buffersize);
    // #endif
    //    break;
    
  case TYPE_MP3:
    if (mode != si_read) 
      main_file = new MP3FILE (tname, si_write, format, buffersize);
    else
      main_file = new MP3FILE (tname, mode, format, buffersize);
    break;

  case TYPE_ALSAFILE:
    {
      string cardstr,devicestr;
      string::const_iterator p = tname.begin();
      while(p != tname.end() && *p != 'C') ++p;
      ++p;
      while(p != tname.end() && isdigit(*p)) {
	cardstr += " ";
	cardstr[cardstr.size() - 1] = *p;
	++p;
      }
      while(p != tname.end() && *p != 'D') ++p;
      ++p;
      while(p != tname.end() && isdigit(*p)) {
	devicestr += " ";
	devicestr[devicestr.size() - 1] = *p;
	++p;
      }
      
      int card = atoi(cardstr.c_str());
      int device = atoi(devicestr.c_str());

#ifdef COMPILE_ALSA
      if (mode != si_read)
	main_file = new ALSADEVICE (card, device, si_write, format,
				    buffersize);
      else
	main_file = new ALSADEVICE (card, device, mode, format, buffersize);
#endif
      break;
    }

  case TYPE_ALSALOOPBACK:
    {
      int card = atoi(get_argument_number(2, argu).c_str());
      int device = atoi(get_argument_number(3, argu).c_str());
      bool loop_mode = true;
      
      if (get_argument_number(4, argu) == "c") {
	loop_mode = false;
      }

#ifdef COMPILE_ALSA
      if (mode != si_read)
	main_file = new ALSALBDEVICE (card, device, si_write, format,
				    buffersize, loop_mode);
      else
	main_file = new ALSALBDEVICE (card, device, mode, format, buffersize, loop_mode);
#endif
      break;
    }

  case TYPE_ALSA:
    {
      int card = atoi(get_argument_number(2, argu).c_str());
      int device = atoi(get_argument_number(3, argu).c_str());

#ifdef COMPILE_ALSA
      if (mode != si_read)
	main_file = new ALSADEVICE (card, device, si_write, format,
				    buffersize);
      else
	main_file = new ALSADEVICE (card, device, mode, format, buffersize);
#endif
      break;
    }

  case TYPE_AUDIOFILE:
    {
#ifdef COMPILE_AF
      if (mode != si_read) 
	main_file = new AUDIOFILE_INTERFACE (tname, si_write, format, buffersize);
      else
	main_file = new AUDIOFILE_INTERFACE (tname, mode, format, buffersize);
#endif
      break;
    }

  case TYPE_RAWFILE:
    {
      main_file = new RAWFILE (tname, mode, format, buffersize, double_buffering);
      break;
    }

  case TYPE_STDIN:
    {
      main_file = new RAWFILE ("-", si_read, format, buffersize);
      break;
    }

  case TYPE_STDOUT:
    {
      main_file = new RAWFILE ("-", si_write, format, buffersize);
      break;
    }

  case TYPE_NULL:
    {
      main_file = new NULLFILE (mode);
      break;
    }

  default: 
    {
    throw(new ECA_ERROR("ECA_CHAINSETUP", "unknown file format; unable to open file " + tname + "."));
    }
  }
  return(main_file);
}

string ECA_CHAINSETUP::get_argument_prefix(const string& argu) {
  // require:
  assert(argu.find('-') != string::npos);
  // ---

  string::const_iterator b = find(argu.begin(), argu.end(), '-');
  string::const_iterator e = find(argu.begin(), argu.end(), ':');

  if (b != argu.end()) {
    ++b;
    if (b !=  argu.end()) {
      ecadebug->msg(6, "(eca-chainsetup) Returning argument prefix " + string(b,e)); 
      return(string(b,e));
    }
  }

  ecadebug->msg(2, "(eca-chainsetup) No argument prefix found.\n");

  return("");

  // ensure:
  assert(argu.size() >= 0);
  // ---
}

void ECA_CHAINSETUP::interpret_chain_operator (const string& argu) {
  if (argu.size() < 2 ||
      argu[0] != '-') 
    return;
  
  string prefix = get_argument_prefix(argu);

  MESSAGE_ITEM otemp;
  
  map<string, CHAIN_OPERATOR*>::const_iterator p = chainop_map.find(prefix);
  if (p != chainop_map.end()) {
    CHAIN_OPERATOR* cop = chainop_map[prefix];
    ecadebug->control_flow("Chainsetup/Adding chain operator \"" +
			   cop->name() + "\"");
    //    otemp << "(eca-chainsetup) Adding effect " << cop->name();
    otemp << "Setting parameters: ";
    for(int n = 0; n < cop->number_of_params(); n++) {
      cop->set_parameter(n + 1, atof(get_argument_number(n + 1, argu).c_str()));
      otemp << cop->get_parameter_name(n + 1) << " = ";
      otemp << cop->get_parameter(n +1);
      if (n + 1 < cop->number_of_params()) otemp << ", ";
    }
    ecadebug->msg(otemp.to_string());
    add_chainop(chainop_map[prefix]);
  }
}

void ECA_CHAINSETUP::interpret_controller(const string& argu) {
  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'k':
    {
    ecadebug->control_flow("Chainsetup/Adding controller source");

    int fpar = atoi(get_argument_number(1, argu).c_str());
    double lowr = atof(get_argument_number(2, argu).c_str());
    double highr = atof(get_argument_number(3, argu).c_str());
    double step = (double)buffersize / SAMPLE_BUFFER::sample_rate;

    if (argu.size() < 3) return;
    switch(argu[2]) {
    case 'a':
      // KTYPE_VOLUME;
      break;

    case 'f':
      {
      //	aparam->active_fx->kenve.back()->type =
      //	KTYPE_FILE_ENVELOPE;
        string genosc_conf = ecaresources->resource("resource-directory") + "/" + ecaresources->resource("resource-file-genosc-envelopes");
	add_gcontroller(new GENERIC_OSCILLATOR(step,
								atof(get_argument_number(4, argu).c_str()), atoi(get_argument_number(5, argu).c_str()), genosc_conf), fpar, lowr, highr);
      break;
      }
    case 'm':
      // KTYPE_MIDI;
	add_gcontroller(new
					 MIDI_CONTROLLER(atoi(get_argument_number(4, argu).c_str()), atoi(get_argument_number(5, argu).c_str())), fpar, lowr, highr);
      break;
    case 'o':
      // KTYPE_OSCILLATOR;
      if (argu.size() < 4) return;
      switch(argu[3]) {
      case 'q':
	// KTYPE_OSCILLATOR_SQUARE
	break;
      case 's':
	add_gcontroller(new
					 SINE_OSCILLATOR(step, atof(get_argument_number(4, argu).c_str()), atof(get_argument_number(5, argu).c_str())), fpar, lowr, highr);
	break;
      case 't':
	// KTYPE_OSKILLATOR_TRIANGLE
	break;
      }
      break;
    case 'p':
      // KTYPE_PITCH;
      break;
    default:
      throw(new ECA_ERROR("ECA_CHAINSETUP", "invalid controller parameters", ECA_ERROR::notice));
    }
    break;
    }
    
  default: { }
  }
}

void ECA_CHAINSETUP::add_default_chain() {
  // require:
  assert(buffersize >= 0 && chaincount >= 0);
  // ---

  chaincount++;
  chains.push_back(new CHAIN(buffersize, chaincount));
  chains.back()->name("default");
  ecadebug->msg(1,"add_default_chain() ");
  active_chainids.push_back("default");

  // ensure:
  assert(chains.back()->name() == "default" &&
	 active_chainids.back() == "default");
  // ---
}


void ECA_CHAINSETUP::add_new_chains(const vector<string>& newchains) {
  for(vector<string>::const_iterator p = newchains.begin(); p != newchains.end(); p++) {
    bool exists = false;
    if (*p == "all") continue;
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->name()) exists = true;
    }
    if (exists == false) {
      chaincount++;
      //      chains.push_back(new CHAIN(buffersize, chaincount, (*p)));
      //      chains.push_back(new CHAIN(buffersize, chaincount, (*p)));
      chains.push_back(new CHAIN(buffersize, chaincount));
      chains.back()->name(*p);
      ecadebug->msg(1,"add_new_chains() added chain " + *p);
    }
  }
  update_chain_mappings();
}

void ECA_CHAINSETUP::remove_chain(const string& name) {
  for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
    if (name == (*q)->name()) {
      delete *q;       
      chains.erase(q);
      break;
    }
  }
  update_chain_mappings();
}

void ECA_CHAINSETUP::add_gcontroller(CONTROLLER_SOURCE* csrc, int fxparam, double low, double high) {
  GCONTROLLER* gtmp;
  for(vector<string>::const_iterator a = active_chainids.begin(); a != active_chainids.end(); a++) {
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*a == (*q)->name() || *a == "all") {
	if ((*q)->chainops.size() == 0) continue;
	gtmp = new GCONTROLLER((*q)->chainops.back(),
			       csrc,
			       (char)fxparam, low, high);
	(*q)->gcontrollers.push_back(gtmp);
      }
    }
  }
}

void ECA_CHAINSETUP::add_chainop(CHAIN_OPERATOR* cotmp) {
  if (chains.size() == 0) add_default_chain();

  for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
    for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
      if (*p == (*q)->name() || *p == "all") {
	ecadebug->msg(1, "Adding chainop to chain " + (*q)->name() + ".");
	(*q)->chainops.push_back(cotmp->clone());
      }
    }
  }
}

void ECA_CHAINSETUP::update_chain_mappings(void) {
  vector<AUDIO_IO*>::size_type a = 0; 
  vector<CHAIN*>::const_iterator q;
  int count = 0;

  for(a = 0; a < inputs.size(); a++) {
    count = 0;
    chains_assigned_to_idev[a].resize(0);
    q = chains.begin();
    while(q != chains.end()) {
      if (a == (*q)->inputid) {
	count++;
	chains_assigned_to_idev[a].push_back((*q)->name());
      }
      ++q;
    }
    number_of_chains_assigned_to_idev[a] = count;
  }    

  for(a = 0; a < (int)outputs.size(); a++) {
    count = 0;
    chains_assigned_to_odev[a].resize(0);
    q = chains.begin();
    while(q != chains.end()) {
      if (a == (*q)->outputid) {
	count++;
	chains_assigned_to_odev[a].push_back((*q)->name());
      }
      ++q;
    }
    number_of_chains_assigned_to_odev[a] = count;
  }    
}

void ECA_CHAINSETUP::add_input(AUDIO_IO* aio) {
  if (chains.size() == 0) add_default_chain();
  inputs.push_back(aio);
  attach_iodev_to_active_chains(aio->label());
}

void ECA_CHAINSETUP::add_output(AUDIO_IO* aiod) {
  if (chains.size() == 0) add_default_chain();
  outputs.push_back(aiod);
  attach_iodev_to_active_chains(aiod->label());
}

string ECA_CHAINSETUP::open_info(const AUDIO_IO* aio) {
  string temp = "(eca-chainsetup) Opening \"" + aio->label();
  temp += "\" for ";
  if (aio->io_mode() == si_read) temp += "reading";
  if (aio->io_mode() == si_write) temp += "writing";
  if (aio->io_mode() == si_readwrite) temp += "reading and writing";
  temp += ".\n";
  temp += aio->format_info();
  return(temp);
}

void ECA_CHAINSETUP::remove_audio_device(const string& label) { 
  vector<AUDIO_IO*>::iterator ci;
  int count = 0;

  ci = inputs.begin();
  while (ci != inputs.end()) {
    if ((*ci)->label() == label) {
      delete *ci;
      //      inputs.erase(ci);
      (*ci) = new NULLFILE(si_read);

      //      ecadebug->msg("(eca-chainsetup) Removing input " + label + ".");

      vector<CHAIN*>::iterator q = chains.begin();
      while(q != chains.end()) {
	if ((*q)->inputid == count) (*q)->inputid = -1;
	//	else if ((*q)->inputid > count) ((*q)->inputid)--;
	++q;
      }
    }
    ++ci;
    ++count;
  }


  count = 0;
  ci = outputs.begin();
  while (ci != outputs.end()) {

    if ((*ci)->label() == label) {

      delete *ci;
      (*ci) = new NULLFILE(si_write);
      //      outputs.erase(ci);

      //      ecadebug->msg("(eca-chainsetup) Removing output " + label + ".");

      vector<CHAIN*>::iterator q = chains.begin();

      while(q != chains.end()) {

	if ((*q)->outputid == count) (*q)->outputid = -1;
	//	else if ((*q)->outputid > count) ((*q)->outputid)--;
	++q;
      }
    }

    ++ci;
    ++count;
  }
  update_chain_mappings();
}

void ECA_CHAINSETUP::load_from_file(const string& filename) { 
  ifstream fin (filename.c_str());
  if (!fin) throw(new ECA_ERROR("ECA_CHAINSETUP", "Couldn't open setup read file: \"" + filename + "\".", ECA_ERROR::retry));

  string temp, another;
  
  while(fin >> temp) {
    ecadebug->msg(5, "(eca-chainseup) Adding \"" + temp + "\" to options (loaded from \"" + filename + "\".");
    options.push_back(temp);
  }
  fin.close();

  setup_filename = filename;
}

void ECA_CHAINSETUP::save(void) { save_to_file(setup_filename); }
void ECA_CHAINSETUP::save_to_file(const string& filename) {
  update_option_strings();

  ofstream fout (filename.c_str());
  if (!fout) {
    cerr << "Going to throw an exception...\n";
    throw(new ECA_ERROR("ECA_CHAINSETUP", "Couldn't open setup save file: \"" +
  			filename + "\".", ECA_ERROR::retry));
  }

  fout << options_general << "\n";
  fout << options_inputs << "\n";
  fout << options_outputs << "\n";
  fout << options_chains << "\n";

  fout.close();
}

void ECA_CHAINSETUP::update_option_strings(void) {
  options_general = general_options_to_string();
  options_inputs = inputs_to_string();
  options_outputs = outputs_to_string();
  options_chains = chains_to_string();
  
  while(options.size() > 0) options.pop_back();

  string temp = options_general + 
                options_inputs + 
                options_outputs +
                options_chains;

  options = string_to_words(temp);
}

string ECA_CHAINSETUP::general_options_to_string(void) { 
  MESSAGE_ITEM t; 

  t << "-b:" << buffersize;

  t << " -n:" << setup_name;

  switch(mixmode) {
  case ep_mm_simple: {
    t << " -m:simple";
    break;
  } 
  case ep_mm_normal: {
    t << " -m:normal";
    break;
  } 
  case ep_mm_mthreaded: {
    t << " -m:mthreaded";
    break;
  } 
  default: { }
  }

  if (raisepriority) t << " -r";
  if (output_openmode == si_write) t << " -x";
  if (double_buffering) t << " -z:db";

  if (region_active()) {
    t << " -y:" << region_start_pos_in_seconds()
      << ","    << region_end_pos_in_seconds();

    if (region_looping()) t << " -yl";
    if (region_sfx_control()) t << " -yp";
  }

  return(t.to_string());
}

string ECA_CHAINSETUP::inputs_to_string(void) { 
  MESSAGE_ITEM t; 

  CHAIN::aio_id_type p = 0;
  while (p < inputs.size()) {
    t << "-a:";
    vector<string> c = get_connected_chains_to_input(p);
    vector<string>::const_iterator cp = c.begin();
    while (cp != c.end()) {
      t << *cp;
      ++cp;
      if (cp != c.end()) t << ",";
    }
    t << " ";
    t << audioio_to_string(inputs[p], "i") << "\n";
    ++p;
  }

  return(t.to_string());
}

string ECA_CHAINSETUP::outputs_to_string(void) { 
  MESSAGE_ITEM t; 

  vector<AUDIO_IO*>::size_type p = 0;
  while (p < outputs.size()) {
    t << "-a:";
    vector<string> c = get_connected_chains_to_output(p);
    vector<string>::const_iterator cp = c.begin();
    while (cp != c.end()) {
      t << *cp;
      ++cp;
      if (cp != c.end()) t << ",";
    }
    t << " ";
    t << audioio_to_string(outputs[p], "o") << "\n";
    ++p;
  }

  return(t.to_string());
}

string ECA_CHAINSETUP::audioio_to_string(AUDIO_IO* aiod, const string& direction) {
  MESSAGE_ITEM t;

  t << "-f:" << (int)aiod->format().bits << "," <<
    (int)aiod->format().channels << ","  << (int)aiod->format().srate;
  t << " -" << direction << " ";
  t << aiod->label();

  return(t.to_string());
}

string ECA_CHAINSETUP::chains_to_string(void) { 
  MESSAGE_ITEM t; 

  vector<CHAIN*>::size_type p = 0;
  while (p < chains.size()) {
    t << "-a:" << chains[p]->name() << " ";
    CHAIN::aio_id_type q = 0;
    while (q < chains[p]->chainops.size()) {
      t << chainop_to_string(chains[p]->chainops[q], chains[p]->gcontrollers) << " ";
      ++q;
    }
    ++p;
    if (p < chains.size()) t << "\n";
  }

  return(t.to_string());
}

string ECA_CHAINSETUP::chainop_to_string(CHAIN_OPERATOR* chainop,
					 vector<GCONTROLLER*>& gctrls) {
  MESSAGE_ITEM t;

  //  if (dynamic_cast<EFFECT_AMPLIFY_SIMPLE*> (chainop))

  //  t << "-" << ECAchainop->id_string();
  t << "-" << ECA_CHAINSETUP::chainop_prefix_map[chainop->name()];
  if (chainop->number_of_params() > 0) t << ":";
  for(int n = 0; n < chainop->number_of_params(); n++) {
    t << chainop->get_parameter(n + 1);
    if (n + 1 < chainop->number_of_params()) t << ",";
  }

  vector<GCONTROLLER*>::size_type p = 0;
  while (p < gctrls.size()) {
    if (chainop == gctrls[p]->chainop_pointer()) {
      t << " " << gcontroller_to_string(gctrls[p]);
    }
    ++p;
  } 

  return(t.to_string());
}

string ECA_CHAINSETUP::gcontroller_to_string(GCONTROLLER* gctrl) {
  MESSAGE_ITEM t; 
  t << "-" << gctrl->ctrl_source_pointer()->id_string() << ":";
  t << (int)gctrl->param_number() << ",";
  t << (int)gctrl->low_range_limit() << ",";
  t << (int)gctrl->high_range_limit();
  if (gctrl->ctrl_source_pointer()->number_of_params() > 0) t << ",";
  for(int n = 0; n < gctrl->ctrl_source_pointer()->number_of_params(); n++) {
    t << gctrl->ctrl_source_pointer()->get_parameter(n + 1);
    if (n + 1 < gctrl->ctrl_source_pointer()->number_of_params()) t << ",";
  }

  return(t.to_string());
}

const CHAIN* ECA_CHAINSETUP::get_chain_with_name(const string& name) const {
  vector<CHAIN*>::const_iterator p = chains.begin();
  while(p != chains.end()) {
    if ((*p)->name() == name) return(*p);
    ++p;
  }
  return(0);
}

vector<string> ECA_CHAINSETUP::get_connected_chains_to_iodev(const
							     string&
							     filename)
  {
  vector<AUDIO_IO*>::size_type p;

  p = 0;
  while (p < inputs.size()) {
    if (inputs[p]->label() == filename)
      return(chains_assigned_to_idev[p]);
    ++p;
  }

  p = 0;
  while (p < outputs.size()) {
    if (outputs[p]->label() == filename)
      return(chains_assigned_to_odev[p]);
    ++p;
  }
  return(*(new vector<string> (0)));
}

void ECA_CHAINSETUP::attach_iodev_to_active_chains(const string& filename) {
  string temp;

  vector<AUDIO_IO*>::size_type c;

  c = 0;
  while (c < inputs.size()) {
    if (inputs[c]->label() == filename) {
      temp += "(eca-chainsetup) Assigning file to chains:";
      for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
	if (*p == "all") temp += " all";
	for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
	  if (*p == (*q)->name() || *p == "all") {
	    (*q)->inputid = c;
	    if (*p != "all") 
	      temp += " " + *p;
	  }
	}
      }
    }
    ++c;
  }

  c = 0;
  while (c < outputs.size()) {
    if (outputs[c]->label() == filename) {
      temp += "(eca-chainsetup) Assigning file to chains:";
      for(vector<string>::const_iterator p = active_chainids.begin(); p!= active_chainids.end(); p++) {
	if (*p == "all") temp += " all";
	for(vector<CHAIN*>::iterator q = chains.begin(); q != chains.end(); q++) {
	  if (*p == (*q)->name() || *p == "all") {
	    (*q)->outputid = c;
	    if (*p != "all") 
	      temp += " " + *p;
	  }
	}
      }
    }
    ++c;
  }
  update_chain_mappings();
  ecadebug->msg(temp);
}
