// ------------------------------------------------------------------------
// eca-main.cpp: Main processing engine
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <vector>
#include <map>
#include <ctime>
#include <unistd.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <pthread.h>

#include <kvutils.h>

#include "eca-chain.h"
#include "eca-session.h"
#include "samplebuffer.h"
#include "eca-chainop.h"
#include "audioio.h"
#include "eca-error.h"
#include "eca-debug.h"
#include "eca-main.h"

COMMAND_QUEUE ecasound_cqueue;

ECA_PROCESSOR::ECA_PROCESSOR(ECA_SESSION* params) 
{
  eparams = params;
  init();
}

ECA_PROCESSOR::~ECA_PROCESSOR(void) {
  ecadebug->msg(1,"ECA_PROCESSOR destructor!");

  stop();
  eparams->status(ep_status_notready);

  pthread_cancel(chain_thread);
  pthread_join(chain_thread,NULL);

  ecadebug->control_flow("Engine/Exiting");
}

void ECA_PROCESSOR::init(void) {
  ecadebug->control_flow("Engine/Initializing");
  st_info_string = "";

  init_pointers();
  eparams->status(ep_status_stopped);

  init_status_variables();
  init_mix_method();

  // ---
  // Handle priority
  // ---
  if (csetup->mixmode != ep_mm_simple ||
      rt_infiles == true ||
      csetup->raisepriority == true) {
    //    getpriority(PRIO_PROCESS, 0);
    int pri_new = -10;
    if (csetup->mixmode == ep_mm_mthreaded) pri_new = -5;
    if (setpriority(PRIO_PROCESS, 0, pri_new) == -1)
      ecadebug->msg("(eca-main) Unable to change priority.");
    
    MESSAGE_ITEM mtemp;
    mtemp << "(eca-main) Raised process priority to ";
    mtemp << pri_new << ", PID: " << (int)getpid() << ".";
    ecadebug->msg(1, mtemp.to_string());
  }
}

void ECA_PROCESSOR::init_pointers(void) {
  if (eparams->is_active_chainsetup_connected() == false) {
    throw(new ECA_ERROR("ECA_PROCESSOR", "Engine startup aborted, no chainsetup connected to session!"));
  }

  csetup = eparams->active_chainsetup;
  inputs = eparams->inputs;
  outputs = eparams->outputs;
  chains = eparams->chains;

  if (inputs == 0 ||
      inputs->size() == 0 ||
      outputs  == 0 ||
      outputs->size() == 0 ||
      chains == 0 ||
      chains->size() == 0) {
    throw(new ECA_ERROR("ECA_PROCESSOR", "Engine startup aborted, session in corrupted state!"));
  }
}

void ECA_PROCESSOR::init_status_variables(void) {
  end_request = false;
  finished_result = false;

  // ---
  // Are we doing multitrack-recording?
  // ---    
    
  rt_infiles = rt_outfiles = false;
    
  for (adev_citer = inputs->begin(); adev_citer != inputs->end(); adev_citer++) {
    if ((*adev_citer)->is_realtime()) rt_infiles = true;
  }

  for (adev_citer = outputs->begin(); adev_citer != outputs->end(); adev_citer++) {
    if ((*adev_citer)->is_realtime()) rt_outfiles = true;
  }

  multitrack_sync_phase = false;
   
  if (rt_infiles == true && rt_outfiles == true &&
      chains->size() > 1 && eparams->iactive)  {
    ecadebug->msg("(eca-main) Multitrack-mode enabled. Changed mixmode to \"normal iactive\"");
    eparams->multitrack_mode = true;
    csetup->mixmode =  ep_mm_normal;
  }
}


void ECA_PROCESSOR::init_mix_method(void) { 
  if (csetup->mixmode == ep_mm_auto) {
    if (eparams->mthreaded_use_of_session == true) eparams->iactive = true;
    if (chains->size() == 1 &&
	inputs->size() == 1 &&
	outputs->size() == 1 &&
	eparams->mthreaded_use_of_session == false) 
      csetup->mixmode = ep_mm_simple;
    else if (csetup->buffersize >= 1024)
      csetup->mixmode = ep_mm_mthreaded;
    else 
      csetup->mixmode = ep_mm_normal;
  }
}

void ECA_PROCESSOR::exec(void) {
  switch(csetup->mixmode) {
  case ep_mm_simple:
    {
      if (eparams->iactive) exec_simple_iactive();
      else exec_simple_passive();
      break;
    }
  case ep_mm_normal:
    {
      if (eparams->iactive) exec_normal_iactive();
      else exec_normal_passive();
      break;
    }
  case ep_mm_mthreaded:
    {
      if (eparams->iactive) exec_mthreaded_iactive();
      else exec_mthreaded_passive();
      break;
    }
  default: 
    {
      exec_normal_iactive();
    }
  }
}

void ECA_PROCESSOR::exec_normal_iactive(void) {
  // ---
  // Enable devices (non-multitrack-mode)
  // ---

  stop();
  
  // ---
  // The main processing loop.
  // ---
  
  ecadebug->control_flow("Engine/Mixmode \"normal iactive\" selected");
  
  while (true) {
    if (finished() == true) {
      if (eparams->loop_active == true) {
	set_position(eparams->loop_start_pos);
	stop();
	start();
      }
      else if (end_request) {
	break;
      }
      else stop();
    }
    if (eparams->iactive) {
      interprete_cqueue();
      if (end_request) {
	break;
      }
      if (eparams->status() != ep_status_running) continue;
    }

    // ---
    // Check looping params
    // ---
    if (eparams->loop_active == true && current_position() >= eparams->loop_end_pos)
      set_position(eparams->loop_start_pos);

    // ---
    // Get input samples.
    // ---
    for(adev_sizet = 0; adev_sizet < inputs->size(); adev_sizet++) {
      if (multitrack_sync_phase 
	  && (*inputs)[adev_sizet]->is_realtime()) continue;
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      (*inputs)[adev_sizet]->get_sample(&eparams->inslots[adev_sizet]);
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }

    // ---
    // Mix from eparams->inslots to chains
    // ---

    for (chain_sizet = 0; chain_sizet < chains->size(); chain_sizet++)
      pthread_mutex_lock(eparams->chain_locks[chain_sizet]);

    mix_to_chains();

    // ---
    // Chainoperator processing phase.
    // ---

    if (eparams->sfx) {
      chain_iter = chains->begin();
      while(chain_iter != chains->end()) {
    	(*chain_iter)->chainop_process();
    	++chain_iter;
      }
    }

    // ---
    // Mix from chains to eparams->outslots.
    // ---
    for(adev_sizet = 0; adev_sizet < outputs->size(); adev_sizet++) {
      pthread_mutex_lock(eparams->out_locks[adev_sizet]);
    }
    mix_from_chains();

    for (chain_sizet = 0; chain_sizet < chains->size(); chain_sizet++)
      pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);

    // ---
    // Write to output file(s).
    // ---
    for(adev_sizet = 0; adev_sizet < outputs->size();
	adev_sizet++) {
      // ---
      // NOTICE! Support for rt-sync:
      // skip output if: sync_phase + realtime-input + non-realtime-output
      // ---
      if (multitrack_sync_phase) {
	if (eparams->is_slave_output(adev_sizet) == false)
	  (*outputs)[adev_sizet]->put_sample(&eparams->outslots[adev_sizet]);
      }
      else {
	(*outputs)[adev_sizet]->put_sample(&eparams->outslots[adev_sizet]);
      }
      pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
    }


    // ---
    // Enable devices (multitrack-mode, preloading)
    // ---        

    start_synced();
    ++(eparams->loop_counter);
  }
}

void ECA_PROCESSOR::exec_normal_passive(void) {
  // ---
  // Enable devices.
  // ---
  start();
  
  ecadebug->control_flow("Engine/Mixmode \"normal passive\" selected");
  while (finished() == false) {
    for(adev_sizet = 0; adev_sizet < inputs->size(); adev_sizet++) {
      (*inputs)[adev_sizet]->get_sample(&eparams->inslots[adev_sizet]);
    }

    mix_to_chains();

    chain_iter = chains->begin();
    while(chain_iter != chains->end()) {
      (*chain_iter)->chainop_process();
      ++chain_iter;
    }
    
    mix_from_chains();

    for(adev_sizet = 0; adev_sizet < outputs->size(); adev_sizet++) {
      (*outputs)[adev_sizet]->put_sample(&eparams->outslots[adev_sizet]);
    }
  }
}


void ECA_PROCESSOR::exec_simple_iactive(void) {
  // ---
  // Enable devices (non-multitrack-mode)
  // ---
  stop();  

  ecadebug->control_flow("Engine/Mixmode \"simple iactive\" selected");
  while (true) {
    if (finished() == true) {
      if (eparams->loop_active == true) {
	set_position(eparams->loop_start_pos);
	stop();
	start();
      }
      else if (end_request) {
	break;
      }
      else stop();
    }

    interprete_cqueue();
    if (end_request) {

      break;
    }
    if (eparams->status() != ep_status_running) continue;

    // ---
    // Check looping params
    // ---
    if (eparams->loop_active == true && current_position() >= eparams->loop_end_pos)
      set_position(eparams->loop_start_pos);

    pthread_mutex_lock(eparams->in_locks[0]);
    (*inputs)[0]->get_sample(&eparams->inslots[0]);
    pthread_mutex_unlock(eparams->in_locks[0]);

    if (eparams->sfx) {
	pthread_mutex_lock(eparams->chain_locks[0]);

      (*chains)[0]->gcontroller_update();
      chainop_citer = (*chains)[0]->chainops.begin();
      while(chainop_citer != (*chains)[0]->chainops.end()) {
	(*chainop_citer)->process(&eparams->inslots[0]);
	++chainop_citer;
      }
      eparams->inslots[0].limit_values();
      pthread_mutex_unlock(eparams->chain_locks[0]);
    }
    pthread_mutex_lock(eparams->out_locks[0]);
    (*outputs)[0]->put_sample(&eparams->inslots[0]);
    pthread_mutex_unlock(eparams->out_locks[0]);
  }
}

void ECA_PROCESSOR::exec_simple_passive(void) {
  // ---
  // Enable devices.
  // ---
  start();
  ecadebug->control_flow("Engine/Mixmode \"simple passive\" selected");
  while (!finished()) {
    (*inputs)[0]->get_sample(&eparams->inslots[0]);
    (*chains)[0]->gcontroller_update();
    chainop_citer = (*chains)[0]->chainops.begin();
    while(chainop_citer != (*chains)[0]->chainops.end()) {
      (*chainop_citer)->process(&eparams->inslots[0]);
      ++chainop_citer;
    }
    
    (*outputs)[0]->put_sample(&eparams->inslots[0]);
  }
}

void ECA_PROCESSOR::set_position(double seconds) {
  bool was_running;
  if (eparams->status() == ep_status_running) was_running = true;
  else was_running = false;
  stop();

  for (adev_sizet = 0; adev_sizet != inputs->size(); adev_sizet++) {
    pthread_mutex_lock(eparams->in_locks[adev_sizet]);
    (*inputs)[adev_sizet]->position_in_seconds(seconds, true);
    pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
  }

  for (adev_sizet = 0; adev_sizet != outputs->size(); adev_sizet++) {
    pthread_mutex_lock(eparams->out_locks[adev_sizet]);
    (*outputs)[adev_sizet]->position_in_seconds(seconds, true);
    pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
  }
  if (was_running == true) start();
}

void ECA_PROCESSOR::change_position(int seconds) {
  bool was_running;
  if (eparams->status() == ep_status_running) was_running = true;
  else was_running = false;
  stop();

  for (adev_sizet = 0; adev_sizet != inputs->size(); adev_sizet++) {
    pthread_mutex_lock(eparams->in_locks[adev_sizet]);
    (*inputs)[adev_sizet]->position_in_seconds((*inputs)[adev_sizet]->position_in_seconds()
                                           + seconds, true);
    pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
  }

  for (adev_sizet = 0; adev_sizet != outputs->size(); adev_sizet++) {
    pthread_mutex_lock(eparams->out_locks[adev_sizet]);
    (*outputs)[adev_sizet]->position_in_seconds((*outputs)[adev_sizet]->position_in_seconds()
                                           + seconds, true);
    pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
  }
  if (was_running == true) start();
}

double ECA_PROCESSOR::current_position(void) {
  eparams->lock_master_input();
  double temp = eparams->master_input()->position_in_seconds_exact();
  eparams->release_master_input();
  return(temp);
}

void ECA_PROCESSOR::stop(void) { 
  if (eparams->status() != ep_status_running) return;
  ecadebug->msg(1, "(eca-main) Stop");
  for (adev_sizet = 0; adev_sizet != inputs->size(); adev_sizet++) {
    pthread_mutex_lock(eparams->in_locks[adev_sizet]);
    (*inputs)[adev_sizet]->rt_stop();
    pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
  }

  for (adev_sizet = 0; adev_sizet != outputs->size(); adev_sizet++) {
    pthread_mutex_lock(eparams->out_locks[adev_sizet]);
    (*outputs)[adev_sizet]->rt_stop();
    pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
  }

  eparams->status(ep_status_stopped);
}

void ECA_PROCESSOR::start(void) {
  if (eparams->status() == ep_status_running) return;
  ecadebug->msg(1, "(eca-main) Start");
  if (eparams->multitrack_mode == true) {
    multitrack_sync_phase = true;

    for (adev_sizet = 0; adev_sizet != inputs->size(); adev_sizet++) {
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      (*inputs)[adev_sizet]->rt_ready();
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }
    for (adev_sizet = 0; adev_sizet != outputs->size(); adev_sizet++) {
      //      if (eparams->is_slave_output(adev_sizet) == false) continue;
      pthread_mutex_lock(eparams->out_locks[adev_sizet]);
      (*outputs)[adev_sizet]->rt_ready();
      pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
    }
  }
  else {
    for (adev_sizet = 0; adev_sizet != inputs->size(); adev_sizet++) {
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      (*inputs)[adev_sizet]->rt_ready();
      (*inputs)[adev_sizet]->rt_activate();
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }
    for (adev_sizet = 0; adev_sizet != outputs->size(); adev_sizet++) {
      pthread_mutex_lock(eparams->out_locks[adev_sizet]);
      (*outputs)[adev_sizet]->rt_ready();
      (*outputs)[adev_sizet]->rt_activate();
      pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
    }
  }

  eparams->status(ep_status_running);
}

void ECA_PROCESSOR::start_synced(void) {
  if (eparams->multitrack_mode == true && multitrack_sync_phase == true) {
    ecadebug->msg(2,"(eca-main) start_synced: sync_phase true, turning off");
    multitrack_sync_phase = false;
    for (adev_sizet = 0; adev_sizet != inputs->size(); 	 adev_sizet++) {
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      (*inputs)[adev_sizet]->rt_activate();
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }
    for (adev_sizet = 0; adev_sizet != outputs->size(); adev_sizet++) {
      //      if (eparams->is_slave_output(adev_sizet)) continue;
      pthread_mutex_lock(eparams->out_locks[adev_sizet]);
      (*outputs)[adev_sizet]->rt_activate();
      pthread_mutex_unlock(eparams->out_locks[adev_sizet]);
    }
  }
}

void ECA_PROCESSOR::interprete_cqueue(void) {
  while (ecasound_cqueue.cmds_available() == true) {
    ecadebug->msg(2,"(eca-main) ecasound_cqueue: cmds available - front " + ecasound_cqueue.front());
    //
    // Basic control commands.
    //
    if (ecasound_cqueue.front() == "end" || ecasound_cqueue.front() == "e") {
      ecadebug->msg(2,"(eca-main) ecasound_cqueue: end!");
      stop();
      end_request = true;
    }
    else if (ecasound_cqueue.front() == "start" || ecasound_cqueue.front() == "t") start();
    else if (ecasound_cqueue.front() == "stop" || ecasound_cqueue.front() == "s") stop();
    //
    // Section/chain (en/dis)abling commands.
    //                
    else if (ecasound_cqueue.front() == "chain" || ecasound_cqueue.front() == "c") {
      ecasound_cqueue.pop_front();
      if (ecasound_cqueue.cmds_available() != true) return;
      if (is_chain_enabled(ecasound_cqueue.front())) {
	disable_chain(ecasound_cqueue.front());
	//	      ecadebug->msg("(eca-main)  Chain " + ecasound_cqueue.front() + " disabled.");
      }
      else {
	enable_chain(ecasound_cqueue.front());
	//	      ecadebug->msg("IA-MODE: Chain " + ecasound_cqueue.front() + " enabled.");
      }
    }
    //
    // Status requests.
    //
    else if (ecasound_cqueue.front() == "sfx") {
      if (eparams->sfx == true) {
	eparams->sfx = false;
	ecadebug->msg("(eca-main) Effect section disabled.");
      }
      else {
	eparams->sfx = true;
	ecadebug->msg("(eca-main) Effect section enabled.");
      }
    }
    else if (ecasound_cqueue.front() == "rewind" || ecasound_cqueue.front() == "rw") {
      ecasound_cqueue.pop_front();
      if (ecasound_cqueue.cmds_available() != true) return;
      change_position(-(int)ceil(atof(ecasound_cqueue.front().c_str())));
      MESSAGE_ITEM mtemp;
      mtemp << "(eca-main) Rewinding (seconds) " << ecasound_cqueue.front();
      ecadebug->msg(mtemp.to_string());
    }
    else if (ecasound_cqueue.front() == "forward" || ecasound_cqueue.front() == "fw") {
      ecasound_cqueue.pop_front();
      if (ecasound_cqueue.cmds_available() != true) return;
      change_position((int)ceil(atof(ecasound_cqueue.front().c_str())));
      MESSAGE_ITEM mtemp;
      mtemp << "(eca-main) Forward (seconds) " << ecasound_cqueue.front();
      ecadebug->msg(mtemp.to_string());                        
    }
    else if (ecasound_cqueue.front() == "setpos") {
      ecasound_cqueue.pop_front();
      if (ecasound_cqueue.cmds_available() != true) return;
      set_position((int)ceil(atof(ecasound_cqueue.front().c_str())));
      MESSAGE_ITEM mtemp;
      mtemp << "(eca-main) Position from beginning (seconds) " << ecasound_cqueue.front();
      ecadebug->msg(mtemp.to_string());                                    
    }
    //
    // Loop commands.
    //
    else if (ecasound_cqueue.front() == "loop") {
      ecadebug->msg("(eca-main) Enabled looping.");
      eparams->loop_active = true;
    }
    else if (ecasound_cqueue.front() == "loop_start") {
      ecasound_cqueue.pop_front();
      if (ecasound_cqueue.cmds_available() != true) return;
      eparams->loop_start_pos = (int)ceil(atof(ecasound_cqueue.front().c_str()));
      MESSAGE_ITEM mtemp;
      mtemp << "(eca-main) Looping starts at position " << eparams->loop_start_pos;
      ecadebug->msg(mtemp.to_string());            
    }
    else if (ecasound_cqueue.front() == "loop_end") {
      ecasound_cqueue.pop_front();
      if (ecasound_cqueue.cmds_available() != true) return;
      eparams->loop_end_pos = (int)ceil(atof(ecasound_cqueue.front().c_str()));
      MESSAGE_ITEM mtemp;
      mtemp << "(eca-main) Looping ends at position " << eparams->loop_end_pos;
      ecadebug->msg(mtemp.to_string());
    }
    ecasound_cqueue.pop_front();
  }
}

bool ECA_PROCESSOR::finished(void) {
  finished_result = true;
  for (adev_citer = outputs->begin(); adev_citer != outputs->end(); adev_citer++) {
    if ((*adev_citer)->finished() == true) {
      ecadebug->msg(2, "(eca-main) Finished... outfile.");
      stop();
      eparams->status(ep_status_finished);
      return(true);
    }
  }
  
  for (adev_sizet = 0; adev_sizet != inputs->size(); adev_sizet++) {
    if (!((*inputs)[adev_sizet]->finished())) {
      finished_result = false;
      break;
    }
    else finished_result = true;
  }
  
  if (finished_result == true && eparams->iactive == false)
    ecadebug->msg(2, "(eca-main) Finished... through.");
  if (finished_result) {
    stop();
    eparams->status(ep_status_finished);
  }
  return(finished_result);
}

void ECA_PROCESSOR::mix_to_chains(void) {
  chain_citer = chains->begin();
  while (chain_citer != chains->end()) {
    for(audioslot_sizet = 0; audioslot_sizet < eparams->inslots.size(); audioslot_sizet++) {
      if ((*chain_citer)->inputid == (int)audioslot_sizet) {
	(*chain_citer)->audioslot.operator=(eparams->inslots[audioslot_sizet]);
	// --- for debugging signal flow
	//	cerr << "[1]Mixing from sbuf eparams->inslots[]" << " nro " << eparams->inslots[audioslot_sizet]->nro <<  " " << eparams->inslots[audioslot_sizet]->average_volume() <<".\n";
	//	cerr << "[1]Mixing to sbuf audioslot[c]" << " nro " << (*chains)[c]->audioslot->nro << " " << (*chains)[c]->audioslot->average_volume() << ".\n";
	// -----------------------------
      }
    }
    ++chain_citer;
  }
}

void ECA_PROCESSOR::mix_from_chains(void) {
  for(audioslot_sizet = 0; audioslot_sizet < eparams->outslots.size(); audioslot_sizet++) {
    eparams->outslots[audioslot_sizet].make_silent();
    chain_citer = chains->begin();
    while (chain_citer != chains->end()) {
      if ((*chain_citer)->outputid == (int)audioslot_sizet) {
	if ((*chain_citer)->is_enabled() == true) {
	  eparams->outslots[audioslot_sizet].add_with_weight((*chain_citer)->audioslot, eparams->number_of_connected_chains_to_output(audioslot_sizet));
	}
	// --- for debugging signal flow
	//	cerr << "[2]Mixing from sbuf audioslot[c]" << " nro " << (*chains)[c]->audioslot->nro <<  " " << (*chains)[c]->audioslot->average_volume() <<".\n";
	//	cerr << "[2]Mixing to sbuf outslot[]" << " nro " << eparams->outslots[audioslot_sizet]->nro <<  " " << eparams->outslots[audioslot_sizet]->average_volume() << ".\n";
	// ------------------------------
      }
      ++chain_citer;
    }
  }
}

bool ECA_PROCESSOR::is_chain_enabled(const string& cname) {
  for(chain_citer = chains->begin(); chain_citer != chains->end(); chain_citer++) {
    if ((*chain_citer)->name() == cname) {
      return((*chain_citer)->is_enabled());
    }
  }
  return(false);
}

void ECA_PROCESSOR::enable_chain(const string& cname) {
  for(chain_iter = chains->begin(); chain_iter != chains->end(); chain_iter++) {
    if ((*chain_iter)->name() == cname) {
      (*chain_iter)->enable();
      return;
    }
  }
}

void ECA_PROCESSOR::disable_chain(const string& cname) {
  for(chain_iter = chains->begin(); chain_iter != chains->end(); chain_iter++) {
    if ((*chain_iter)->name() == cname) {
      (*chain_iter)->disable();
      return;
    }
  }
}

void ECA_PROCESSOR::exec_mthreaded_iactive(void) {

  ecadebug->control_flow("Engine/Mixmode \"multithreaded interactive\" selected");
  stop();

  for (chain_sizet = 0; chain_sizet < chains->size(); chain_sizet++)
    eparams->chain_ready_for_submix[chain_sizet] = false;

  int submix_pid = pthread_create(&chain_thread, NULL, mthread_process_chains, ((void*)eparams));

  if (submix_pid != 0)
    throw(new ECA_ERROR("ECA-MAIN", "Unable to create a new thread (mthread_process_chains)."));

  unsigned long usleep_count;
  while (true) {
    usleep_count = 5;
    if (finished() == true) {
      if (eparams->loop_active == true) {
	set_position(eparams->loop_start_pos);
	stop();
    	start();
      }
      else if (end_request) {
	break;
      }
      else stop();
    }

    
    interprete_cqueue();
    if (end_request) {
      break;
    }
    if (eparams->status() != ep_status_running) {
      if (usleep_count < 1000) ++usleep_count;
      usleep(usleep_count);
      continue;
    }


    // ---
    // Check looping params
    // ---
    if (eparams->loop_active == true && current_position() >= eparams->loop_end_pos)
      set_position(eparams->loop_start_pos);

    for(adev_sizet = 0; adev_sizet < inputs->size(); adev_sizet++) {
      pthread_mutex_lock(eparams->in_locks[adev_sizet]);
      (*inputs)[adev_sizet]->get_sample(&eparams->inslots[adev_sizet]);
    }
    
    chain_sizet = 0;
    usleep_count = 5;
    while (chain_sizet < chains->size()) {
      if ((*chains)[chain_sizet]->outputid == -1) {
	// chain not connected to a ouput, skip
	++chain_sizet;
	continue;
      }
      pthread_mutex_lock(eparams->chain_locks[chain_sizet]);
      if (eparams->chain_ready_for_submix[chain_sizet] == true) {
	pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);
	//	usleep(20);
	usleep(++usleep_count);
	continue;
      }

      for(audioslot_sizet = 0; audioslot_sizet < eparams->inslots.size(); audioslot_sizet++) {
	if ((*chains)[chain_sizet]->inputid == static_cast<int>(audioslot_sizet)) {
	  (*chains)[chain_sizet]->audioslot.operator=(eparams->inslots[audioslot_sizet]);
	}
      }
	
      eparams->chain_ready_for_submix[chain_sizet] = true;
      pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);
      ++chain_sizet;
    }

    for(adev_sizet = 0; adev_sizet < inputs->size(); adev_sizet++) {
      pthread_mutex_unlock(eparams->in_locks[adev_sizet]);
    }


    ++(eparams->loop_counter);

    start_synced();
  }
  pthread_cancel(chain_thread);
  pthread_join(chain_thread,NULL);
}

void ECA_PROCESSOR::exec_mthreaded_passive(void) {

  ecadebug->control_flow("Engine/Mixmode \"multithreaded passive\" selected");
  start();

  for (chain_sizet = 0; chain_sizet < chains->size(); chain_sizet++)
    eparams->chain_ready_for_submix[chain_sizet] = false;

  int submix_pid = pthread_create(&chain_thread, NULL, mthread_process_chains, ((void*)eparams));

  if (submix_pid != 0)
    throw(new ECA_ERROR("ECA-MAIN", "Unable to create a new thread (mthread_process_chains)."));

  unsigned long usleep_count;
  while (!finished()) {
    for(adev_sizet = 0; adev_sizet < inputs->size(); adev_sizet++) {
      (*inputs)[adev_sizet]->get_sample(&eparams->inslots[adev_sizet]);
    }

    chain_sizet = 0;
    usleep_count = 5;
    while (chain_sizet < chains->size()) {
      pthread_mutex_lock(eparams->chain_locks[chain_sizet]);
      if (eparams->chain_ready_for_submix[chain_sizet] == true) {
	pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);
	//	usleep(20);
	usleep(++usleep_count);
	continue;
      }

      for(audioslot_sizet = 0; audioslot_sizet < eparams->inslots.size(); audioslot_sizet++) {
	if ((*chains)[chain_sizet]->inputid == static_cast<int>(audioslot_sizet)) {
	  //	  cerr << "IM" << audioslot_sizet << ":"       <<
	  //	  &(eparams->inslots[audioslot_sizet]) 	       
	  // << "->" << &((*chains)[chain_sizet]->audioslot) << "\n";
	  (*chains)[chain_sizet]->audioslot.operator=(eparams->inslots[audioslot_sizet]);
	}
      }
      eparams->chain_ready_for_submix[chain_sizet] = true;
      pthread_mutex_unlock(eparams->chain_locks[chain_sizet]);
      ++chain_sizet;
    }
  }
  pthread_cancel(chain_thread);
  pthread_join(chain_thread,NULL);
}


void *mthread_process_chains(void* params) {

  pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);  // other
  // threads can stop this one
  pthread_setcanceltype(PTHREAD_CANCEL_DEFERRED, NULL);

  ECA_SESSION* ecaparams = (ECA_SESSION*)params; 
  
  ecadebug->control_flow("Submix-thread ready");
  ecadebug->msg(1,"(eca-main) Submix-pid: " + kvu_numtostr(getpid()));

  // ---
  // Handle general parameters.
  // ---
  getpriority(PRIO_PROCESS, 0);
  if (setpriority(PRIO_PROCESS, 0, -10) == -1)
    ecadebug->msg("(eca-main) Unable to change priority.");
  
  MESSAGE_ITEM mtemp;
  mtemp << "(eca-main) Raised submix-thread priority to -10";
  mtemp << ", PID: " << (int)getpid() << ".";
  
  ecadebug->msg(1, mtemp.to_string());
  
  //  vector<AUDIO_IO_DEVICE*>** outputs = &(ecaparams->outputs);
  //  vector<CHAIN*>** chains = &(ecaparams->chains);

  vector<AUDIO_IO_DEVICE*>* outputs = ecaparams->outputs;
  vector<CHAIN*>* chains = ecaparams->chains;

  unsigned int n;
  unsigned long usleep_count;
  //  vector<CHAIN*>::const_iterator chain_citer;
  vector<SAMPLE_BUFFER>::size_type audioslot_sizet;

  while(true) {
    n = 0;
    usleep_count = 5;
    while (n < (chains)->size()) {
      if ((*chains)[n]->outputid == -1) {
	// chain not connected to a ouput, skip
	++n;
	continue;
      }
      pthread_mutex_lock(ecaparams->chain_locks[n]);
      if (ecaparams->chain_ready_for_submix[n] == false) {
	pthread_mutex_unlock(ecaparams->chain_locks[n]);
	usleep(++usleep_count);
	pthread_testcancel();
	continue;
      }

      if (ecaparams->sfx) {
	//	cerr << "C" << n << &((**chains)[n]->audioslot) << "\n";
	(*chains)[n]->chainop_process();
      }

      ++n;
    }

    for(audioslot_sizet = 0; audioslot_sizet < ecaparams->outslots.size(); audioslot_sizet++) {
      pthread_mutex_lock(ecaparams->out_locks[audioslot_sizet]);
      ecaparams->outslots[audioslot_sizet].make_silent();
      n = 0;
      while (n < (chains)->size()) {
	if ((*chains)[n]->outputid == -1) {
	  // chain not connected to a ouput, skip
	  ++n;
	  continue;
	}
	if ((*chains)[n]->outputid == static_cast<int>(audioslot_sizet)) {
	  if ((*chains)[n]->is_enabled() == true) {
	    //	    cerr << "OM" << audioslot_sizet << ":" <<   &((**chains)[n]->audioslot) << "->" <<     &(ecaparams->outslots[audioslot_sizet]) << "\n";
	    ecaparams->outslots[audioslot_sizet].add_with_weight((*chains)[n]->audioslot, ecaparams->number_of_connected_chains_to_output(audioslot_sizet));
	  }
	  ecaparams->chain_ready_for_submix[n] = false;
	  pthread_mutex_unlock(ecaparams->chain_locks[n]);
	}
	else if ((*chains)[n]->outputid == -1) {
	  ecaparams->chain_ready_for_submix[n] = false;
	  pthread_mutex_unlock(ecaparams->chain_locks[n]);
	}
	++n;
      }

      if (ecaparams->status() != ep_status_running) {
	pthread_mutex_unlock(ecaparams->out_locks[audioslot_sizet]);
	break;
      }

      (*outputs)[audioslot_sizet]->put_sample(&ecaparams->outslots[audioslot_sizet]);
      pthread_mutex_unlock(ecaparams->out_locks[audioslot_sizet]);
    }
    pthread_testcancel();
  }
  cerr << "(eca-main/submix) You should never see this message!\n";
}

