// ------------------------------------------------------------------------
// samplebuffer.cpp: Routines and classes for handling sample buffers.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <vector>
#include <cmath>
#include <cstdio>

#include <kvutils.h>

#include "samplebuffer.h"
#include "eca-debug.h"
#include "eca-error.h"

SAMPLE_BUFFER::single_type SAMPLE_BUFFER::max_value(char channel) {
  single_type max_temp = 0.0;

  buf_iter = buffer.begin();
  while(buf_iter != buffer.end()) {
    if (buf_iter->sample[channel] > max_temp) max_temp = buf_iter->sample[channel];
    ++buf_iter;
  }

  return(max_temp);
}

SAMPLE_BUFFER::single_type SAMPLE_BUFFER::min_value(char channel) {
 single_type min_temp = 0.0;

  buf_iter = buffer.begin();
  while(buf_iter != buffer.end()) {
    if (buf_iter->sample[channel] < min_temp) min_temp = buf_iter->sample[channel];
    ++buf_iter;
  }

  return(min_temp);
}

void SAMPLE_BUFFER::limit_values(void) {
  buf_iter = buffer.begin();
  while(buf_iter != buffer.end()) {
    for (channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
      if (buf_iter->sample[channel_sizet] > SAMPLE_SPECS::max_value)
	buf_iter->sample[channel_sizet] = SAMPLE_SPECS::max_value;
      else if (buf_iter->sample[channel_sizet] < SAMPLE_SPECS::min_value) 
	buf_iter->sample[channel_sizet] = SAMPLE_SPECS::min_value;
    }
    ++buf_iter;
  }
}

void SAMPLE_BUFFER::previous(void) {
    index--;
    if (index < 0) validindex = false;
    else validindex = true;
}

void SAMPLE_BUFFER::last(void) {
    index = buffer.size() - 1;
    validindex = true;
}

void SAMPLE_BUFFER::buffer_to_iobuf(int bits, bool iobuf_little_endian, unsigned short int channels) {
  // ---
  // Contents of sample buffer is copied to 'iobuf_uchar'.
  // Convert to sample width 'bits' and 'channels'.
  // If 'iobuf_little_endian' convert to use LSB, MSB byte order,
  // otherwise use MSB, LSB (16bit data only).
  // ---

  for(buf_sizet = 0, ubuf_sizet = 0; buf_sizet < buffer.size(); buf_sizet++) {
    switch (bits) {
    case 16: 
      {
	for(channel_sizet = 0; channel_sizet < channels; channel_sizet++) {
	  s16temp = (signed short int)(single_type)(buffer[buf_sizet].sample[channel_sizet] / s16_to_st_constant);
	  // --- for debugging signal flow
	  //	  if (buf_sizet == 0) 
	  //	      printf("converted to s16 %d (hex:%x)", s16temp, (unsigned short int)s16temp);
	  // ------------------------------

	  // little endian: (LSB, MSB) (Intel).
	  // big endian: (MSB, LSB) (Motorola).
	  // ---
	  if (iobuf_little_endian && SAMPLE_BUFFER::is_system_littleendian ||
	      !iobuf_little_endian && !SAMPLE_BUFFER::is_system_littleendian) {
	    iobuf_uchar[ubuf_sizet++] = (unsigned char)(s16temp & 0xff);
	    iobuf_uchar[ubuf_sizet++] = (unsigned char)((s16temp >> 8) & 0xff);
	  }
	  else {
	    iobuf_uchar[ubuf_sizet++] = (unsigned char)((s16temp >> 8) & 0xff);
	    iobuf_uchar[ubuf_sizet++] = (unsigned char)(s16temp & 0xff);
	  }
	}
	break;
      }

    case 8:
      {
	for(channel_sizet = 0; channel_sizet < channels; channel_sizet++) {
	  iobuf_uchar[ubuf_sizet++] = (unsigned char)((single_type)(buffer[buf_sizet].sample[channel_sizet] / u8_to_st_constant) + u8_to_st_delta);
	}
	break;
      }
    }
  }
}


void SAMPLE_BUFFER::iobuf_to_buffer(int bits, bool iobuf_little_endian, unsigned short int channels) {
  // ---
  // Contents of 'iobuf_uchar' is copied to main sample buffer.
  // Convert from sample width 'bits' and 'channels'.
  // If 'iobuf_little_endian', presume source data to be in 
  // LSB, MSB byte order. Otherwise use MSB, LSB. (16bit data only)
  // ---

  for(buf_sizet = 0, ubuf_sizet = 0; buf_sizet < buffer.size(); buf_sizet++) {
    switch (bits) {
    case 16:
      {
	for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
	  if (channel_sizet < channels) {
	    // little endian: (LSB, MSB) (Intel).
	    // big endian: (MSB, LSB) (Motorola).
	    if (iobuf_little_endian && SAMPLE_BUFFER::is_system_littleendian ||
	      !iobuf_little_endian && !SAMPLE_BUFFER::is_system_littleendian) {
	      ps16temp[0] = iobuf_uchar[ubuf_sizet++];
	      ps16temp[1] = iobuf_uchar[ubuf_sizet++];
	    }
	    else {
	      //	      if (buf_sizet == 0) cerr << "H";
	      ps16temp[1] = iobuf_uchar[ubuf_sizet++];
	      ps16temp[0] = iobuf_uchar[ubuf_sizet++];
	    }
	    buffer[buf_sizet].sample[channel_sizet] = (single_type)(*(signed short int*)ps16temp) * s16_to_st_constant;
	    // --- for debugging signal flow
	    // if (buf_sizet == 0) {
	    //	 printf(" ... converted to %d (hex:%x)...", s16temp,s16temp); 
	    //	 printf(" ... and scaled to %.2f.\n", buffer[buf_sizet].sample[channel_sizet]);
	    // }
	  }
	  else
	    buffer[buf_sizet].sample[channel_sizet] = buffer[buf_sizet].sample[0];
	}
      }
      break;
    case 8: 
      {
	for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
	  if (channel_sizet < channels) {
	    buffer[buf_sizet].sample[channel_sizet] = (unsigned char)iobuf_uchar[ubuf_sizet++];
	    buffer[buf_sizet].sample[channel_sizet] -= u8_to_st_delta;
	    buffer[buf_sizet].sample[channel_sizet] *= u8_to_st_constant;
	  }
	  else
	    buffer[buf_sizet].sample[channel_sizet] = buffer[buf_sizet].sample[0];
	}
	break;
      }
    }
  }
}

void SAMPLE_BUFFER::make_silent(void) {
  buf_iter = buffer.begin();
  while(buf_iter != buffer.end()) {
    for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
      buf_iter->sample[channel_sizet] = SAMPLE_SPECS::silent_value;
    }
    ++buf_iter;
  }
}

void SAMPLE_BUFFER::divide_by(double dvalue) {
  buf_iter = buffer.begin();
  while(buf_iter != buffer.end()) {
    for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
      buf_iter->sample[channel_sizet] = buf_iter->sample[channel_sizet] / dvalue;
    }
    ++buf_iter;
  }
}

void SAMPLE_BUFFER::add(const SAMPLE_BUFFER& x) {
  for(buf_sizet = 0; buf_sizet < buffer.size(); buf_sizet++) {  
    if (buf_sizet == x.buffer.size()) break;

    for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
      buffer[buf_sizet].sample[channel_sizet] += x.buffer[buf_sizet].sample[channel_sizet];
    }
  }
}

void SAMPLE_BUFFER::add_with_weight(const SAMPLE_BUFFER& x, int weight) {
  for(buf_sizet = 0; buf_sizet < buffer.size(); buf_sizet++) {
    if (buf_sizet == x.buffer.size()) break;

    for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
      buffer[buf_sizet].sample[channel_sizet] += 
	x.buffer[buf_sizet].sample[channel_sizet] / weight;
    }
  }
}

SAMPLE_BUFFER::single_type SAMPLE_BUFFER::average_volume(void) {
  temp_avg = 0;
  buf_iter = buffer.begin();
  while(buf_iter != buffer.end()) {
    for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
      temp_avg += fabs(buf_iter->sample[channel_sizet] - SAMPLE_SPECS::silent_value);
    }
    ++buf_iter;
  }
  return(temp_avg / (single_type)buffer.size() / (single_type)SAMPLE_SPECS::ch_count);
}

SAMPLE_BUFFER::single_type SAMPLE_BUFFER::average_RMS_volume(void) {
  temp_avg = 0;
  buf_iter = buffer.begin();
  while(buf_iter != buffer.end()) {
    for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
      temp_avg += buf_iter->sample[channel_sizet] * buf_iter->sample[channel_sizet];
    }
    ++buf_iter;
  }
  return(sqrt(temp_avg / (single_type)buffer.size()) / (single_type)SAMPLE_SPECS::ch_count);
}


SAMPLE_BUFFER::single_type SAMPLE_BUFFER::average_volume(char channel,
							 int
							 count_samples) 
{
  temp_avg = 0.0;
  if (count_samples == 0) count_samples = (int)buffer.size();
  for(buf_sizet = 0; buf_sizet < count_samples; buf_sizet++) {
    temp_avg += fabs(buffer[buf_sizet].sample[channel] - SAMPLE_SPECS::silent_value);
  }
  return(sqrt(temp_avg / (single_type)count_samples));
}

SAMPLE_BUFFER::single_type SAMPLE_BUFFER::average_RMS_volume(char channel,
							     int
							     count_samples) 
{
  temp_avg = 0.0;
  if (count_samples == 0) count_samples = (int)buffer.size();
  for(buf_sizet = 0; buf_sizet < count_samples; buf_sizet++) {
    temp_avg += buffer[buf_sizet].sample[channel] *
      buffer[buf_sizet].sample[channel];
  }
  return(temp_avg / (single_type)count_samples);
}

void SAMPLE_BUFFER::length_in_samples(const int len) {
  // this routine should change the size of buffers without
  // losing sample-data or integrity

  if (buffer.size() != len) resize(len);
}

void SAMPLE_BUFFER::resize(int buffersize) {
  if (buffersize > buffer.size()) {
    delete iobuf_uchar;

    iobuf_uchar = new unsigned char [buffersize * SAMPLE_SPECS::ch_count * 2];
  }

  buffer.resize(buffersize);

  validindex = false;
}

SAMPLE_BUFFER::SAMPLE_BUFFER (int buffersize) :
  buffer(buffersize)
{

  validindex = false;

  MESSAGE_ITEM mitem;
  mitem << "Creating sample_buffer with size (samples) " << buffer.size() << ".";
  ecadebug->msg(1, mitem.to_string());

  buffer = vector<sample_type> (buffersize);

  iobuf_uchar = new unsigned char [buffer.size() * SAMPLE_SPECS::ch_count * 2];
}


SAMPLE_BUFFER::~SAMPLE_BUFFER (void) {
    while (!buffer.empty()) buffer.pop_back();

    delete iobuf_uchar;
}

SAMPLE_BUFFER& SAMPLE_BUFFER::operator=(const SAMPLE_BUFFER& x) {
  //
  // For better performance, doesn't copy IO-buffers.
  //
  
  if (this != &x) {
    //       throw(new ECA_ERROR("SAMPLEBUFFER", "Buffer size mismatch.", stop));
    if (buffer.size() != x.buffer.size()) resize(x.buffer.size());
    
    for(buf_sizet = 0; buf_sizet < buffer.size(); buf_sizet++) {
      for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
	buffer[buf_sizet].sample[channel_sizet] = x.buffer[buf_sizet].sample[channel_sizet];
      }
    }

    return *this;
  }
}

SAMPLE_BUFFER::SAMPLE_BUFFER (const SAMPLE_BUFFER& x) :
  buffer(x.buffer.size())
{
  //
  // For better performance, doesn't copy IO-buffers.
  //

  if (x.buffer.size() == 0)
    throw(new ECA_ERROR("SAMPLEBUFFER", "Tried to construct an empty buffer."));

  validindex = false;

  //  buffer = vector<sample_type> (x.buffer.size());
  
  MESSAGE_ITEM mitem;
  mitem << "Creating sample_buffer with size (samples) " << buffer.size() << ".";
  ecadebug->msg(1, mitem.to_string());

  for(buf_sizet = 0; buf_sizet < buffer.size(); buf_sizet++) {  
    for(channel_sizet = 0; channel_sizet < SAMPLE_SPECS::ch_count; channel_sizet++) {
      buffer[buf_sizet].sample[channel_sizet] = x.buffer[buf_sizet].sample[channel_sizet];
    }
  }
  
  iobuf_uchar = new unsigned char [buffer.size() * SAMPLE_SPECS::ch_count * 2];
}
