// ------------------------------------------------------------------------
// ecatools-fixdc.cpp: A simple command-line tools for fixing DC-offset.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <vector>
#include <cstdio>
#include <signal.h>

#include <kvutils.h>

#include <samplebuffer.h>
#include <eca-debug.h>
#include <eca-error.h>
#include <eca-controller.h>
#include <eca-main.h>
#include <eca-session.h>
#include <audiofx.h>
#include <eca-version.h>

#include "ecatools_fixdc.h"

int main(int argc, char *argv[])
{
  ecadebug->set_debug_level(0);
  ecadebug->disable();

  signal(SIGTERM, signal_handler);
  signal(SIGINT, signal_handler);
  signal(SIGQUIT, signal_handler);

  COMMAND_LINE cline = COMMAND_LINE (argc, argv);

  if (cline.size() < 2) {
    print_usage();
    return(1);
  }

  try {
    string filename;
    string tempfile;
    vector<double> dcfix_value (SAMPLE_BUFFER::ch_count);
    EFFECT_DCFIND* dcfind;
    EFFECT_DCFIX* dcfix;

    for(int n = 1; n < cline.size(); n++) {
      filename = cline[n];
      ecatools_fixdc_tempfile = string(tmpnam(NULL));
      ecatools_fixdc_tempfile += ".wav";
      for(int m = 0;m < 2; m++) {
	ECA_SESSION esession;
	ECA_CONTROLLER ectrl (&esession);
  
	ectrl.new_chainsetup("default");
	ectrl.activate_chainsetup("default");
	if (m == 0) {
	  cerr << "Calculating DC-offset for file \"" << filename << "\".\n";
	  ectrl.add_audio_device(filename, ECA_CONTROLLER::input);
	  ectrl.add_audio_device(ecatools_fixdc_tempfile, ECA_CONTROLLER::output);
	  dcfind = new EFFECT_DCFIND();
	  ectrl.add_chain_operator((CHAIN_OPERATOR*)dcfind);
	}
	else {
	  cerr << "Fixing DC-offset \"" << filename << "\"";
	  cerr << " (left: " << dcfix_value[SAMPLE_BUFFER::ch_left];
	  cerr << ", right: " << dcfix_value[SAMPLE_BUFFER::ch_right]
	       << ").\n";
	  ectrl.add_audio_device(ecatools_fixdc_tempfile, ECA_CONTROLLER::input);
	  ectrl.add_audio_device(filename, ECA_CONTROLLER::output);

	  dcfix = new EFFECT_DCFIX(dcfix_value[SAMPLE_BUFFER::ch_left],
				   dcfix_value[SAMPLE_BUFFER::ch_right]);
	  ectrl.add_chain_operator((CHAIN_OPERATOR*)dcfix);
	}
	ectrl.connect_active_chainsetup();
  
	ECA_PROCESSOR emain (&esession);
	emain.exec();
	
	if (m == 0) {
	  EFFECT_DCFIND* pdcfind = (EFFECT_DCFIND*)ectrl.get_chain_operator("default", 0);
	  for(int nm = 0; nm < SAMPLE_BUFFER::ch_count; nm++) {
	    dcfix_value[nm] = pdcfind->get_deltafix(nm);
	  }
	}
      }
      remove(ecatools_fixdc_tempfile.c_str());
    }
  }
  catch(ECA_ERROR* e) {
    cerr << "---\nERROR: [" << e->error_section() << "] : \"" << e->error_msg() << "\"\n\n";
  }
  catch(...) {
    cerr << "\nCaught an unknown exception.\n";
  }
  return(0);
}

void print_usage(void) {
  cerr << "****************************************************************************\n";
  cerr << "* [1mecatools_fixdc, " << ecatools_fixdc_version;
  cerr << " (linked to ecasound " << ecasound_version 
       << ")\n";
  cerr << "* (C) 1997-1999 Kai Vehmanen, released under GPL licence[0m \n";
  cerr << "****************************************************************************\n";

  cerr << "\nUSAGE: ecatools_fixdc file1 [ file2, ... fileN ]\n\n";
}

void signal_handler(int signum) {
  cerr << "Unexpected interrupt... cleaning up.\n";
  remove(ecatools_fixdc_tempfile.c_str());
  exit(1);
}




