// ------------------------------------------------------------------------
// eca-qtwavedata.cpp: Qt-widget for the actual wave data.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <vector>
#include <fstream>
#include <cmath>
#include <unistd.h>
#include <sys/stat.h>

#include <qapplication.h>
#include <qwidget.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qprogressdialog.h>
#include <qlabel.h>
#include <qpushbutton.h>

#include <kvutils.h>

#include "audioio.h"
#include "eca-qtwavedata.h"
#include "debug.h"

QEWaveData::QEWaveData(AUDIO_IO_DEVICE* iod, QWidget *parent=0, const char *name=0 )
        : QWidget( parent, name )
{
  startTimer(250);
  //  setMaximumHeight(100);
  setMinimumSize(400, 150);
  iodevice = iod;
  buffersize_save = iodevice->buffersize();
  xposcoord = 0;

  set_wave_color(QColor("royal blue"));
  set_background_color(Qt::white);
  set_position_color(Qt::black);
  set_minmax_color(Qt::black);
  set_zeroline_color(Qt::black);

  if (iodevice->is_realtime() == true ||
      iodevice->length_in_samples() == 0 ||
      iodevice->io_mode() == si_write) 
    valid_iodevice = false;
  else 
    valid_iodevice = true;

}

void QEWaveData::timerEvent( QTimerEvent * ) {
  if (!valid_iodevice) return;

  newxposcoord = (int)((double)iodevice->position_in_samples() / iodevice->length_in_samples() * width());
  if (newxposcoord != xposcoord) {
    // if updateWaveData was cancelled...
    p.begin(this);
    if (xposcoord * step < waveblocks.size()) {
      p.setPen(background_color);
      p.drawLine(xposcoord, 0, xposcoord, height());

      p.setPen(wave_color);
      ycoord = height() / 2 / SAMPLE_BUFFER::ch_count;

      for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
	ycoord += (height() / SAMPLE_BUFFER::ch_count) * ch;      
	//	p.drawLine(xposcoord, ycoord - (int)(waveblocks[xposcoord * step].min[ch] / 32767.0 * waveheight),
	//		   xposcoord, ycoord - (int)(waveblocks[xposcoord * step].max[ch] / 32767.0 * waveheight));
	p.drawLine(xposcoord, ycoord - (int)(waveblock_minimum(ch, xposcoord * step, step) / 32767.0 * waveheight),
		   xposcoord, ycoord - (int)(waveblock_maximum(ch, xposcoord * step, step) / 32767.0 * waveheight));
      }
    }
    else {
      p.setPen(background_color);
      p.drawLine(xposcoord, 0, xposcoord, height());
    }

    ycoord = height() / 2 / SAMPLE_BUFFER::ch_count;
    for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
      ycoord += (height() / SAMPLE_BUFFER::ch_count) * ch;
      //      p.setPen(position_color);
      p.setPen(minmax_color);
      p.drawPoint(xposcoord, ycoord - waveheight);
      p.drawPoint(xposcoord, ycoord + waveheight);
      p.setPen(zeroline_color);
      p.drawPoint(xposcoord, ycoord);
    }
    
    xposcoord = newxposcoord;
    p.setPen(position_color);
    p.drawLine(xposcoord, 0, xposcoord, height());
    p.end();
    //    repaint(true);
  }
}

void QEWaveData::updateWaveData(bool force)
{
  if (!valid_iodevice) return;
  calculate_buffersize();

  if (!force) {
    load_ews_data();
    if (waveblocks.size() > 0) return;
  }

  QProgressDialog progress ("Analyzing wave data...","Cancel",(int)(iodevice->length_in_seconds_exact()*10.0),0,0,true);
  progress.setProgress(0);
  progress.show();

  long save_pos = iodevice->position_in_samples();
  iodevice->first();
  iodevice->buffersize(buffersize);

  SAMPLE_BUFFER* t = new SAMPLE_BUFFER (buffersize);
  QEWaveBlock blocktmp;
  //  waveblocks.resize((int)(iodevice->length_in_samples() / QEWaveData::buffersize));
  vector<QEWaveBlock>::size_type bnum = 0;
  while(!iodevice->finished()) {
    iodevice->get_sample(t);
    //    emit setProgress(iodevice->position_in_seconds());   
    progress.setProgress((int)(iodevice->position_in_seconds_exact() * 10.0));
    // progress->repaint();
    if (progress.wasCancelled()) break;

    for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {

      blocktmp.max[ch] = (int)(32767.0 * t->max_value(ch) / SAMPLE_BUFFER::max_amplitude);
      //      cerr << blocktmp.max[ch];
      //      cerr << "-";
      blocktmp.min[ch] = (int)(32767.0 * t->min_value(ch) / SAMPLE_BUFFER::max_amplitude);
      //      cerr << blocktmp.min[ch];
      //      cerr << "<br>\n";
    }

    waveblocks.push_back(blocktmp);
    //    waveblocks[bnum] = blocktmp;
    ++bnum;
    if (bnum >= iodevice->length_in_samples() / buffersize) break;
  }

  iodevice->position_in_samples(save_pos);
  iodevice->buffersize(buffersize_save);

  save_ews_data();
}

void QEWaveData::paintEvent( QPaintEvent* e )
{
  if (!valid_iodevice) return;
  
  ur = e->rect(); 
  pix.resize(ur.size());
  //  pix.fill( this, ur.topLeft() );     // fill with widget background
  pix.fill(background_color);
  p.begin( &pix );

  samples_per_pixel = iodevice->length_in_samples() / width();
  ecadebug->msg(2,"eca-qtwaveform::drawWaveform(), samples_per_pixel " + kvu_numtostr(samples_per_pixel));

  // set step size
  step = samples_per_pixel / buffersize;
  if (step == 0) step = 1;

  waveheight = height() / SAMPLE_BUFFER::ch_count / 2 - 5;  

  // draw the actual waveform
  //  p.setPen(QColor("steel blue"));
  p.setPen(wave_color);
  xcoord = 0;
  for(t = 0; t < waveblocks.size(); t += step) {
    ycoord = height() / 2 / SAMPLE_BUFFER::ch_count;
    for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
      ycoord += (height() / SAMPLE_BUFFER::ch_count) * ch;
      //      p.drawLine(xcoord, ycoord - (int)(waveblocks[t].min[ch] / 32767.0 * waveheight),
      //	       xcoord, ycoord - (int)(waveblocks[t].max[ch] / 32767.0 * waveheight));
      p.drawLine(xcoord, ycoord - (int)(waveblock_minimum(ch, t, step) / 32767.0 * waveheight),
	       xcoord, ycoord - (int)(waveblock_maximum(ch, t, step) / 32767.0 * waveheight));
    }
    ++xcoord;
  }

  // draw max and mix limit lines
  p.setPen(minmax_color);
  ycoord = height() / 2 / SAMPLE_BUFFER::ch_count;
  for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
    ycoord += (height() / SAMPLE_BUFFER::ch_count) * ch;
    p.drawLine(0, ycoord - waveheight,
	       width(), ycoord - waveheight);
    p.drawLine(0, ycoord + waveheight,
	       width(), ycoord + waveheight);
    p.drawLine(0, ycoord,
	       width(), ycoord);
  }

  p.setPen(zeroline_color);
  p.drawLine(xposcoord, 0, xposcoord, height());
  
  p.end();
  bitBlt( this, ur.topLeft(), &pix );
}

int QEWaveData::waveblock_minimum(int channel, int from, int step) {
  int c = from;
  int minimum = waveblocks[c].min[channel];

  for(++c; c < from + step; c++) {
    if (waveblocks[c].min[channel] < minimum) 
      minimum = waveblocks[c].min[channel];
  }

  return(minimum);
}

int QEWaveData::waveblock_maximum(int channel, int from, int step) {
  int c = from;
  int maximum = waveblocks[c].max[channel];

  for(++c; c < from + step; c++) {
    if (waveblocks[c].max[channel] < maximum) 
      maximum = waveblocks[c].max[channel];
  }
  
  return(maximum);
}

void QEWaveData::calculate_buffersize(void) {
  buffersize = 16384;
  while(iodevice->length_in_samples() / buffersize <
	QApplication::desktop()->width()) {
    buffersize /= 2;
    if (buffersize == 0) {
      buffersize = 256; 
      break;
    }
  }
  ecadebug->msg(2, "(eca-qtwavedata) Set buffersize to " + kvu_numtostr(buffersize) + ".");
}

void QEWaveData::load_ews_data(void) { 
  //  FILE* f1 = fopen(string(iodevice->label() + ".ews").c_str(),"rb");
  //  if (!f1) return;
  //  waveblocks.resize(iodevice->length_in_samples() / buffersize);
  //  fread(&waveblocks, 1, sizeof(waveblocks), f1);
  //  fclose(f1);
  string newfile = iodevice->label() + ".ews";

  struct stat wfile;
  stat(iodevice->label().c_str(), &wfile);
  struct stat ewsfile;
  stat(newfile.c_str(), &ewsfile);

  if (wfile.st_ctime >= ewsfile.st_ctime) {
    waveblocks.resize(0);
    return;
  }

  fstream f1;
  f1.open(newfile.c_str(), ios::in);

  int buftemp;
  f1 >> buftemp;
  if (buftemp != buffersize) {
    f1.close();
    return;
  }
  waveblocks.resize(iodevice->length_in_samples() / buffersize);
  if (!f1) return;
  for(int t = 0; t < waveblocks.size(); t++) {
    for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
      f1 >> waveblocks[t].min[ch];
      f1 >> waveblocks[t].max[ch];
    }
  }
  f1.close();
}

void QEWaveData::save_ews_data(void) { 

  struct stat wfile;
  stat(iodevice->label().c_str(), &wfile);
  struct stat ewsfile;
  int res = stat(string(iodevice->label() + ".ews").c_str(), &ewsfile);
  if (wfile.st_size == ewsfile.st_size) {
    return;
  }

  fstream f1;
  f1.open(string(iodevice->label() + ".ews").c_str(), ios::out);
  //  FILE* f1 = fopen(string(iodevice->label() + ".ews").c_str(),"wb");
  if (!f1) return;
  f1 << buffersize << " ";
  for(int t = 0; t < waveblocks.size(); t++) {
    for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
      f1 << waveblocks[t].min[ch] << " ";
      f1 << waveblocks[t].max[ch] << " ";
    }
  }
  f1.close();
}
