// ------------------------------------------------------------------------
// eca-qtwavedata.cpp: Qt-widget for the actual wave data.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <string>
#include <vector>
#include <cmath>

#include <qwidget.h>
#include <qpainter.h>
#include <qpixmap.h>
#include <qprogressdialog.h>
#include <qlabel.h>
#include <qpushbutton.h>

#include <kvutils.h>

#include "audioio.h"
#include "eca-qtwavedata.h"
#include "debug.h"

QEWaveData::QEWaveData(AUDIO_IO_DEVICE* iod, QWidget *parent=0, const char *name=0 )
        : QWidget( parent, name )
{
  startTimer(250);
  //  setMaximumHeight(100);
  setMinimumSize(400, 150);
  iodevice = iod;
  buffersize_save = iodevice->buffersize();
  xposcoord = 0;

  if (iodevice->is_realtime() == true ||
      iodevice->length_in_samples() == 0 ||
      iodevice->io_mode() == si_write) 
    valid_iodevice = false;
  else 
    valid_iodevice = true;
}

void QEWaveData::timerEvent( QTimerEvent * ) {
  if (!valid_iodevice) return;

  newxposcoord = (int)((double)iodevice->position_in_samples() / iodevice->length_in_samples() * width());
  if (newxposcoord != xposcoord) {
    // if updateWaveData was cancelled...
    p.begin(this);
    if (xposcoord * step < waveblocks.size()) {
      p.setPen(backgroundColor());
      p.drawLine(xposcoord, 0, xposcoord, height());

      p.setPen(QColor("steel blue"));
      ycoord = height() / 2 / SAMPLE_BUFFER::ch_count;

      for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
	ycoord += (height() / SAMPLE_BUFFER::ch_count) * ch;      
	p.drawLine(xposcoord, ycoord - (int)(waveblocks[xposcoord * step].min[ch] / 32767.0 * waveheight),
		   xposcoord, ycoord - (int)(waveblocks[xposcoord * step].max[ch] / 32767.0 * waveheight));
      }
    }
    else {
      p.setPen(backgroundColor());
      p.drawLine(xposcoord, 0, xposcoord, height());
    }

    p.setPen(QColor(black));
    ycoord = height() / 2 / SAMPLE_BUFFER::ch_count;
    for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
      ycoord += (height() / SAMPLE_BUFFER::ch_count) * ch;
      p.setPen(QColor(black));
      p.drawPoint(xposcoord, ycoord - waveheight);
      p.drawPoint(xposcoord, ycoord + waveheight);
      p.drawPoint(xposcoord, ycoord);
    }
    
    xposcoord = newxposcoord;
    p.setPen(QColor(black));
    p.drawLine(xposcoord, 0, xposcoord, height());
    p.end();
    //    repaint(true);
  }
}

void QEWaveData::updateWaveData(void)
{
  if (!valid_iodevice) return;

  //  emit setTotalSteps(iodevice->length_in_seconds());
  //  repaint(true);

  QProgressDialog progress ("Analyzing wave data...","Cancel",(int)(iodevice->length_in_seconds_exact()*10.0),0,0,true);
  progress.setProgress(0);
  progress.show();

  long save_pos = iodevice->position_in_samples();
  iodevice->first();
  iodevice->buffersize(QEWaveData::buffersize);

  SAMPLE_BUFFER* t = new SAMPLE_BUFFER (QEWaveData::buffersize);
  QEWaveBlock blocktmp;
  //  waveblocks.resize((int)(iodevice->length_in_samples() / QEWaveData::buffersize));
  vector<QEWaveBlock>::size_type bnum = 0;
  while(!iodevice->finished()) {
    iodevice->get_sample(t);
    //    emit setProgress(iodevice->position_in_seconds());   
    progress.setProgress((int)(iodevice->position_in_seconds_exact() * 10.0));
    // progress->repaint();
    if (progress.wasCancelled()) break;

    for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {

      blocktmp.max[ch] = (int)(32767.0 * t->max_value(ch) / SAMPLE_BUFFER::max_amplitude);
      //      cerr << blocktmp.max[ch];
      //      cerr << "-";
      blocktmp.min[ch] = (int)(32767.0 * t->min_value(ch) / SAMPLE_BUFFER::max_amplitude);
      //      cerr << blocktmp.min[ch];
      //      cerr << "<br>\n";
    }

    waveblocks.push_back(blocktmp);
    //    waveblocks[bnum] = blocktmp;
    ++bnum;
    if (bnum >= iodevice->length_in_samples() / QEWaveData::buffersize) break;
  }

  iodevice->position_in_samples(save_pos);
  iodevice->buffersize(buffersize_save);
}

void QEWaveData::paintEvent( QPaintEvent* e )
{
  if (!valid_iodevice) return;
  
  ur = e->rect(); 
  pix.resize(ur.size());
  pix.fill( this, ur.topLeft() );     // fill with widget background
  p.begin( &pix );

  samples_per_pixel = iodevice->length_in_samples() / width();
  ecadebug->msg(2,"eca-qtwaveform::drawWaveform(), samples_per_pixel " + kvu_numtostr(samples_per_pixel));

  // set step size
  step = samples_per_pixel / QEWaveData::buffersize;
  if (step == 0) step = 1;

  waveheight = height() / SAMPLE_BUFFER::ch_count / 2 - 5;  

  // draw the actual waveform
  p.setPen(QColor("steel blue"));
  xcoord = 0;
  for(t = 0; t < waveblocks.size(); t += step) {
    ycoord = height() / 2 / SAMPLE_BUFFER::ch_count;
    for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
      ycoord += (height() / SAMPLE_BUFFER::ch_count) * ch;
      p.drawLine(xcoord, ycoord - (int)(waveblocks[t].min[ch] / 32767.0 * waveheight),
	       xcoord, ycoord - (int)(waveblocks[t].max[ch] / 32767.0 * waveheight));
    }
    ++xcoord;
  }

  // draw max and mix limit lines
  p.setPen(QColor(black));
  ycoord = height() / 2 / SAMPLE_BUFFER::ch_count;
  for(char ch = 0; ch < SAMPLE_BUFFER::ch_count; ch++) {
    ycoord += (height() / SAMPLE_BUFFER::ch_count) * ch;
    p.drawLine(0, ycoord - waveheight,
	       width(), ycoord - waveheight);
    p.drawLine(0, ycoord + waveheight,
	       width(), ycoord + waveheight);
    p.drawLine(0, ycoord,
	       width(), ycoord);
  }

  p.setPen(QColor(black));
  p.drawLine(xposcoord, 0, xposcoord, height());
  
  p.end();
  bitBlt( this, ur.topLeft(), &pix );
}
