// ------------------------------------------------------------------------
// eca-qtchainsetup: Qt widget representing a ECA_CHAINSETUP object and 
//                   its state.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>
#include <iostream>
#include <string>

#include <qwidget.h>
#include <qpushbutton.h>
#include <qmessagebox.h>

#include "chain.h"
#include "eca-controller.h"
#include "eca-chainsetup.h"

#include "eca-qtchain.h"
#include "eca-qtwaveform.h"
#include "qlistview-dumb.h"
#include "eca-qtchainsetup.h"

QEChainsetup::QEChainsetup (ECA_CONTROLLER* econtrol, const
			    ECA_CHAINSETUP* csetup, QWidget *parent=0,
			    const char *name=0) 
  : ctrl(econtrol), chainsetup(csetup)
{
  setMinimumSize( 600, 400 );
  //  setMaximumSize( 1024, 768 );

  startTimer(1000);

  string caption = "qtecasound - chainsetup: " + csetup->name();
  setCaption(caption.c_str());

  topLayout = new QVBoxLayout( this );
  gen_buttons = new QHBoxLayout();
  file_buttons = new QHBoxLayout();
  chain_buttons = new QHBoxLayout();

  init_filesetuplist();
  init_chainsetuplist();

  init_gen_buttons();
  init_file_buttons();
  init_chain_buttons();

  update_layout();
}

void QEChainsetup::not_implemented(void) {
  QMessageBox* mbox = new QMessageBox(this, "mbox");
  mbox->information(this, "qtecasound", "This feature is not implemented...",0);
}

void QEChainsetup::init_chainview(void) {
  QListViewItem* item = chainsetupview->selectedItem();
  if (item != 0) {
    const CHAIN* chain =
      chainsetup->get_chain_with_name(item->text(0).latin1());
    if (chain != 0) {
      QEChain* csetup = new QEChain(ctrl, chain);
      csetup->show();
    }
  }
}

void QEChainsetup::init_gen_buttons(void) {
  QFont butfont ("Helvetica", 12, QFont::Normal);

  QPushButton* ffocus = new QPushButton( "Focus to (f)iles", this, "ffocus" );
  ffocus->setFont(butfont);
  gen_buttons->addWidget( ffocus, 1, 0);

  QPushButton* cfocus = new QPushButton( "Focus to (c)hains", this, "cfocus" );
  cfocus->setFont(butfont);
  gen_buttons->addWidget( cfocus, 1, 0);

  QPushButton* quit = new QPushButton( "(Q)uit", this, "quit" );
  quit->setFont(butfont);
  gen_buttons->addWidget( quit, 2, 0);

  QObject::connect( cfocus, SIGNAL(clicked()), chainsetupview,
		    SLOT(setFocus()));
  QObject::connect( ffocus, SIGNAL(clicked()), filesetupview,
		    SLOT(setFocus()));
  QObject::connect( quit, SIGNAL(clicked()), this, SLOT(close()));
  //  connect(quit, SIGNAL(clicked()), this, SLOT(emsg_quit()) );
}

void QEChainsetup::update_layout(void) {
  // if (topLayout != 0) delete topLayout;
  // topLayout = new QVBoxLayout( this );

  topLayout->addLayout(gen_buttons, 0);

  topLayout->addLayout(file_buttons, 0);
  topLayout->addWidget(filesetupview, 0, 0);

  topLayout->addLayout(chain_buttons, 0);
  topLayout->addWidget(chainsetupview, 0, 0);
}

void QEChainsetup::init_file_buttons(void) { 
  QFont butfont ("Helvetica", 12, QFont::Normal);

  QPushButton* newb = new QPushButton( "(N)ew file", this, "newb" );
  newb->setFont(butfont);
  file_buttons->addWidget( newb, 1, 0);

  QPushButton* remove = new QPushButton( "(R)emove from setup", this, "remove" );
  remove->setFont(butfont);
  file_buttons->addWidget( remove, 1, 0);

  QPushButton* change = new QPushButton( "Change (s)ettings", this, "change" );
  change->setFont(butfont);
  file_buttons->addWidget( change, 1, 0);


  QPushButton* wform = new QPushButton( "(W)ave form view", this, "wform" );
  wform->setFont(butfont);
  file_buttons->addWidget( wform, 2, 0);

  QObject::connect( newb, SIGNAL(clicked()), this, SLOT(not_implemented()));
  QObject::connect( remove, SIGNAL(clicked()), this, SLOT(not_implemented()));
  QObject::connect( change, SIGNAL(clicked()), this, SLOT(not_implemented()));
  QObject::connect( wform, SIGNAL(clicked()), this, SLOT(init_waveform()));
}

void QEChainsetup::init_chain_buttons(void) { 
  QFont butfont ("Helvetica", 12, QFont::Normal);

  QPushButton* newb = new QPushButton( "New ch(a)in", this, "newb" );
  newb->setFont(butfont);
  chain_buttons->addWidget( newb, 1, 0);

  QPushButton* delchain = new QPushButton( "(D)elete chain", this, "delchain" );
  delchain->setFont(butfont);
  chain_buttons->addWidget( delchain, 1, 0);

  QPushButton* enabdis = new QPushButton( "(E)n/disable", this, "enabdis" );
  enabdis->setFont(butfont);
  chain_buttons->addWidget( enabdis, 1, 0);

  QPushButton* setup = new QPushButton( "Setu(p)", this, "setup" );
  setup->setFont(butfont);
  chain_buttons->addWidget( setup, 2, 0);

  QObject::connect( enabdis, SIGNAL(clicked()), this, SLOT(button_switch_chain_mode()));
  QObject::connect( delchain, SIGNAL(clicked()), this, SLOT(not_implemented()));
  QObject::connect( setup, SIGNAL(clicked()), this, SLOT(init_chainview()));
  QObject::connect( newb, SIGNAL(clicked()), this, SLOT(not_implemented()));
}
 
void QEChainsetup::button_switch_chain_mode(void) {
  string name = ""; 

  QListViewItem* item = chainsetupview->selectedItem();
  if (item != 0) {
    ctrl->command("c " + string(item->text(0).latin1()));
  }
}

void QEChainsetup::init_filesetuplist (void) {

  filesetupview = new QListView_dumb(this, "filesetupview");

  filesetupview->setMinimumSize( 600, 100 );
  //  filesetupview->setMaximumSize( width() / 2, height() / 2);

  filesetupview->addColumn("File");
  filesetupview->addColumn("Mode");
  filesetupview->addColumn("Bits/Ch/Rate");
  filesetupview->addColumn("Position");
  filesetupview->addColumn("Length");
  filesetupview->addColumn("Realtime");
  filesetupview->addColumn("Chains");

  //  chainsetupview->addColumn("Chains");  
  //  chainsetupview->addColumn("Active");  

  filesetupview->setAllColumnsShowFocus( TRUE ); 

  update_filesetuplist();

  int pixelsleft = width() - 10;
  for(int n = 1; n < 7; n++) {
    pixelsleft -= filesetupview->columnWidth(n);
  }  

  if (pixelsleft > 0) {
    filesetupview->setColumnWidthMode(0, QListView::Manual);
    filesetupview->setColumnWidth(0, pixelsleft);
  }

  filesetupview->setGeometry(0, 0, width(), 100);

  filesetupview->show();
}

void QEChainsetup::update_filesetuplist (bool clean) {
  QListViewItem* selected = filesetupview->selectedItem();
  QString selname = ""; 
  if (selected != 0) selname = selected->text(0);

  if (clean) {
    filesetupview->clear();
    update_filesetup_clean(chainsetup->inputs, selname);    
    update_filesetup_clean(chainsetup->outputs, selname);
  }
  else {
    update_filesetup(chainsetup->inputs, selname);    
    update_filesetup(chainsetup->outputs, selname);
  }

  filesetupview->triggerUpdate();
}

void QEChainsetup::update_filesetup_clean (const vector<AUDIO_IO_DEVICE*>&
					   flist, const QString& selname) {
  aiod_sizet = 0;

  while(aiod_sizet < flist.size()) {
    //    cerr << "Adding a new one!\n";
    cs_namestring = QString(flist[aiod_sizet]->label().c_str());

    cs_chainstring = ""; 
    if (flist[aiod_sizet]->io_mode() == si_read) {
      cs_modestring = "input"; 
      cs_chainstring = ctrl->connected_chains_input(aiod_sizet).c_str();
      cs_posstring.sprintf("%.2f", chainsetup->inputs[aiod_sizet]->position_in_seconds_exact());
      cs_lenstring.sprintf("%.2f", chainsetup->inputs[aiod_sizet]->length_in_seconds_exact());
    }
    else {
      cs_modestring = "output"; 
      cs_chainstring = ctrl->connected_chains_output(aiod_sizet).c_str();
      cs_posstring.sprintf("%.2f", chainsetup->outputs[aiod_sizet]->position_in_seconds_exact());
      cs_lenstring.sprintf("%.2f", chainsetup->outputs[aiod_sizet]->length_in_seconds_exact());
    }

    if (flist[aiod_sizet]->is_realtime()) 
      cs_rtstring = "yes";
    else
      cs_rtstring = "no";

  
    cs_format.sprintf("%d/%d/%d", 
		   flist[aiod_sizet]->format().bits,
		   flist[aiod_sizet]->format().channels,
		   flist[aiod_sizet]->format().srate);
    newitem = new QListViewItem(filesetupview,
				cs_namestring,
				cs_modestring,
				cs_format,
				cs_posstring,
				cs_lenstring,
				cs_rtstring,
				cs_chainstring);
    if (newitem->text(0) == selname) filesetupview->setSelected(newitem, true);
    ++aiod_sizet;
  }
}

void QEChainsetup::update_filesetup (const vector<AUDIO_IO_DEVICE*>&
  				     flist, const QString& selname) {

  for(aiod_sizet = 0; aiod_sizet < flist.size(); aiod_sizet++) {
    newitem = filesetupview->firstChild();
    while(newitem != 0) {
      if (newitem->text(0) ==
	  QString(flist[aiod_sizet]->label().c_str()))
	break;
      newitem = newitem->nextSibling();
    }
    if (newitem == 0) continue;

    if (flist[aiod_sizet]->io_mode() == si_read) {
      cs_posstring.sprintf("%.2f", chainsetup->inputs[aiod_sizet]->position_in_seconds_exact());
      cs_lenstring.sprintf("%.2f", chainsetup->inputs[aiod_sizet]->length_in_seconds_exact());
    }
    else {
      cs_posstring.sprintf("%.2f", chainsetup->outputs[aiod_sizet]->position_in_seconds_exact());
      cs_lenstring.sprintf("%.2f", chainsetup->outputs[aiod_sizet]->length_in_seconds_exact());
    }
    
    newitem->setText(3, cs_posstring);
    newitem->setText(4, cs_lenstring);
  }
}

void QEChainsetup::init_chainsetuplist (void) {

  chainsetupview = new QListView_dumb(this, "chainsetupview");

  //  chainsetupview->setMaximumSize( width() / 2, height() / 2);
  chainsetupview->setMinimumSize( 600, 100 );

  chainsetupview->addColumn("Chain");
  chainsetupview->addColumn("Chain operators");  
  chainsetupview->addColumn("Controllers");  
  chainsetupview->addColumn("Enabled");  
  //  chainsetupview->addColumn("Chains");  
  //  chainsetupview->addColumn("Active");  

  chainsetupview->setAllColumnsShowFocus( TRUE ); 

  update_chainsetuplist_clean();

  int pixelsleft = width() - 10;
  for(int n = 1; n < 4; n++) 
    pixelsleft -= chainsetupview->columnWidth(n);
  
  if (pixelsleft > 0) {
    chainsetupview->setColumnWidthMode(0, QListView::Manual);
    chainsetupview->setColumnWidth(0, pixelsleft);
  }

  chainsetupview->setGeometry(0, 0, width(), 100);

  chainsetupview->show();
}

void QEChainsetup::update_chainsetuplist_clean(void) {
  QListViewItem* selected = chainsetupview->selectedItem();
  QString selname = ""; 
  if (selected != 0) selname = selected->text(0);

  chainsetupview->clear();

  vector<CHAIN*>::const_iterator p = chainsetup->chains.begin();
  while(p != chainsetup->chains.end()) {
    //    cerr << "Adding a new one!\n";

    QString astring;
    if ((*p)->is_enabled())
      astring = "yes";
    else
      astring = "no";

    newitem = new QListViewItem(chainsetupview,
				(*p)->cname().c_str(),
				kvu_numtostr((*p)->chainops.size()).c_str(),
				kvu_numtostr((*p)->gcontrollers.size()).c_str(),
				astring);
    if (newitem->text(0) == selname) chainsetupview->setSelected(newitem, true);
    ++p;
  }

  chainsetupview->triggerUpdate();
}

void QEChainsetup::update_chainsetuplist () {
  vector<CHAIN*>::const_iterator p = chainsetup->chains.begin();
  while(p != chainsetup->chains.end()) {
    newitem = chainsetupview->firstChild();
    while(newitem != 0) {
      if (newitem->text(0) ==
	  QString((*p)->cname().c_str()))
	break;
      newitem = newitem->nextSibling();
    }
    if (newitem == 0) {
      ++p;
      continue;
    }

    QString astring;
    if ((*p)->is_enabled())
      astring = "yes";
    else
      astring = "no";

    newitem->setText(3, astring);

    ++p;
  }

  chainsetupview->triggerUpdate();
}

void QEChainsetup::init_waveform(void) {

  ctrl->stop();
  QEWaveForm* wform = 0;
  string name = ""; 

  //  cerr << "a";
  QListViewItem* item = filesetupview->selectedItem();
  if (item != 0) {
    name = string(item->text(0).latin1());
    AUDIO_IO_DEVICE* dev = 0;
    dev = ctrl->lock_audio_device(chainsetup->name(), name);
    if (dev != 0) {
      //      cerr << "b";
      wform = new QEWaveForm(dev);
      //      cerr << "c";
      if (wform != 0) {
	wform->updateWaveData();
	//	cerr << "e";
	ctrl->release_audio_device(chainsetup->name(), name);
	//	cerr << "f";
	wform->show();  
      }
      //      cerr << "g";
    }
  }
  //  connect(wform, SIGNAL(setProgress(int)), progress,
   //  SLOT(setProgress(int)));

  //  if (was_running) ctrl.command("start");
}

void QEChainsetup::keyPressEvent(QKeyEvent* kevent) {
  switch(tolower(kevent->ascii())) {
  case 'a': 
    {
      // new chain
      not_implemented();
      break;
    }

  case 'c': 
    {
      // focus to chains
      chainsetupview->setFocus();
      break;
    }

  case 'e': 
    {
      // enable/disable chain
      button_switch_chain_mode();
      break;
    }

  case 'f': 
    {
      // focus to files
      filesetupview->setFocus();
      break;
    }

  case 'n': 
    {
      // new file
      not_implemented();
      break;
    }

  case 'p': 
    {
      init_chainview();
      break;
    }

  case 'q': 
    {
      close();
      break;
    }

  case 'r': 
    {
      // remove file
      not_implemented();
      break;
    }

  case 's': 
    {
      // change file
      not_implemented();
      break;
    }

  case 'w': 
    {
      // 
      init_waveform();
      break;
    }
  }
  kevent->ignore();
}

void QEChainsetup::timerEvent( QTimerEvent * ) {
  update_filesetuplist(false);
  update_chainsetuplist();
}



