// ------------------------------------------------------------------------
// audioio-cdr.cpp: CDDA/CDR audio file format input/output
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <cmath>
#include <string>
#include <cstring>
#include <cassert>

#include "samplebuffer.h"
#include "audioio.h"
#include "audioio-cdr.h"

#include "error.h"
#include "debug.h"

CDRFILE::CDRFILE(const string& name, const SIMODE mode, const
		 AIO_PARAMS& fmt, int bsize) 
  :  AUDIO_IO_DEVICE(name, mode, fmt, bsize) {

    if (mode == si_read) {
      fobject=fopen(name.c_str(),"rb");
    }
    else if (mode == si_write) {
      fobject=fopen(name.c_str(),"w+b");

    }
    
    curpos_value = 0;
    samples = 0;

    AIO_PARAMS sffmt;

    sffmt.channels = 2;
    sffmt.bits = 16;
    sffmt.srate = 44100;

    format(sffmt);

    if (mode == si_read) {
        get_length_in_bytes();
    }
    finished(false);
}

CDRFILE::~CDRFILE(void) {
    pad_to_sectorsize();

    fclose(fobject);
}

void CDRFILE::get_sample(SAMPLE_BUFFER* t) {
  samples_read = fread(t->iobuf_uchar, format().align, buffersize(), fobject);

  if (!ferror(fobject)) {  
    t->iobuf_to_buffer(format().bits, false, format().channels);
    t->length_in_samples(samples_read);
    curpos_value += samples_read * format().align;
    if (feof(fobject)) finished(true);
  }
  else
    throw(new ECA_ERROR("AUDIOIO-CDR","Read error!"));
}

void CDRFILE::pad_to_sectorsize(void) {
    int padsamps = CDRFILE::sectorsize - (samples % CDRFILE::sectorsize);

    if (padsamps == CDRFILE::sectorsize) {
      return;
    }
    for(int n = 0; n < padsamps; n++) fputc(0,fobject);

    assert(ftell(fobject) %  CDRFILE::sectorsize == 0);
}

void CDRFILE::put_sample(SAMPLE_BUFFER* t) {
    curpos_value = curpos_value + (t->length_in_samples() * format().align);
    t->buffer_to_iobuf(format().bits, false, format().channels);
    fwrite(t->iobuf_uchar, format().align, t->length_in_samples(), fobject);
    samples = samples + t->length_in_samples() * format().align;
    if (ferror(fobject) || t->length_in_samples() <  t->length_in_samples()) finished(true);
}

void CDRFILE::first(void) {
    curpos_value = 0;
    finished(false);
    fseek(fobject,0,SEEK_SET);
}

void CDRFILE::last(void) {
    curpos_value = get_curpos();
    fseek(fobject,-format().align,SEEK_CUR);
}

long int CDRFILE::length_in_bytes(void)  const { return(length_in_bytes_value); }
long int CDRFILE::get_length_in_bytes(void) {
    long int save = ftell(fobject);
    fseek(fobject,0,SEEK_END);
    length_in_bytes_value = ftell(fobject);
    fseek(fobject,save,SEEK_SET);
    return(length_in_bytes_value);
}

long int CDRFILE::get_curpos(void) {
    long int save = ftell(fobject);
    first();
    curpos_value = save - ftell(fobject);
    fseek(fobject,save,SEEK_SET);
    return(curpos_value);
}

void CDRFILE::position_in_samples(long pos) {
    long temp = pos * format().align;
    first();
    finished(false);
    
    if (temp < 0) return;

    long save = ftell(fobject);
    fseek(fobject, temp, SEEK_SET);
    curpos_value = ftell(fobject) - save;
}
