#ifndef _AUDIOFX_H
#define _AUDIOFX_H

#include <vector>

#include <kvutils.h>

#include "debug.h"
#include "chainop.h"
#include "samplebuffer.h"

class AUDIOFX_BASE : public CHAIN_OPERATOR {
//
// The base for all audio effects.
//

public:
  
  // Functions returning info about effect and its parameters.
  // ---

  virtual void set_parameter(int param, double value) { }
  virtual double get_parameter(int param) { }

  AUDIOFX_BASE(void) { support_for_dynamic_parameters(false); }
  virtual AUDIOFX_BASE* clone(void) = 0;
  virtual ~AUDIOFX_BASE (void) { }
};

#include "audiofx_filter.h"
#include "audiofx_timebased.h"

class EFFECT_AMPLIFY : public AUDIOFX_BASE {
//
// Amplify effect with clip control. 
//
  double kerroin;
  int nm, num_of_clipped, maxnum_of_clipped;

 public:

  string label(void) { return("Amplify"); }
  string id_string(void) { return("ea"); }
  string params(void) { return("amp-%, max-clipped"); }
  int number_of_params(void) { return(2); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_AMPLIFY* clone(void)  { return new EFFECT_AMPLIFY(*this); }
  EFFECT_AMPLIFY (double multiplier_percent, int max_clipped);
  ~EFFECT_AMPLIFY (void) { }
};

class EFFECT_AMPLIFY_SIMPLE: public AUDIOFX_BASE {
//
// Amplify effect without clip control.
//
  double kerroin;
  int nm;

 public:

  string label(void) { return("Simple amplify"); }
  string id_string(void) { return("eas"); }
  string params(void) { return("amp-%"); }
  int number_of_params(void) { return(1); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_AMPLIFY_SIMPLE* clone(void)  { return new EFFECT_AMPLIFY_SIMPLE(*this); }
  EFFECT_AMPLIFY_SIMPLE (double multiplier_percent);
  ~EFFECT_AMPLIFY_SIMPLE (void) {  }
};

class EFFECT_ANALYZE : public AUDIOFX_BASE {
//
// Analyzes sample volume.
//
    int nm;
    unsigned long int num_of_samples[SAMPLE_BUFFER::ch_count]; // number of samples processed
    unsigned long int ranges[16][SAMPLE_BUFFER::ch_count];     // tyyliin ... -32768->30720 ... 30719->32767

 public:
    
    string label(void) { return("Volume-analyze"); }
    string id_string(void) { return("eg"); }

    void process(SAMPLE_BUFFER::sample_type *insample);
    void status(string*);

    EFFECT_ANALYZE* clone(void)  { return new EFFECT_ANALYZE(*this); }
    EFFECT_ANALYZE (const EFFECT_ANALYZE& x);
    EFFECT_ANALYZE (void);
    ~EFFECT_ANALYZE (void);
};

class EFFECT_COMPRESS : public AUDIOFX_BASE {
// 
// Dynamic compressor. 
// 
  double crate;
  double threshold;

  unsigned short int nm;  // temp value (channel)
  double delta, new_value;
  bool first_time;
  SAMPLE_BUFFER::single_type s[SAMPLE_BUFFER::ch_count];
  SAMPLE_BUFFER::sample_type temp;     // temporary values
  SAMPLE_BUFFER::sample_type lastin, lastout;

 public:

  string label(void) { return("Compress"); }
  string id_string(void) { return("ec"); }
  string params(void) { return("compression_rate, threshold"); }
  int number_of_params(void) { return(2); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_COMPRESS* clone(void)  { return new EFFECT_COMPRESS(*this); }
  EFFECT_COMPRESS (const EFFECT_COMPRESS& x);
  EFFECT_COMPRESS::EFFECT_COMPRESS (double compress_rate, double thold);
};

class EFFECT_DCFIND : public AUDIOFX_BASE {
//
// Calculates DC-offset.
//

private:

  double pos_sum[SAMPLE_BUFFER::ch_count], neg_sum[SAMPLE_BUFFER::ch_count];
  double num_of_samples[SAMPLE_BUFFER::ch_count];

  SAMPLE_BUFFER::single_type deltafix[SAMPLE_BUFFER::ch_count];
  SAMPLE_BUFFER::single_type tempval;
  
  unsigned short int nm;  // temp value (channel)

public:
    
  string label(void) { return("DC-Find"); }
  string id_string(void) { return("ezf"); }

  void status(string*);
  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_DCFIND* clone(void)  { return new EFFECT_DCFIND(*this); }
  EFFECT_DCFIND (const EFFECT_DCFIND& x);
  EFFECT_DCFIND::EFFECT_DCFIND (void);
};

class EFFECT_DCFIX : public AUDIOFX_BASE {
//
// Fixes DC-offset.
//

private:

  SAMPLE_BUFFER::single_type deltafix[SAMPLE_BUFFER::ch_count];
  unsigned short int nm;  // temp value (channel)

public:

  string label(void) { return("DC-Fix"); }
  string id_string(void) { return("ezx"); }
  string params(void) { return("delta-value-left, delta_value-right)"); }
  int number_of_params(void) { return(2); }

  void set_parameter(int param, double value);
  double get_parameter(int param);

  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_DCFIX* clone(void)  { return new EFFECT_DCFIX(*this); }
  EFFECT_DCFIX (const EFFECT_DCFIX& x);
  EFFECT_DCFIX::EFFECT_DCFIX (double delta_left, double delta_right);
};

class EFFECT_NOISEGATE_MONO : public AUDIOFX_BASE {
//
// Noise gate. A mono-summed signal is used to control the gate.
//
    double th_level;
    double th_time;
    double atime, htime, rtime;
    
    double th_time_lask;
    double attack_lask;
    double hold_lask;
    double release_lask;

    SAMPLE_BUFFER::sample_type temp_sample;

    double kerroin;
    
    unsigned short int nm;

    enum { ng_waiting, ng_attacking, ng_active, ng_holding, ng_releasing } ng_status;

    public:

    string label(void) { return("Noisegate mono"); }
    string id_string(void) { return("enm"); }
    string params(void) { return("threshold_level, th_time, attack, hold, release"); }
    int number_of_params(void) { return(5); }
    void set_parameter(int param, double value);
    double get_parameter(int param);

    void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_NOISEGATE_MONO* clone(void)  { return new EFFECT_NOISEGATE_MONO(*this); }
  EFFECT_NOISEGATE_MONO (double thlevel_percent, double thtime, double atime, double htime, double rtime);
};

class EFFECT_NORMALIZE : public AUDIOFX_BASE {
//
// Volume normalizing. Find the maximum value by which the whole signal can be 
// multiplied without clipping. You can use this with the amplify effect.
//

private:
  
  double max;
  double kerroin;
  int nm;

public:

  string label(void) { return("Normalize"); }
  string id_string(void) { return("ev"); }

  void status(string*);
  void process(SAMPLE_BUFFER::sample_type *insample);

  EFFECT_NORMALIZE* clone(void)  { return new EFFECT_NORMALIZE(*this); }
  EFFECT_NORMALIZE::EFFECT_NORMALIZE (void);
};

class EFFECT_NORMAL_PAN : public AUDIOFX_BASE {

private:

    double l_kerroin, r_kerroin;

public:

    string label(void) { return("Normal pan"); }
    string id_string(void) { return("epp"); }
    string params(void) { return("right-%"); }
    int number_of_eparams(void) { return(1); }
    void set_parameter(int param, double value);
    double get_parameter(int param);


    void process(SAMPLE_BUFFER::sample_type *insample);
    
    EFFECT_NORMAL_PAN* clone(void)  { return new EFFECT_NORMAL_PAN(*this); }
    EFFECT_NORMAL_PAN(double right_percent);
};

#endif


