// ------------------------------------------------------------------------
// eca-session.cpp: Ecasound runtime setup and parameters.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <config.h>

#include <string>
#include <cstring>
#include <algorithm>
#include <vector>
#include <pthread.h>

#include <kvutils.h>

#include "eca-resources.h"

#include "chain.h"
#include "audiofx.h"
#include "audioio.h"

#include "error.h"
#include "debug.h"

#include "eca-comhelp.h"
#include "eca-session.h"
#include "eca-chainsetup.h"

ECA_SESSION::ECA_SESSION(void) {
  set_defaults();
  ecaresources.load();
  active_chainsetup = 0;
}

ECA_SESSION::~ECA_SESSION(void) {
  ecadebug->msg(1,"ECA_SESSION destructor!");
  ecaresources.save();

  for(vector<ECA_CHAINSETUP*>::iterator q = chainsetups.begin(); q != chainsetups.end(); q++) {
    delete *q;
  }

  ecadebug->control_flow("Closing session");
}

ECA_SESSION::ECA_SESSION(COMMAND_LINE& cline) {
  set_defaults();
  ecaresources.load();
  active_chainsetup = 0;

  interpret_general_options(cline);

  if (chainsetups.size() == 0) {
    ECA_CHAINSETUP* comline_setup = new ECA_CHAINSETUP(&ecaresources, cline);
    add_setup(comline_setup);
    set_active_setup(comline_setup->name());
  }

  // These were added for convenience...
  if (cline.size() < 2) {
    // No parameters, let's give some help.
    interpret_general_option("-h");
  }
  if (inputs->size() == 0) {
    // Still no inputs? If not in interactive mode, there really isn't
    // anything left to do.
    if (iactive == false) 
      throw(new ECA_ERROR("ECA_SESSION","Nothing to do!"));
  }
}

void ECA_SESSION::set_defaults(void) {
  //  buffersize = SB_BUFFERSIZE; // set to default buffersize

  if (ecaresources.resource("default-to-interactive-mode")
      == "true") iactive = true;
  else
    iactive = false;

  if (ecaresources.resource("default-to-raisepriority")
      == "true") raisepriority = true;
  else
    raisepriority = false;

  enable_outqueue = false;
  mthreaded_use_of_session = false;
  mthreaded_io = false;
  mixmode = ep_mm_auto;
  multitrack_mode = false;

  loop_active = false;
  loop_start_pos = 0;
  loop_end_pos = 0;

  sfx = true;
  loop_counter = 0;

  pthread_mutex_init(&status_lock, NULL);
  status(ep_status_notready);
}

void ECA_SESSION::init_mutexes(void) {
  ecadebug->msg(1,"(eca-main) Initializing chain mutexes.");
  while(chain_locks.size() != 0) {
    delete chain_locks.back();
    chain_locks.pop_back();
  }
  for(int n = 0; n < chains->size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    chain_locks.push_back(new_mutex);
  }

  ecadebug->msg(1,"(eca-main) Initializing input mutexes.");
  while(in_locks.size() != 0) {
    delete in_locks.back();
    in_locks.pop_back();
  }
  for(int n = 0; n < chains->size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    in_locks.push_back(new_mutex);
  }

  ecadebug->msg(1,"(eca-main) Initializing output mutexes.");
  while(out_locks.size() != 0) {
    delete out_locks.back();
    out_locks.pop_back();
  }
  for(int n = 0; n < outputs->size(); n++) {
    pthread_mutex_t* new_mutex = new pthread_mutex_t;
    pthread_mutex_init(new_mutex, NULL);
    out_locks.push_back(new_mutex);
  }
}

void ECA_SESSION::add_new_setup(const string& name) {
  ECA_CHAINSETUP* newsetup = new ECA_CHAINSETUP (&ecaresources, name,
						  false);
  add_setup(newsetup);
}

void ECA_SESSION::add_setup(ECA_CHAINSETUP* comline_setup) {
  chainsetups.push_back(comline_setup);
}

void ECA_SESSION::set_active_setup(const string& name) {
  if (active_chainsetup != 0 && name == active_chainsetup->name()) return;
  vector<ECA_CHAINSETUP*>::const_iterator p = chainsetups.begin();
  while(p != chainsetups.end()) {
    if ((*p)->name() == name) {
      ecadebug->msg("(eca-session) Setting chainsetup \"" + name + "\" active.");
      (*p)->enable();
      if ((*p)->is_valid() == false) {
	throw(new ECA_ERROR("ECA-SESSION","Chainsetup not valid, can't set it active.", ECA_ERROR::retry));
      }
      else {
	if (active_chainsetup != 0) active_chainsetup->disable();
	active_chainsetup = *p;
	connect_active_setup();
      }
    }
    ++p;
  }
}

void ECA_SESSION::save_active_setup(const string& name) {
  active_chainsetup->save_to_file(name);
}

void ECA_SESSION::load_active_setup(const string& name) {
  vector<ECA_CHAINSETUP*>::const_iterator p = chainsetups.begin();
  while(p != chainsetups.end()) {
    if ((*p)->name() == name)
      throw(new ECA_ERROR("ECA-SESSION","Chainsetup \"" + name + 
			  "\" already exists.", ECA_ERROR::retry));
    ++p;
  }

  ECA_CHAINSETUP* new_setup = new ECA_CHAINSETUP(&ecaresources, name);
  add_setup(new_setup);
  set_active_setup(new_setup->name());
}

void ECA_SESSION::connect_active_setup(void) {
  inputs = &(active_chainsetup->inputs);
  select_master_input();

  outputs = &(active_chainsetup->outputs);
  chains = &(active_chainsetup->chains);

  init_mutexes();

  buffersize = active_chainsetup->buffersize;
  mixmode = active_chainsetup->mixmode;
  raisepriority = active_chainsetup->raisepriority;

  while(inslots.size() != 0) inslots.pop_back();
  while(inslots.size() != inputs->size()) inslots.push_back(SAMPLE_BUFFER(buffersize));

  while(outslots.size() != 0) outslots.pop_back();
  while(outslots.size() != outputs->size()) outslots.push_back(SAMPLE_BUFFER(buffersize));

}

void ECA_SESSION::select_master_input(void) {
  master_input_length = 0;
  CHAIN::aio_id_type p = 0;
  while (p < inputs->size()) {
    if ((*inputs)[p]->length_in_samples() > master_input_length) {
      master_input_length = (*inputs)[p]->length_in_samples();
      master_input_id = p;
    }
    ++p;
  }

  ecadebug->msg(1, "(eca-session) Setting master_input_id: " +
		kvu_numtostr(master_input_id) + ", length: " +
		kvu_numtostr(master_input_length) + ".");
}

void ECA_SESSION::interpret_general_options(COMMAND_LINE& cline) {
  cline.back_to_start();
  while(cline.ready()) {
    string temp = cline.next_argument();
    interpret_general_option(temp);
  }

 cline.back_to_start();    
 while(cline.ready()) {
   string argu = cline.next_argument();
   string argu_param = cline.next();
   if (argu_param.size() > 0) {
     if (argu_param[0] == '-') {
       cline.previous();
       argu_param == "";
     }
   }

    interpret_chainsetup(argu, argu_param);
  }
}

void ECA_SESSION::interpret_general_option (const string& argu) {
  //  if (argu.size() == 0) throw(new ECA_ERROR("ECAPARAMS", "empty argument", retry));

  if (argu.size() < 2) return;
  if (argu[0] != '-') return;
  switch(argu[1]) {
  case 'c':
    iactive = true;
    ecadebug->msg(0, "(eca-session) Interactive mode enabled."); 
    break;

  case 'd':
    {
      ecadebug->set_debug_level(atoi(get_argument_number(1, argu).c_str()));
      MESSAGE_ITEM mtempd;
      mtempd << "(eca-session) Set debug level to: " << ecadebug->get_debug_level();
      ecadebug->msg(mtempd.to_string());
      break;
    }
  case 'h':      // help!
    cout << ecasound_parameter_help();
    break;

  default: { }
  }
}

void ECA_SESSION::interpret_chainsetup (const string& argu,
					const string& toinen) {
  if (argu.size() == 0) return;
  
  string tname = get_argument_number(1, argu);
  if (tname == "") tname = toinen;
  else if (tname[0] == '-') tname = toinen;
  else tname = argu;

  if (argu.size() < 2) return;
  switch(argu[1]) {
  case 's':
    load_active_setup(tname);
    break;
  }
}

bool ECA_SESSION::is_slave_output(CHAIN::aio_id_type aiod) {
  if ((*outputs)[aiod]->is_realtime()) return(false);
  vector<CHAIN*>::iterator q = chains->begin();
  while(q != chains->end()) {
    if ((*q)->outputid == aiod) {
      if ((*inputs)[(*q)->inputid]->is_realtime()) {
	ecadebug->msg(2,"(eca-session) slave output detected: " + (*outputs)[(*q)->outputid]->label());
	return(true);
      }
    }
    ++q;
  }
  return(false);
}

void ECA_SESSION::status(const EP_STATUS temp) {
  pthread_mutex_lock(&status_lock);
  ep_status = temp;
  pthread_mutex_unlock(&status_lock);
}

EP_STATUS ECA_SESSION::status(void) {
  pthread_mutex_lock(&status_lock);
  EP_STATUS temp = ep_status;
  pthread_mutex_unlock(&status_lock);
  return(temp);
}

long ECA_SESSION::length_in_samples(void) const { 
  if (inputs->size() < master_input_id + 1) return(0);
  return((*inputs)[master_input_id]->length_in_samples()); 
}

double ECA_SESSION::length_in_seconds_exact(void) const { 
  return((double)length_in_samples() / SAMPLE_BUFFER::sample_rate); 
}

long ECA_SESSION::position_in_samples(void) const { 
  if (inputs->size() < master_input_id + 1) return(0);
  return((*inputs)[master_input_id]->position_in_samples()); 
}

double ECA_SESSION::position_in_seconds_exact(void) const {
    return((double)position_in_samples() / SAMPLE_BUFFER::sample_rate); 
}
