// ------------------------------------------------------------------------
// eca-qtsession.cpp: Qt widget representing a ECA_SESSION object and 
//                    its state.
// Copyright (C) 1999 Kai Vehmanen (kaiv@wakkanet.fi)
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
// ------------------------------------------------------------------------

#include <qwidget.h>
#include <qapplication.h>
#include <qpushbutton.h>
#include <qlayout.h>
#include <qfont.h>
#include <qlistview.h>
#include <qstring.h>
#include <qfiledialog.h>
#include <qdir.h>
#include <qmessagebox.h>

#include "eca-session.h"
#include "eca-chainsetup.h"
#include "eca-controller.h"

#include "eca-qtsession.h"

#include "error.h"

QESession::QESession (ECA_CONTROLLER* econtrol, const ECA_SESSION*
		      esession, QWidget *parent=0, const char *name=0)
  : QWidget( parent, name ), ctrl(econtrol), ecasession(esession)
{
  setMinimumSize( 300, 200 );
  //  setMaximumSize( 1024, 768 );

  string caption = "qtecasound / session setup";
  setCaption(caption.c_str());

  QBoxLayout* topLayout = new QVBoxLayout( this );
  QBoxLayout* buttons = new QHBoxLayout();

  init_buttons(buttons);
  init_chainsetuplist();

  topLayout->addLayout(buttons, 1);
  topLayout->addWidget(chainsetupview, 0, 0);

}

void QESession::init_buttons(QBoxLayout* buttons) {
  QFont butfont ("Helvetica", 12, QFont::Normal);

  QPushButton* newb = new QPushButton( "(N)ew", this, "newb" );
  newb->setFont(butfont);
  buttons->addWidget( newb, 1, 1 );

  QPushButton* open = new QPushButton( "(O)pen", this, "open" );
  open->setFont(butfont);
  buttons->addWidget( open, 1, 1 );

  QPushButton* save = new QPushButton( "(S)ave", this, "save" );
  save->setFont(butfont);
  buttons->addWidget( save, 1, 1 );

  QPushButton* activate = new QPushButton( "(A)ctivate", this, "activat" );
  activate->setFont(butfont);
  buttons->addWidget( activate, 1, 1 );

  QPushButton* setup = new QPushButton( "Setu(p)", this, "setup" );
  setup->setFont(butfont);
  buttons->addWidget( setup, 1, 1 );

  QPushButton* quit = new QPushButton( "(Q)uit", this, "quit" );
  quit->setFont(butfont);
  buttons->addWidget( quit, 1, 1);

  QObject::connect( open, SIGNAL(clicked()), this, SLOT(button_open()));
  QObject::connect( save, SIGNAL(clicked()), this, SLOT(button_save()));
  QObject::connect( newb, SIGNAL(clicked()), this, SLOT(button_new()));
  QObject::connect( activate, SIGNAL(clicked()), this, SLOT(button_activate()));
  QObject::connect( setup, SIGNAL(clicked()), this, SLOT(button_open_chainsetup()));
  QObject::connect( quit, SIGNAL(clicked()), this, SLOT(close()));
  //  connect(rewind, SIGNAL(clicked()), this, SLOT(emsg_rewind()) );
  //  connect(quit, SIGNAL(clicked()), this, SLOT(emsg_quit()) );
}

void QESession::init_chainsetuplist (void) {

  chainsetupview = new QListView(this, "chainsetupview");

  //  chainsetupview->setGeometry(0, 0, width(), height());

  chainsetupview->addColumn("Chainsetup");
  chainsetupview->addColumn("Inputs");
  chainsetupview->addColumn("Outputs");
  chainsetupview->addColumn("Chains");
  chainsetupview->addColumn("Active");

  chainsetupview->setMinimumSize( width(), height() );

  chainsetupview->setColumnAlignment(0, AlignLeft);
  chainsetupview->setColumnAlignment(1, AlignRight);
  chainsetupview->setColumnAlignment(2, AlignRight);
  chainsetupview->setColumnAlignment(3, AlignRight);
  chainsetupview->setColumnAlignment(4, AlignRight);

  chainsetupview->setAllColumnsShowFocus( TRUE ); 

  update_chainsetuplist();

  chainsetupview->show();
}

void QESession::update_chainsetuplist (void) {
  chainsetupview->clear();

  vector<ECA_CHAINSETUP*>::const_iterator p = ecasession->get_chainsetups().begin();
  while(p != ecasession->get_chainsetups().end()) {
    //    cerr << "Adding a new one!\n";
    QString astring;
    if ((*p)->name() == ecasession->get_active_chainsetup()->name())
      astring = "yes";
    else
      astring = "no";

    QListViewItem* newitem = new QListViewItem(chainsetupview,
					       (*p)->name().c_str(),
					       kvu_numtostr((*p)->inputs.size()).c_str(),
					       kvu_numtostr((*p)->outputs.size()).c_str(),
					       kvu_numtostr((*p)->chains.size()).c_str(),
					       astring);;
    ++p;
  }
  chainsetupview->triggerUpdate();
}

void QESession::button_open(void) {

  QFileDialog* fdialog = new QFileDialog();
  if (current_dir.exists()) fdialog->setDir(current_dir);
  QString filename = fdialog->getOpenFileName();
  if (!filename.isEmpty()) {
    try {
      ctrl->load_chainsetup(filename.latin1(), false);
    }
    catch(ECA_ERROR* e) {

      if (e->error_action() != ECA_ERROR::stop) {
	QMessageBox* mbox = new QMessageBox(this, "mbox");

	QString errormsg = "";
	errormsg += "Error while loading chainsetup: \"";
	errormsg += e->error_section().c_str();
	errormsg += "\"; ";
	errormsg += e->error_msg().c_str();
	mbox->information(this, "qtecasound", errormsg,0);
      }
    }
    //    catch(...) { throw; }

    update_chainsetuplist();
  }
}

void QESession::button_save(void) {

  QFileDialog* fdialog = new QFileDialog();
  if (current_dir.exists()) fdialog->setDir(current_dir);
  QString filename = fdialog->getSaveFileName();
  if (!filename.isEmpty()) {
    try {
      ctrl->save_chainsetup(filename.latin1(), false);
    }
    catch(ECA_ERROR* e) {
      if (e->error_action() != ECA_ERROR::stop) {
	QMessageBox* mbox = new QMessageBox(this, "mbox");

	QString errormsg = "";
	errormsg += "Error while saving chainsetup: \"";
	errormsg += e->error_section().c_str();
	errormsg += "\"; ";
	errormsg += e->error_msg().c_str();
	mbox->information(this, "qtecasound", errormsg,0);
      }
    }
  }
}

void QESession::button_new(void) {
  ctrl->new_chainsetup("untitled");
  update_chainsetuplist();
}

void QESession::button_activate(void) { 
  QListViewItem* item = chainsetupview->selectedItem();
  if (item != 0) {
    try {
      ctrl->activate_chainsetup(item->text(0).latin1(), false);
      update_chainsetuplist();
    }
    catch(ECA_ERROR* e) {
      if (e->error_action() != ECA_ERROR::stop) {
	QMessageBox* mbox = new QMessageBox(this, "mbox");

	QString errormsg = "";
	errormsg += "Error while activating chainsetup: \"";
	errormsg += e->error_section().c_str();
	errormsg += "\"; ";
	errormsg += e->error_msg().c_str();
	mbox->information(this, "qtecasound", errormsg,0);
      }
    }
  }
}

void QESession::button_open_chainsetup(void) { }

void QESession::keyPressEvent(QKeyEvent* kevent) {
  switch(tolower(kevent->ascii())) {
  case 'q': 
    {
      close();
      break;
    }
  case 'n':
    {    
      button_new();
      break;
    }
  case 'o': 
    {    
      button_open();
      break;
    }
  case 'a': 
    {    
      button_activate();
      break;
    }
  case 'p': 
    {    
      button_open_chainsetup();
      break;
    }
  }
  kevent->ignore();
}
